# Contributing to Galactus One

## Branching Strategy

- **`dev`** is the base branch for all development work. Create feature branches from `dev` and merge pull requests back into `dev`.
- **`main`** is production only. Merges to `main` are done from `dev` when ready to release.

```text
feature-branch --> dev --> main
                    ^       ^
             development  production
```

**Branch naming:** Use conventional prefixes so branch purpose is clear: `feature/` (new features), `fix/` (bug fixes), `chore/` (maintenance), `docs/` (documentation), `refactor/`, `test/`. Add a short kebab-case description, e.g. `feature/login-page`, `fix/relay-websocket-timeout`, `docs/deployment-prereqs`.

## Linting

A single root ESLint config (`eslint.config.mjs`) applies to the whole monorepo. Dashboard uses browser globals; relay and shared use Node globals. Running lint from the root or from a workspace both use this config.

Run from the repo root:

```bash
npm run lint          # Lint all workspaces
npm run lint:fix      # Lint and auto-fix all workspaces
```

Or from an individual app/package directory (scripts run ESLint from root so the root config applies):

```bash
npm run lint          # Lint just that workspace
npm run lint:fix      # Lint and auto-fix just that workspace
```

## Formatting

Prettier is configured at the repo root (`.prettierrc`).

```bash
npm run format        # Format all files
npm run format:check  # Check formatting without writing
```

## Pre-commit Hooks

husky + lint-staged runs automatically on staged files before each commit:

1. **Prettier** formats the files
2. **ESLint** checks and fixes issues

Commits that fail lint will be blocked. Fix the issues and try again.

## Commit Messages

Use [Conventional Commits](https://www.conventionalcommits.org/):

```text
type(scope): description

[optional body]

[optional footer]
```

**Examples:**

```text
feat(dashboard): add login page
fix(relay): handle WebSocket timeout
chore(deps): update dependencies
docs(readme): add deployment instructions
refactor(shared): simplify type exports
test(relay): add integration tests for HTTP relay
```

**Common types:**

| Type       | Description                                             |
| ---------- | ------------------------------------------------------- |
| `feat`     | A new feature                                           |
| `fix`      | A bug fix                                               |
| `chore`    | Maintenance tasks (deps, configs, etc.)                 |
| `docs`     | Documentation only changes                              |
| `refactor` | Code change that neither fixes a bug nor adds a feature |
| `test`     | Adding or updating tests                                |
| `style`    | Formatting, whitespace, etc. (no code change)           |
| `ci`       | CI/CD configuration changes                             |

commitlint enforces this via a husky `commit-msg` hook. Invalid commit messages will be rejected.

## Development Workflow

### 1. Create a feature branch from `dev`

```bash
git checkout dev
git pull origin dev
git checkout -b feature/my-feature
```

### 2. Make changes and commit using conventional commits

```bash
git add .
git commit -m "feat(dashboard): add new component"
```

### 3. Push and create a pull request targeting `dev`

```bash
git push -u origin feature/my-feature
```

### 4. Review and merge

After review and merge to `dev`, changes will be deployed to staging.

### 5. Release to production

When ready for production, merge `dev` into `main`.
