/**
 * React Error Boundary
 *
 * Catches JavaScript errors anywhere in the child component tree,
 * logs the error, and displays a fallback UI instead of crashing the app.
 *
 * This is critical for production resilience - without an error boundary,
 * any uncaught error (especially in cleanup functions or async callbacks)
 * will crash the entire React tree with no recovery.
 */

import React from 'react';

interface ErrorBoundaryProps {
  children: React.ReactNode;
  fallback?: React.ReactNode;
}

interface ErrorBoundaryState {
  hasError: boolean;
  error?: Error;
  errorInfo?: React.ErrorInfo;
}

export class ErrorBoundary extends React.Component<ErrorBoundaryProps, ErrorBoundaryState> {
  constructor(props: ErrorBoundaryProps) {
    super(props);
    this.state = { hasError: false };
  }

  static getDerivedStateFromError(error: Error): Partial<ErrorBoundaryState> {
    // Update state so the next render will show the fallback UI
    return { hasError: true, error };
  }

  componentDidCatch(error: Error, errorInfo: React.ErrorInfo): void {
    // Log the error for debugging
    console.error('ErrorBoundary caught an error:', error);
    console.error('Component stack:', errorInfo.componentStack);

    this.setState({ errorInfo });
  }

  handleReload = (): void => {
    window.location.reload();
  };

  handleGoHome = (): void => {
    window.location.href = '/';
  };

  render(): React.ReactNode {
    if (this.state.hasError) {
      // Allow custom fallback UI
      if (this.props.fallback) {
        return this.props.fallback;
      }

      // Default fallback UI
      return (
        <div className="bg-background flex min-h-screen flex-col items-center justify-center p-4">
          <div className="bg-card border-border w-full max-w-md rounded-lg border p-6 shadow-lg">
            <div className="text-center">
              <h1 className="text-destructive mb-2 text-xl font-bold">Something went wrong</h1>
              <p className="text-muted-foreground mb-4 text-sm">
                An unexpected error occurred. This has been logged for investigation.
              </p>

              {this.state.error && (
                <div className="bg-muted mb-4 rounded p-3 text-left">
                  <p className="text-muted-foreground break-all font-mono text-xs">
                    {this.state.error.message}
                  </p>
                </div>
              )}

              <div className="flex justify-center gap-3">
                <button
                  onClick={this.handleReload}
                  className="bg-primary text-primary-foreground hover:bg-primary/90 rounded-lg px-4 py-2 text-sm font-medium transition-colors"
                >
                  Reload Page
                </button>
                <button
                  onClick={this.handleGoHome}
                  className="bg-secondary text-secondary-foreground hover:bg-secondary/90 rounded-lg px-4 py-2 text-sm font-medium transition-colors"
                >
                  Go Home
                </button>
              </div>
            </div>
          </div>
        </div>
      );
    }

    return this.props.children;
  }
}

export default ErrorBoundary;
