/**
 * Markets Table Component
 */

import React from 'react';
import type { NBAMarketRow } from '../types';
import { formatDate } from '../lib/formatters';
import { formatOdds } from '../lib/oddsConversion';
import { LiquidityBar } from '@/components/atoms/LiquidityBar';

interface MarketsTableProps {
  markets: NBAMarketRow[];
  onMarketClick: (market: NBAMarketRow) => void;
}

const MarketsTable: React.FC<MarketsTableProps> = ({ markets, onMarketClick }) => {
  const handleRowClick = (e: React.MouseEvent, market: NBAMarketRow) => {
    // Don't open if clicking on a button or link
    if (
      (e.target as HTMLElement).tagName === 'BUTTON' ||
      (e.target as HTMLElement).tagName === 'A'
    ) {
      return;
    }
    onMarketClick(market);
  };

  return (
    <div className="table-section">
      <h2>CONSOLIDATED MONEYLINES</h2>
      <div className="table-container">
        <table id="markets-table">
          <thead>
            <tr>
              <th>GAME DATE</th>
              <th>AWAY TEAM</th>
              <th>HOME TEAM</th>
              <th>AWAY KALSHI</th>
              <th>AWAY BID/ASK</th>
              <th>HOME KALSHI</th>
              <th>HOME BID/ASK</th>
              <th>AWAY POLY</th>
              <th>HOME POLY</th>
              <th>AWAY K LIQ</th>
              <th>HOME K LIQ</th>
              <th>AWAY P LIQ</th>
              <th>HOME P LIQ</th>
            </tr>
          </thead>
          <tbody id="markets-tbody">
            {markets.length === 0 ? (
              <tr>
                <td
                  colSpan={13}
                  style={{
                    textAlign: 'center',
                    color: '#888',
                    fontStyle: 'italic',
                    padding: '40px',
                  }}
                >
                  No markets available
                </td>
              </tr>
            ) : (
              markets.map((market) => (
                <tr
                  key={market.market_ticker}
                  data-ticker={market.market_ticker}
                  style={{ cursor: 'pointer' }}
                  onClick={(e) => handleRowClick(e, market)}
                >
                  <td>{formatDate(market.gameDate)}</td>
                  <td>{market.awayTeam}</td>
                  <td>{market.homeTeam}</td>
                  <td>{market.awayProb !== null ? formatOdds(market.awayProb) : '—'}</td>
                  <td className="bid-ask-cell">
                    {market.awayYesBid != null &&
                    market.awayYesAsk != null &&
                    market.awayProb !== null ? (
                      <>
                        <div className="bid-ask">
                          <span className="bid">{market.awayYesBid.toFixed(4)}</span>
                          <span className="ask">{market.awayYesAsk.toFixed(4)}</span>
                        </div>
                        <div className="spread">
                          {(market.awayYesAsk - market.awayYesBid).toFixed(4)} (
                          {(
                            ((market.awayYesAsk - market.awayYesBid) / market.awayProb) *
                            100
                          ).toFixed(2)}
                          %)
                        </div>
                      </>
                    ) : (
                      '—'
                    )}
                  </td>
                  <td>{market.homeProb !== null ? formatOdds(market.homeProb) : '—'}</td>
                  <td className="bid-ask-cell">
                    {market.homeYesBid != null &&
                    market.homeYesAsk != null &&
                    market.homeProb !== null ? (
                      <>
                        <div className="bid-ask">
                          <span className="bid">{market.homeYesBid.toFixed(4)}</span>
                          <span className="ask">{market.homeYesAsk.toFixed(4)}</span>
                        </div>
                        <div className="spread">
                          {(market.homeYesAsk - market.homeYesBid).toFixed(4)} (
                          {(
                            ((market.homeYesAsk - market.homeYesBid) / market.homeProb) *
                            100
                          ).toFixed(2)}
                          %)
                        </div>
                      </>
                    ) : (
                      '—'
                    )}
                  </td>
                  <td>—</td>
                  <td>—</td>
                  <td className="liquidity-cell">
                    <LiquidityBar value={market.awayLiq} />
                  </td>
                  <td className="liquidity-cell">
                    <LiquidityBar value={market.homeLiq} />
                  </td>
                  <td className="liquidity-cell">
                    <LiquidityBar value={0} />
                  </td>
                  <td className="liquidity-cell">
                    <LiquidityBar value={0} />
                  </td>
                </tr>
              ))
            )}
          </tbody>
        </table>
      </div>
    </div>
  );
};

export default MarketsTable;
