/**
 * DateDisplay - Consistent date display
 *
 * Handles various input formats and renders dates consistently as YYYY-MM-DD.
 */

import { cn } from '@/lib/utils';

interface DateDisplayProps {
  /** Date value - accepts ISO string, Date object, or YYYY-MM-DD string */
  value: string | Date | null | undefined;

  /** Text to show when value is null/undefined (default: '—') */
  nullText?: string;

  /** Additional CSS classes */
  className?: string;
}

/**
 * Parse and format a date to YYYY-MM-DD
 */
function formatToYYYYMMDD(value: string | Date): string | null {
  // If already in YYYY-MM-DD format, return as-is
  if (typeof value === 'string' && /^\d{4}-\d{2}-\d{2}$/.test(value)) {
    return value;
  }

  try {
    // For ISO datetime strings (contain "T"), extract the date portion directly
    // to avoid UTC→local timezone shift (e.g. "2026-02-17T20:00:00Z" → Feb 16 in Pacific)
    if (typeof value === 'string' && value.includes('T')) {
      const datePart = value.split('T')[0];
      if (/^\d{4}-\d{2}-\d{2}$/.test(datePart)) {
        return datePart;
      }
    }

    const d = value instanceof Date ? value : new Date(value);
    if (!Number.isFinite(d.getTime())) return null;

    // Use UTC getters to avoid timezone shift for Date objects parsed from UTC strings
    const year = d.getUTCFullYear();
    const month = String(d.getUTCMonth() + 1).padStart(2, '0');
    const day = String(d.getUTCDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  } catch {
    return null;
  }
}

/**
 * Renders a date in consistent YYYY-MM-DD format
 */
export function DateDisplay({ value, nullText = '—', className }: DateDisplayProps) {
  if (value === null || value === undefined) {
    return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
  }

  const formatted = formatToYYYYMMDD(value);
  if (!formatted) {
    return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
  }

  return <span className={className}>{formatted}</span>;
}

export default DateDisplay;
