/**
 * EmptyState - Placeholder for empty lists/content
 */

import { Inbox, Search, FileQuestion, ShoppingCart, Bell } from 'lucide-react';
import { cn } from '@/lib/utils';

type EmptyStateVariant = 'default' | 'search' | 'orders' | 'positions' | 'notifications';

interface EmptyStateProps {
  variant?: EmptyStateVariant;
  title?: string;
  description?: string;
  action?: {
    label: string;
    onClick: () => void;
  };
  size?: 'sm' | 'md' | 'lg';
  className?: string;
}

const defaultContent: Record<
  EmptyStateVariant,
  { icon: React.ReactNode; title: string; description: string }
> = {
  default: {
    icon: <Inbox />,
    title: 'No data',
    description: 'There is nothing to display here yet.',
  },
  search: {
    icon: <Search />,
    title: 'No results found',
    description: 'Try adjusting your search or filters.',
  },
  orders: {
    icon: <ShoppingCart />,
    title: 'No open orders',
    description: 'Place an order to see it here.',
  },
  positions: {
    icon: <FileQuestion />,
    title: 'No positions',
    description: "You don't have any open positions.",
  },
  notifications: {
    icon: <Bell />,
    title: 'No notifications',
    description: "You're all caught up!",
  },
};

export function EmptyState({
  variant = 'default',
  title,
  description,
  action,
  size = 'md',
  className,
}: EmptyStateProps) {
  const content = defaultContent[variant];

  const sizeClasses = {
    sm: { container: 'py-6', icon: 32, title: 'text-sm', description: 'text-xs' },
    md: { container: 'py-10', icon: 48, title: 'text-base', description: 'text-sm' },
    lg: { container: 'py-16', icon: 64, title: 'text-lg', description: 'text-base' },
  };

  const sizes = sizeClasses[size];

  return (
    <div
      className={cn(
        'flex flex-col items-center justify-center text-center',
        sizes.container,
        className
      )}
    >
      <div className="text-muted-foreground/50 mb-4">
        {content.icon && (
          <div style={{ width: sizes.icon, height: sizes.icon }}>{content.icon}</div>
        )}
      </div>
      <h3 className={cn('text-foreground mb-1 font-medium', sizes.title)}>
        {title || content.title}
      </h3>
      <p className={cn('text-muted-foreground max-w-xs', sizes.description)}>
        {description || content.description}
      </p>
      {action && (
        <button
          onClick={action.onClick}
          className="bg-primary text-primary-foreground hover:bg-primary/90 mt-4 rounded-lg px-4 py-2 text-sm font-medium transition-colors"
        >
          {action.label}
        </button>
      )}
    </div>
  );
}
