/**
 * LiquidityBar - Visual representation of liquidity depth
 *
 * A horizontal bar that shows liquidity amount with color gradient:
 * - Red: Low liquidity
 * - Yellow: Medium liquidity
 * - Green: High liquidity
 *
 * The bar width represents the relative liquidity compared to a max value.
 * Label is displayed INSIDE the bar, right-aligned.
 */

import { memo } from 'react';
import { cn } from '@/lib/utils';
import { formatCompactCurrency } from '@/lib/formatters';

interface LiquidityBarProps {
  /** Liquidity value in dollars */
  value: number;
  /** Maximum value for scaling (default: 100000) */
  maxValue?: number;
  /** Height in pixels (default: 20) */
  height?: number;
  /** Show the value label inside the bar */
  showLabel?: boolean;
  /** Additional class names */
  className?: string;
}

/**
 * Get color class based on liquidity value
 * Uses theme-aware colors that adapt to light/dark mode
 */
function getLiquidityColor(value: number, maxValue: number): string {
  const ratio = Math.min(value / maxValue, 1);

  // Thresholds for color transitions (using theme-aware liq-* colors)
  if (ratio < 0.2) {
    return 'bg-liq-red';
  }
  if (ratio < 0.4) {
    return 'bg-liq-orange';
  }
  if (ratio < 0.6) {
    return 'bg-liq-yellow';
  }
  if (ratio < 0.8) {
    return 'bg-liq-lime';
  }
  return 'bg-liq-green';
}

export const LiquidityBar = memo(function LiquidityBar({
  value,
  maxValue = 100000,
  height = 20,
  showLabel = true,
  className,
}: LiquidityBarProps) {
  const widthPercent = Math.min((value / maxValue) * 100, 100);
  const colorClass = getLiquidityColor(value, maxValue);

  return (
    <div
      className={cn('bg-muted/50 relative w-full overflow-hidden rounded', className)}
      style={{ height }}
    >
      {/* Bar fill */}
      <div
        className={cn('h-full rounded-l transition-all duration-300', colorClass)}
        style={{ width: `${widthPercent}%` }}
      />

      {/* Label - inside the bar, right-aligned */}
      {showLabel && (
        <span
          className="text-foreground/80 absolute inset-0 flex items-center justify-end pr-2 text-xs font-medium"
          style={{ textShadow: 'var(--text-overlay-shadow)' }}
        >
          {formatCompactCurrency(value)}
        </span>
      )}
    </div>
  );
});

export default LiquidityBar;
