/**
 * Money - Consistent monetary value display
 *
 * Handles formatting, comma separators, signs, and styling automatically.
 * Use this component for ALL monetary displays to ensure consistency.
 */

import { cn } from '@/lib/utils';

export type MoneyVariant = 'default' | 'pnl' | 'plain';

interface MoneyProps {
  /** The monetary value to display */
  value: number | null | undefined;

  /** Display variant */
  variant?: MoneyVariant;

  /**
   * Show + sign for positive values
   * Default: true for 'pnl' variant, false otherwise
   */
  showSign?: boolean;

  /**
   * Show $ prefix
   * Default: true for 'default', false for 'pnl' and 'plain'
   */
  showDollarSign?: boolean;

  /**
   * Color positive/negative values
   * Default: true for 'pnl' variant, false otherwise
   */
  colorize?: boolean;

  /** Number of decimal places (default: 2) */
  decimals?: number;

  /** Text to show when value is null/undefined (default: '—') */
  nullText?: string;

  /** Additional CSS classes */
  className?: string;
}

/**
 * Format a number with locale-aware comma separators
 */
function formatWithCommas(value: number, decimals: number): string {
  return value.toLocaleString('en-US', {
    minimumFractionDigits: decimals,
    maximumFractionDigits: decimals,
  });
}

/**
 * Renders a monetary value with proper formatting and styling
 */
export function Money({
  value,
  variant = 'default',
  showSign,
  showDollarSign,
  colorize,
  decimals = 2,
  nullText = '—',
  className,
}: MoneyProps) {
  // Handle null/undefined
  if (value === null || value === undefined) {
    return <span className={cn('text-muted-foreground', className)}>{nullText}</span>;
  }

  // Determine defaults based on variant
  const isPnl = variant === 'pnl';
  const isPlain = variant === 'plain';

  const shouldShowSign = showSign ?? isPnl;
  const shouldShowDollar = showDollarSign ?? (!isPnl && !isPlain);
  const shouldColorize = colorize ?? isPnl;

  // Format the absolute value with commas
  const absFormatted = formatWithCommas(Math.abs(value), decimals);

  // Build the display string
  const sign = value < 0 ? '-' : shouldShowSign && value > 0 ? '+' : '';
  const dollar = shouldShowDollar ? '$' : '';
  const display = `${sign}${dollar}${absFormatted}`;

  // Determine color class (uses theme-aware colors)
  let colorClass = '';
  if (shouldColorize) {
    if (value > 0) {
      colorClass = 'text-positive';
    } else if (value < 0) {
      colorClass = 'text-negative';
    }
  }

  return <span className={cn(colorClass, className)}>{display}</span>;
}

export default Money;
