/**
 * PriceText - Display a price value with optional cents symbol
 * Supports centi-cents (fractional cents) for Kalshi API precision
 *
 * Kalshi prices are in centi-cents (1/100th of a cent):
 * - API value 5230 = 52.30¢ = $0.5230
 * - API value 100 = 1.00¢ = $0.0100
 */

import { cn } from '@/lib/utils';
import { TEXT_SIZE_CLASSES, type SizeVariant } from '@/lib/sizeClasses';

interface PriceTextProps {
  /** Price value - interpretation depends on `unit` prop */
  value: number;
  /**
   * How to interpret the value:
   * - 'centicents': Raw API value (5230 = 52.30¢) - default for API data
   * - 'cents': Cents with optional decimals (52.3 = 52.3¢)
   * - 'dollars': Dollar amount (0.523 = 52.3¢)
   */
  unit?: 'centicents' | 'cents' | 'dollars';
  /** Number of decimal places to show (0-2 for cents display) */
  decimals?: number;
  /** Display format: 'cents' shows ¢, 'dollars' shows $ */
  format?: 'cents' | 'dollars';
  size?: SizeVariant;
  /** Whether to show the currency symbol */
  showSymbol?: boolean;
  className?: string;
}

/**
 * Convert value to cents based on unit
 */
function toCents(value: number, unit: 'centicents' | 'cents' | 'dollars'): number {
  if (unit === 'centicents') return value / 100;
  if (unit === 'dollars') return value * 100;
  return value;
}

export function PriceText({
  value,
  unit = 'cents',
  decimals,
  format = 'cents',
  size = 'md',
  showSymbol = true,
  className,
}: PriceTextProps) {
  const centsValue = toCents(value, unit);

  // Determine decimal places: default to 0 for whole cents, 1-2 for fractional
  const defaultDecimals = centsValue % 1 === 0 ? 0 : (centsValue * 10) % 1 === 0 ? 1 : 2;
  const decimalPlaces = decimals ?? defaultDecimals;

  if (format === 'dollars') {
    const dollarValue = centsValue / 100;
    const dollarDecimals = decimals ?? (dollarValue % 0.01 === 0 ? 2 : 4);
    return (
      <span className={cn('font-mono tabular-nums', TEXT_SIZE_CLASSES[size], className)}>
        {showSymbol && <span className="text-muted-foreground">$</span>}
        {dollarValue.toFixed(dollarDecimals)}
      </span>
    );
  }

  return (
    <span className={cn('font-mono tabular-nums', TEXT_SIZE_CLASSES[size], className)}>
      {centsValue.toFixed(decimalPlaces)}
      {showSymbol && <span className="text-muted-foreground">¢</span>}
    </span>
  );
}

/**
 * CentiCentPrice - Convenience component for raw Kalshi API prices
 * Takes centi-cent values directly (5230 = 52.30¢)
 */
export function CentiCentPrice({
  value,
  decimals = 2,
  ...props
}: Omit<PriceTextProps, 'unit'> & { value: number }) {
  return (
    <PriceText
      value={value}
      unit="centicents"
      decimals={decimals}
      {...props}
    />
  );
}

export default PriceText;
