/**
 * Sentiment Gauge
 * Visual indicator for market sentiment (bullish/bearish/neutral)
 */

import { cn } from '@/lib/utils';
import { TrendingUp, TrendingDown, Minus } from 'lucide-react';

type SentimentLevel = 'very_bearish' | 'bearish' | 'neutral' | 'bullish' | 'very_bullish';

interface SentimentGaugeProps {
  value: number; // -100 to 100
  label?: string;
  showValue?: boolean;
  size?: 'sm' | 'md' | 'lg';
  className?: string;
}

const sizeConfig = {
  sm: { width: 80, height: 6, icon: 12, text: 'text-xs' },
  md: { width: 120, height: 8, icon: 14, text: 'text-sm' },
  lg: { width: 160, height: 10, icon: 16, text: 'text-base' },
};

const getSentimentLevel = (value: number): SentimentLevel => {
  if (value <= -60) return 'very_bearish';
  if (value <= -20) return 'bearish';
  if (value <= 20) return 'neutral';
  if (value <= 60) return 'bullish';
  return 'very_bullish';
};

const sentimentConfig: Record<
  SentimentLevel,
  { color: string; label: string; bgGradient: string }
> = {
  very_bearish: {
    color: 'text-red-500',
    label: 'Very Bearish',
    bgGradient: 'from-red-600 to-red-500',
  },
  bearish: { color: 'text-red-400', label: 'Bearish', bgGradient: 'from-red-500 to-orange-500' },
  neutral: { color: 'text-slate-400', label: 'Neutral', bgGradient: 'from-slate-500 to-slate-400' },
  bullish: {
    color: 'text-green-400',
    label: 'Bullish',
    bgGradient: 'from-green-500 to-emerald-400',
  },
  very_bullish: {
    color: 'text-green-500',
    label: 'Very Bullish',
    bgGradient: 'from-green-600 to-green-500',
  },
};

export function SentimentGauge({
  value,
  label,
  showValue = true,
  size = 'md',
  className,
}: SentimentGaugeProps) {
  const config = sizeConfig[size];
  const sentiment = getSentimentLevel(value);
  const sentimentStyle = sentimentConfig[sentiment];

  // Normalize value to 0-100 for positioning
  const normalizedValue = (value + 100) / 2;

  const Icon = value > 20 ? TrendingUp : value < -20 ? TrendingDown : Minus;

  return (
    <div className={cn('flex flex-col gap-1', className)}>
      {label && <span className="text-xs text-slate-500">{label}</span>}

      <div className="flex items-center gap-2">
        <Icon
          className={cn('flex-shrink-0', sentimentStyle.color)}
          style={{ width: config.icon, height: config.icon }}
        />

        {/* Gauge Bar */}
        <div
          className="relative overflow-hidden rounded-full bg-slate-700"
          style={{ width: config.width, height: config.height }}
        >
          {/* Background gradient showing full range */}
          <div className="absolute inset-0 bg-gradient-to-r from-red-500 via-slate-400 to-green-500 opacity-30" />

          {/* Indicator position */}
          <div
            className="absolute bottom-0 top-0 w-1.5 rounded-full bg-white shadow-lg transition-all duration-300"
            style={{ left: `calc(${normalizedValue}% - 3px)` }}
          />
        </div>

        {showValue && (
          <span className={cn('font-medium tabular-nums', config.text, sentimentStyle.color)}>
            {value > 0 ? '+' : ''}
            {value}
          </span>
        )}
      </div>

      <span className={cn('text-xs', sentimentStyle.color)}>{sentimentStyle.label}</span>
    </div>
  );
}

// Compact inline variant
export function SentimentBadge({ value, className }: { value: number; className?: string }) {
  const sentiment = getSentimentLevel(value);
  const config = sentimentConfig[sentiment];
  const Icon = value > 20 ? TrendingUp : value < -20 ? TrendingDown : Minus;

  return (
    <div
      className={cn(
        'inline-flex items-center gap-1 rounded-full px-2 py-0.5 text-xs font-medium',
        'bg-gradient-to-r',
        config.bgGradient,
        'text-white',
        className
      )}
    >
      <Icon className="h-3 w-3" />
      <span>
        {value > 0 ? '+' : ''}
        {value}
      </span>
    </div>
  );
}

export type { SentimentLevel };
