/**
 * Sortable table header cell
 *
 * Extends Th with click-to-sort behavior and ascending/descending indicators.
 */

import { cn } from '@/lib/utils';
import { ArrowUp, ArrowDown, ArrowUpDown } from 'lucide-react';

export type SortDirection = 'asc' | 'desc' | null;

interface SortableThProps extends React.ThHTMLAttributes<HTMLTableCellElement> {
  children: React.ReactNode;
  sortKey: string;
  activeSortKey: string | null;
  activeSortDir: SortDirection;
  onSort: (key: string) => void;
  className?: string;
}

export function SortableTh({
  children,
  sortKey,
  activeSortKey,
  activeSortDir,
  onSort,
  className,
  ...rest
}: SortableThProps) {
  const isActive = activeSortKey === sortKey;

  return (
    <th
      {...rest}
      className={cn(
        'whitespace-nowrap px-3 py-3 text-left text-[13px] font-semibold uppercase tracking-wider',
        'text-muted-foreground',
        'hover:text-foreground cursor-pointer select-none transition-colors',
        className
      )}
      onClick={() => onSort(sortKey)}
    >
      <span className="inline-flex items-center gap-1">
        {children}
        {isActive && activeSortDir === 'asc' && <ArrowUp className="h-3 w-3" />}
        {isActive && activeSortDir === 'desc' && <ArrowDown className="h-3 w-3" />}
        {!isActive && <ArrowUpDown className="h-3 w-3 opacity-30" />}
      </span>
    </th>
  );
}
