/**
 * Sparkline - Mini inline chart for showing trends
 */

import { useMemo } from 'react';
import { cn } from '@/lib/utils';

interface SparklineProps {
  data: number[];
  width?: number;
  height?: number;
  strokeWidth?: number;
  color?: 'auto' | 'green' | 'red' | 'blue' | 'primary';
  showArea?: boolean;
  className?: string;
}

export function Sparkline({
  data,
  width = 80,
  height = 24,
  strokeWidth = 1.5,
  color = 'auto',
  showArea = true,
  className,
}: SparklineProps) {
  const { path, areaPath, strokeColor, fillColor } = useMemo(() => {
    if (data.length < 2) {
      return { path: '', areaPath: '', strokeColor: '', fillColor: '' };
    }

    const min = Math.min(...data);
    const max = Math.max(...data);
    const range = max - min || 1;

    const padding = 2;
    const chartWidth = width - padding * 2;
    const chartHeight = height - padding * 2;

    const points = data.map((value, index) => {
      const x = padding + (index / (data.length - 1)) * chartWidth;
      const y = padding + chartHeight - ((value - min) / range) * chartHeight;
      return { x, y };
    });

    const linePath = points.map((p, i) => `${i === 0 ? 'M' : 'L'} ${p.x} ${p.y}`).join(' ');

    const area = `${linePath} L ${points[points.length - 1].x} ${height - padding} L ${padding} ${height - padding} Z`;

    // Determine color based on trend
    const trend = data[data.length - 1] - data[0];
    let stroke: string;
    let fill: string;

    const colorMap: Record<string, { stroke: string; fill: string }> = {
      green: { stroke: '#22c55e', fill: 'rgba(34, 197, 94, 0.1)' },
      red: { stroke: '#ef4444', fill: 'rgba(239, 68, 68, 0.1)' },
      blue: { stroke: '#3b82f6', fill: 'rgba(59, 130, 246, 0.1)' },
      primary: { stroke: 'hsl(var(--primary))', fill: 'hsla(var(--primary), 0.1)' },
    };

    if (color === 'auto') {
      const selectedColor = trend >= 0 ? colorMap.green : colorMap.red;
      stroke = selectedColor.stroke;
      fill = selectedColor.fill;
    } else {
      stroke = colorMap[color].stroke;
      fill = colorMap[color].fill;
    }

    return { path: linePath, areaPath: area, strokeColor: stroke, fillColor: fill };
  }, [data, width, height, color]);

  if (data.length < 2) {
    return (
      <div
        className={cn('text-muted-foreground flex items-center justify-center', className)}
        style={{ width, height }}
      >
        <span className="text-[10px]">--</span>
      </div>
    );
  }

  return (
    <svg
      width={width}
      height={height}
      className={className}
      viewBox={`0 0 ${width} ${height}`}
    >
      {showArea && (
        <path
          d={areaPath}
          fill={fillColor}
        />
      )}
      <path
        d={path}
        fill="none"
        stroke={strokeColor}
        strokeWidth={strokeWidth}
        strokeLinecap="round"
        strokeLinejoin="round"
      />
    </svg>
  );
}
