/**
 * Shared table cell components
 *
 * Use these instead of raw <th> and <td> elements for consistent styling.
 *
 * Typography hierarchy (light mode optimized):
 * - Headers: 13px semibold, uppercase, #6B7280 (muted-foreground)
 * - Data cells: 14px regular, #111827 (foreground)
 * - Increased padding for better readability
 */

import { cn } from '@/lib/utils';

interface ThProps extends React.ThHTMLAttributes<HTMLTableCellElement> {
  children: React.ReactNode;
  className?: string;
}

/**
 * Table header cell with consistent styling
 *
 * Light mode: 13px semibold uppercase in medium gray
 * Increased padding for spacious feel
 */
export function Th({ children, className, ...rest }: ThProps) {
  return (
    <th
      {...rest}
      className={cn(
        'whitespace-nowrap px-3 py-3 text-left text-[13px] font-semibold uppercase tracking-wider',
        'text-muted-foreground',
        className
      )}
    >
      {children}
    </th>
  );
}

interface TdProps extends React.TdHTMLAttributes<HTMLTableCellElement> {
  children: React.ReactNode;
  className?: string;
}

/**
 * Table data cell with consistent styling
 *
 * Light mode: 14px in near-black for maximum contrast
 * Increased vertical padding (py-3 = 12px each side)
 */
export function Td({ children, className, ...rest }: TdProps) {
  return (
    <td
      {...rest}
      className={cn('text-foreground px-3 py-3 text-sm', className)}
    >
      {children}
    </td>
  );
}
