/**
 * DevSeriesExplorer — Market table for dev mode.
 *
 * Series selection happens in the header sub-tabs (ValueDashboardView).
 * This component renders the market table for the selected series,
 * sorted by volume descending with richer data columns.
 */

import { useMemo } from 'react';
import { RefreshCw } from 'lucide-react';
import { cn } from '@/lib/utils';
import type { KalshiMarket, KalshiOrderbook } from '@/types';

interface DevSeriesExplorerProps {
  selectedTicker: string | null;
  selectedLabel?: string;
  markets: KalshiMarket[];
  orderbooks: Map<string, KalshiOrderbook>;
  loading: boolean;
  onRefresh: () => void;
}

export function DevSeriesExplorer({
  selectedTicker,
  selectedLabel,
  markets,
  orderbooks,
  loading,
  onRefresh,
}: DevSeriesExplorerProps) {
  // Sort markets by volume descending (most liquid first)
  const sortedMarkets = useMemo(() => {
    return [...markets].sort((a, b) => b.volume - a.volume);
  }, [markets]);

  // Group by event ticker for visual grouping
  const eventGroups = useMemo(() => {
    const groups = new Map<string, number>();
    for (const m of sortedMarkets) {
      groups.set(m.event_ticker, (groups.get(m.event_ticker) ?? 0) + 1);
    }
    return groups;
  }, [sortedMarkets]);

  const totalVolume = useMemo(() => {
    return markets.reduce((sum, m) => sum + m.volume, 0);
  }, [markets]);

  const totalOI = useMemo(() => {
    return markets.reduce((sum, m) => sum + m.open_interest, 0);
  }, [markets]);

  if (!selectedTicker) {
    return (
      <div className="text-muted-foreground flex h-48 items-center justify-center text-sm">
        Select a series above to view open markets
      </div>
    );
  }

  return (
    <div className="bg-card border-border rounded-lg border">
      <div className="border-border flex items-center justify-between border-b px-3 py-2">
        <div className="flex items-center gap-3">
          <span className="text-sm font-medium text-amber-400">
            {selectedLabel || selectedTicker}
          </span>
          {selectedLabel && (
            <span className="text-muted-foreground font-mono text-[10px]">{selectedTicker}</span>
          )}
          <span className="text-muted-foreground text-xs">
            {markets.length} market{markets.length !== 1 ? 's' : ''} across {eventGroups.size} event
            {eventGroups.size !== 1 ? 's' : ''}
          </span>
          {totalVolume > 0 && (
            <span className="text-muted-foreground text-xs">
              Vol: {totalVolume.toLocaleString()} | OI: {totalOI.toLocaleString()}
            </span>
          )}
        </div>
        <div className="flex items-center gap-2">
          {loading && <span className="text-muted-foreground text-[10px]">refreshing...</span>}
          <button
            onClick={onRefresh}
            disabled={loading}
            className="text-muted-foreground hover:text-foreground transition-colors disabled:opacity-50"
            title="Refresh markets"
          >
            <RefreshCw className={cn('h-3.5 w-3.5', loading && 'animate-spin')} />
          </button>
        </div>
      </div>

      {loading && markets.length === 0 ? (
        <div className="text-muted-foreground p-4 text-center text-xs">Loading markets...</div>
      ) : markets.length === 0 ? (
        <div className="text-muted-foreground p-4 text-center text-xs">
          No open markets for this series
        </div>
      ) : (
        <div className="overflow-x-auto">
          <table className="w-full text-xs">
            <thead>
              <tr className="border-border border-b">
                <th className="text-muted-foreground px-3 py-2 text-left font-medium">Ticker</th>
                <th className="text-muted-foreground px-3 py-2 text-left font-medium">Title</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Yes Bid</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Yes Ask</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Last</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Volume</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">OI</th>
                <th className="text-muted-foreground px-3 py-2 text-right font-medium">Liq</th>
              </tr>
            </thead>
            <tbody>
              {sortedMarkets.map((m) => {
                const ob = orderbooks.get(m.market_ticker);
                const yesBid = ob?.yes?.[0]?.[0] ?? m.yes_bid;
                const noBid = ob?.no?.[0]?.[0] ?? m.no_bid;
                const yesAsk = noBid != null ? 100 - noBid : m.yes_ask;
                const lastPrice = m.last_price;

                return (
                  <tr
                    key={m.market_ticker}
                    className="border-border hover:bg-muted/30 border-b last:border-b-0"
                  >
                    <td className="px-3 py-1.5 font-mono text-[10px]">{m.market_ticker}</td>
                    <td className="max-w-xs truncate px-3 py-1.5">{m.title}</td>
                    <td className="px-3 py-1.5 text-right font-mono">
                      {yesBid != null ? `${yesBid}¢` : '—'}
                    </td>
                    <td className="px-3 py-1.5 text-right font-mono">
                      {yesAsk != null ? `${yesAsk}¢` : '—'}
                    </td>
                    <td
                      className={cn(
                        'px-3 py-1.5 text-right font-mono',
                        lastPrice != null ? 'text-foreground' : 'text-muted-foreground'
                      )}
                    >
                      {lastPrice != null ? `${lastPrice}¢` : '—'}
                    </td>
                    <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
                      {m.volume.toLocaleString()}
                    </td>
                    <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
                      {m.open_interest.toLocaleString()}
                    </td>
                    <td className="text-muted-foreground px-3 py-1.5 text-right font-mono">
                      {m.liquidity.toLocaleString()}
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}
