/**
 * AppShell - Main layout wrapper for authenticated state
 *
 * Provides:
 * - Sidebar navigation
 * - Header with status/settings
 * - Main content area
 * - Status bar
 */

import { ReactNode } from 'react';
import { cn } from '@/lib/utils';
import { useVersionCheck } from '@/hooks/useVersionCheck';

interface AppShellProps {
  sidebar: ReactNode;
  header: ReactNode;
  statusBar?: ReactNode;
  children: ReactNode;
  className?: string;
}

export function AppShell({ sidebar, header, statusBar, children, className }: AppShellProps) {
  const { updateAvailable, reload } = useVersionCheck();

  return (
    <div className={cn('bg-background text-foreground flex h-screen overflow-hidden', className)}>
      {/* Sidebar */}
      <aside className="bg-muted border-border/50 flex w-16 flex-shrink-0 flex-col border-r">
        {sidebar}
      </aside>

      {/* Main area */}
      <div className="flex min-w-0 flex-1 flex-col">
        {/* Header */}
        <header className="border-border/50 bg-muted h-12 flex-shrink-0 border-b">{header}</header>

        {/* Update banner */}
        {updateAvailable && (
          <div className="bg-primary/15 border-primary/30 flex items-center justify-center gap-3 border-b px-4 py-2">
            <span className="text-sm">New version deployed.</span>
            <button
              onClick={reload}
              className="bg-primary text-primary-foreground rounded px-3 py-1 text-xs font-medium hover:opacity-90"
            >
              Reload now
            </button>
          </div>
        )}

        {/* Content */}
        <main className="flex-1 overflow-auto">{children}</main>

        {/* Status bar */}
        {statusBar && (
          <footer className="border-border/50 bg-muted/30 h-6 flex-shrink-0 border-t">
            {statusBar}
          </footer>
        )}
      </div>
    </div>
  );
}
