/**
 * MarketExplorer - Market discovery with hierarchical drill-down
 *
 * Hierarchy: Category → Series → Events → Markets
 *
 * Progressive disclosure - drill down step by step in the same vertical space
 * Navigation is URL-driven for proper back button support and deep linking.
 */

import { useState, useCallback, useMemo } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  RefreshCw,
  ChevronRight,
  ArrowLeft,
  Layers,
  Calendar,
  TrendingUp,
  BarChart3,
  Grid3X3,
  List,
  Flame,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { LoadingSpinner } from '@/components/atoms/LoadingState';
import { EmptyState } from '@/components/atoms/EmptyState';
import { Dropdown } from '@/components/atoms/Dropdown';
import { SearchInput } from '@/components/molecules/SearchInput';
import { TabNav } from '@/components/molecules/TabNav';
import { TopMoversPanel } from '@/components/organisms/TopMoversPanel';
import { SeriesCard } from './SeriesCard';
import { EventCard } from './EventCard';
import { MarketCard } from './MarketCard';
import { MarketRow } from './MarketRow';
import { useMarketExplorerData, type ExploreParams } from './useMarketExplorerData';
import type { KalshiApiClient } from '@/lib/kalshiApi';
import type { KalshiSeries, KalshiMarket, KalshiEvent } from '@/types';
import {
  transformMarkets,
  getGainers,
  getLosers,
  toTopMover,
  type TransformedMarket,
} from '@/lib/marketTransform';

type SeriesSortKey = 'volume' | 'title';
type EventSortKey = 'date' | 'title';
type MarketSortKey = 'volume' | 'change' | 'price' | 'title';
type MarketTab = 'all' | 'trending' | 'new';
type ViewMode = 'grid' | 'list';

interface MarketExplorerProps {
  apiClient: KalshiApiClient;
  onMarketSelect?: (market: KalshiMarket) => void;
  params?: ExploreParams;
  className?: string;
}

const CATEGORIES = [
  { id: 'Politics', label: 'Politics' },
  { id: 'Sports', label: 'Sports' },
  { id: 'Economics', label: 'Economics' },
  { id: 'Crypto', label: 'Crypto' },
  { id: 'Climate and Weather', label: 'Weather' },
  { id: 'Entertainment', label: 'Entertainment' },
  { id: 'Tech', label: 'Tech' },
  { id: 'Science', label: 'Science' },
];

type DrillLevel = 'category' | 'series' | 'events' | 'markets';

export function MarketExplorer({
  apiClient,
  onMarketSelect,
  params,
  className,
}: MarketExplorerProps) {
  const navigate = useNavigate();

  // Data from hook
  const {
    categoryCounts,
    seriesInCategory,
    eventsInSeries,
    marketsInEvent,
    selectedSeries,
    selectedEvent,
    loading,
    error,
    refresh,
  } = useMarketExplorerData(apiClient, params);

  // Derive current level from URL params
  const currentLevel: DrillLevel = useMemo(() => {
    if (params?.event) return 'markets';
    if (params?.series) return 'events';
    if (params?.category) return 'series';
    return 'category';
  }, [params?.category, params?.series, params?.event]);

  // Search/filter
  const [searchQuery, setSearchQuery] = useState('');

  // Sort state
  const [seriesSort, setSeriesSort] = useState<SeriesSortKey>('volume');
  const [eventsSort, setEventsSort] = useState<EventSortKey>('date');
  const [marketSort, setMarketSort] = useState<MarketSortKey>('volume');

  // Market view state
  const [marketTab, setMarketTab] = useState<MarketTab>('all');
  const [viewMode, setViewMode] = useState<ViewMode>('grid');

  const handleCategorySelect = useCallback(
    (categoryId: string) => {
      setSearchQuery('');
      navigate(`/app/explore/${encodeURIComponent(categoryId)}`);
    },
    [navigate]
  );

  const handleSeriesSelect = useCallback(
    (series: KalshiSeries) => {
      if (!params?.category) return;
      setSearchQuery('');
      navigate(
        `/app/explore/${encodeURIComponent(params.category)}/${encodeURIComponent(series.series_ticker)}`
      );
    },
    [navigate, params?.category]
  );

  const handleEventSelect = useCallback(
    (event: KalshiEvent) => {
      if (!params?.category || !params?.series) return;
      setSearchQuery('');
      navigate(
        `/app/explore/${encodeURIComponent(params.category)}/${encodeURIComponent(params.series)}/${encodeURIComponent(event.event_ticker)}`
      );
    },
    [navigate, params?.category, params?.series]
  );

  const handleMarketSelect = useCallback(
    (market: KalshiMarket) => {
      if (!market?.market_ticker) return;
      navigate(`/app/market/${encodeURIComponent(market.market_ticker)}`);
      onMarketSelect?.(market);
    },
    [navigate, onMarketSelect]
  );

  const goBack = useCallback(() => {
    setSearchQuery('');
    if (currentLevel === 'markets' && params?.category && params?.series) {
      navigate(
        `/app/explore/${encodeURIComponent(params.category)}/${encodeURIComponent(params.series)}`
      );
      return;
    }
    if (currentLevel === 'events' && params?.category) {
      navigate(`/app/explore/${encodeURIComponent(params.category)}`);
      return;
    }
    if (currentLevel === 'series') {
      navigate('/app/explore');
    }
  }, [currentLevel, params?.category, params?.series, navigate]);

  const filteredItems = useMemo(() => {
    const query = searchQuery.toLowerCase().trim();

    if (currentLevel === 'series') {
      let items = seriesInCategory;
      if (query) {
        items = items.filter(
          (s) =>
            s.title?.toLowerCase().includes(query) || s.series_ticker?.toLowerCase().includes(query)
        );
      }
      return [...items].sort((a, b) => {
        if (seriesSort === 'volume') {
          const volA = a.volume ?? (a.volume_fp ? parseFloat(a.volume_fp) : 0);
          const volB = b.volume ?? (b.volume_fp ? parseFloat(b.volume_fp) : 0);
          return volB - volA;
        }
        return (a.title || '').localeCompare(b.title || '');
      });
    }

    if (currentLevel === 'events') {
      let items = eventsInSeries;
      if (query) {
        items = items.filter(
          (e) =>
            e.title?.toLowerCase().includes(query) ||
            e.event_ticker?.toLowerCase().includes(query) ||
            e.subtitle?.toLowerCase().includes(query) ||
            e.sub_title?.toLowerCase().includes(query)
        );
      }
      return [...items].sort((a, b) => {
        if (eventsSort === 'date') {
          if (a.strike_date && b.strike_date) {
            return new Date(a.strike_date).getTime() - new Date(b.strike_date).getTime();
          }
          if (a.strike_date) return -1;
          if (b.strike_date) return 1;
          return (a.title || '').localeCompare(b.title || '');
        }
        return (a.title || '').localeCompare(b.title || '');
      });
    }

    return [];
  }, [currentLevel, searchQuery, seriesInCategory, eventsInSeries, seriesSort, eventsSort]);

  const breadcrumbs = useMemo(() => {
    const crumbs: { label: string; ticker?: string; onClick?: () => void }[] = [
      {
        label: 'Categories',
        onClick: currentLevel !== 'category' ? () => navigate('/app/explore') : undefined,
      },
    ];

    if (params?.category) {
      crumbs.push({
        label: params.category,
        onClick:
          currentLevel !== 'series'
            ? () => navigate(`/app/explore/${encodeURIComponent(params.category!)}`)
            : undefined,
      });
    }

    if (params?.series) {
      crumbs.push({
        label: selectedSeries?.title ?? params.series,
        ticker: params.series,
        onClick:
          currentLevel !== 'events'
            ? () =>
                navigate(
                  `/app/explore/${encodeURIComponent(params.category!)}/${encodeURIComponent(params.series!)}`
                )
            : undefined,
      });
    }

    if (params?.event) {
      crumbs.push({
        label: selectedEvent?.title ?? params.event,
        ticker: params.event,
      });
    }

    return crumbs;
  }, [
    currentLevel,
    params?.category,
    params?.series,
    params?.event,
    selectedSeries,
    selectedEvent,
    navigate,
  ]);

  const transformedMarkets = useMemo(() => {
    return transformMarkets(marketsInEvent);
  }, [marketsInEvent]);

  const filteredTransformedMarkets = useMemo(() => {
    const query = searchQuery.toLowerCase().trim();
    let markets = transformedMarkets;

    if (query) {
      markets = markets.filter(
        (m) =>
          m.title.toLowerCase().includes(query) ||
          m.ticker.toLowerCase().includes(query) ||
          m.subtitle?.toLowerCase().includes(query)
      );
    }

    if (marketTab === 'trending') {
      markets = markets.filter((m) => Math.abs(m.change24h) > 0.1);
    }
    if (marketTab === 'new') {
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
      markets = markets.filter((m) => {
        const openTime = m.raw.open_time ? new Date(m.raw.open_time) : null;
        return openTime && openTime > sevenDaysAgo;
      });
    }

    const sortFns: Record<MarketSortKey, (a: TransformedMarket, b: TransformedMarket) => number> = {
      volume: (a, b) => b.volume24h - a.volume24h,
      change: (a, b) => Math.abs(b.change24h) - Math.abs(a.change24h),
      price: (a, b) => b.yesPrice - a.yesPrice,
      title: (a, b) => a.title.localeCompare(b.title),
    };
    return [...markets].sort(sortFns[marketSort]);
  }, [transformedMarkets, searchQuery, marketTab, marketSort]);

  const topMovers = useMemo(() => {
    const gainers = getGainers(transformedMarkets, 3).map(toTopMover);
    const losers = getLosers(transformedMarkets, 3).map(toTopMover);
    return [...gainers, ...losers];
  }, [transformedMarkets]);

  const isLoading = loading === 'loading';

  return (
    <div className={cn('flex h-full flex-col', className)}>
      {/* Breadcrumb navigation */}
      <div className="border-border flex items-center gap-2 border-b px-4 py-3 text-sm">
        {currentLevel !== 'category' && (
          <button
            onClick={goBack}
            className="hover:bg-muted mr-1 rounded p-1 transition-colors"
            title="Go back"
          >
            <ArrowLeft className="h-4 w-4" />
          </button>
        )}

        {breadcrumbs.map((crumb, i) => {
          // Don't show ticker badge if the label is already the ticker
          const showTickerBadge = crumb.ticker && crumb.label !== crumb.ticker;
          const isLast = i === breadcrumbs.length - 1;

          return (
            <div
              key={i}
              className="flex items-center gap-2"
            >
              {i > 0 && <ChevronRight className="text-muted-foreground h-4 w-4" />}
              {crumb.onClick ? (
                <button
                  onClick={crumb.onClick}
                  className="text-muted-foreground hover:text-foreground transition-colors hover:underline"
                >
                  {crumb.label}
                </button>
              ) : crumb.ticker ? (
                <button
                  onClick={() => navigator.clipboard.writeText(crumb.ticker!)}
                  className="cursor-pointer font-medium hover:underline"
                  title={`Click to copy: ${crumb.ticker}`}
                >
                  {crumb.label}
                </button>
              ) : (
                <span className="font-medium">{crumb.label}</span>
              )}
              {showTickerBadge && !isLast && crumb.onClick && (
                <button
                  onClick={crumb.onClick}
                  className="text-muted-foreground bg-muted hover:bg-muted/80 hover:text-foreground rounded px-1.5 py-0.5 font-mono text-xs transition-colors"
                  title={crumb.ticker}
                >
                  {crumb.ticker}
                </button>
              )}
            </div>
          );
        })}

        <button
          onClick={refresh}
          disabled={isLoading}
          className="hover:bg-muted ml-auto rounded p-1.5 transition-colors"
          title="Refresh"
        >
          <RefreshCw className={cn('text-muted-foreground h-4 w-4', isLoading && 'animate-spin')} />
        </button>
      </div>

      {/* Search bar (when not at category level) */}
      {currentLevel !== 'category' && (
        <div className="border-border border-b p-4">
          <SearchInput
            value={searchQuery}
            onChange={setSearchQuery}
            placeholder={`Search ${currentLevel}...`}
            size="md"
          />
        </div>
      )}

      {/* Content area */}
      <div className="flex-1 overflow-auto">
        {isLoading && (
          <div className="flex items-center justify-center py-12">
            <LoadingSpinner size="lg" />
          </div>
        )}

        {!isLoading && error && (
          <div className="p-8 text-center">
            <p className="text-destructive text-sm">{error}</p>
            <button
              onClick={refresh}
              className="bg-primary text-primary-foreground hover:bg-primary/90 mt-4 rounded-lg px-4 py-2 text-sm transition-colors"
            >
              Try Again
            </button>
          </div>
        )}

        {/* Category selection */}
        {!isLoading && !error && currentLevel === 'category' && (
          <div className="p-4">
            <div className="grid grid-cols-2 gap-3 md:grid-cols-3 lg:grid-cols-4">
              {CATEGORIES.map((cat) => {
                const count = categoryCounts.get(cat.id) || 0;
                return (
                  <button
                    key={cat.id}
                    onClick={() => handleCategorySelect(cat.id)}
                    className="bg-card border-border hover:border-primary/50 hover:bg-muted/30 group rounded-lg border p-4 text-left transition-colors"
                  >
                    <div className="group-hover:text-primary font-medium transition-colors">
                      {cat.label}
                    </div>
                    <div className="text-muted-foreground mt-1 text-sm">
                      {count.toLocaleString()} series
                    </div>
                  </button>
                );
              })}
            </div>
          </div>
        )}

        {/* Series list */}
        {!isLoading && !error && currentLevel === 'series' && (
          <>
            <div className="border-border bg-muted/30 text-muted-foreground flex items-center justify-between border-b px-4 py-2 text-xs">
              <span>
                <Layers className="mr-1.5 inline h-3.5 w-3.5" />
                {(filteredItems as KalshiSeries[]).length.toLocaleString()} series
              </span>
              <Dropdown
                value={seriesSort}
                onChange={(v) => setSeriesSort(v as SeriesSortKey)}
                options={[
                  { value: 'volume', label: 'Volume' },
                  { value: 'title', label: 'Title A-Z' },
                ]}
                size="sm"
                variant="ghost"
                className="w-28"
              />
            </div>
            <div className="grid grid-cols-1 gap-3 p-4 sm:grid-cols-2 lg:grid-cols-3">
              {(filteredItems as KalshiSeries[]).map((series) => (
                <SeriesCard
                  key={series.series_ticker}
                  series={series}
                  onClick={() => handleSeriesSelect(series)}
                />
              ))}
              {filteredItems.length === 0 && (
                <div className="col-span-full">
                  <EmptyState
                    variant="search"
                    size="sm"
                    title="No series found"
                  />
                </div>
              )}
            </div>
          </>
        )}

        {/* Events list */}
        {!isLoading && !error && currentLevel === 'events' && (
          <>
            <div className="border-border bg-muted/30 text-muted-foreground flex items-center justify-between border-b px-4 py-2 text-xs">
              <span>
                <Calendar className="mr-1.5 inline h-3.5 w-3.5" />
                {(filteredItems as KalshiEvent[]).length.toLocaleString()} events
              </span>
              <Dropdown
                value={eventsSort}
                onChange={(v) => setEventsSort(v as EventSortKey)}
                options={[
                  { value: 'date', label: 'Date' },
                  { value: 'title', label: 'Title A-Z' },
                ]}
                size="sm"
                variant="ghost"
                className="w-28"
              />
            </div>
            <div className="grid grid-cols-1 gap-3 p-4 sm:grid-cols-2 lg:grid-cols-3">
              {(filteredItems as KalshiEvent[]).map((event) => (
                <EventCard
                  key={event.event_ticker}
                  event={event}
                  onClick={() => handleEventSelect(event)}
                />
              ))}
              {filteredItems.length === 0 && (
                <div className="col-span-full">
                  <EmptyState
                    variant="search"
                    size="sm"
                    title="No events found"
                  />
                </div>
              )}
            </div>
          </>
        )}

        {/* Markets list */}
        {!isLoading && !error && currentLevel === 'markets' && (
          <div className="flex flex-1">
            <div className="flex min-w-0 flex-1 flex-col">
              {/* Tabs and controls */}
              <div className="border-border bg-muted/30 flex items-center justify-between gap-4 border-b px-4 py-2">
                <TabNav
                  tabs={[
                    {
                      id: 'all',
                      label: `All (${transformedMarkets.length})`,
                      icon: <BarChart3 className="h-3.5 w-3.5" />,
                    },
                    { id: 'trending', label: 'Trending', icon: <Flame className="h-3.5 w-3.5" /> },
                    { id: 'new', label: 'New', icon: <TrendingUp className="h-3.5 w-3.5" /> },
                  ]}
                  activeTab={marketTab}
                  onChange={(id) => setMarketTab(id as MarketTab)}
                  variant="pills"
                  size="sm"
                />

                <div className="flex items-center gap-2">
                  <Dropdown
                    value={marketSort}
                    onChange={(v) => setMarketSort(v as MarketSortKey)}
                    options={[
                      { value: 'volume', label: 'Volume' },
                      { value: 'change', label: 'Change' },
                      { value: 'price', label: 'Price' },
                      { value: 'title', label: 'Title' },
                    ]}
                    size="sm"
                    variant="ghost"
                    className="w-24"
                  />
                  <div className="border-border flex items-center overflow-hidden rounded-md border">
                    <button
                      onClick={() => setViewMode('grid')}
                      className={cn(
                        'p-1.5 transition-colors',
                        viewMode === 'grid'
                          ? 'bg-primary text-primary-foreground'
                          : 'hover:bg-muted'
                      )}
                      title="Grid view"
                    >
                      <Grid3X3 className="h-4 w-4" />
                    </button>
                    <button
                      onClick={() => setViewMode('list')}
                      className={cn(
                        'p-1.5 transition-colors',
                        viewMode === 'list'
                          ? 'bg-primary text-primary-foreground'
                          : 'hover:bg-muted'
                      )}
                      title="List view"
                    >
                      <List className="h-4 w-4" />
                    </button>
                  </div>
                </div>
              </div>

              {/* Markets display */}
              <div className="flex-1 overflow-auto">
                {filteredTransformedMarkets.length > 0 ? (
                  viewMode === 'grid' ? (
                    <div className="grid grid-cols-1 gap-4 p-4 sm:grid-cols-2 xl:grid-cols-3">
                      {filteredTransformedMarkets.map((market) => (
                        <MarketCard
                          key={market.ticker}
                          market={market}
                          onClick={() => handleMarketSelect(market.raw)}
                        />
                      ))}
                    </div>
                  ) : (
                    <div className="divide-border divide-y">
                      {filteredTransformedMarkets.map((market) => (
                        <MarketRow
                          key={market.ticker}
                          market={market}
                          onClick={() => handleMarketSelect(market.raw)}
                        />
                      ))}
                    </div>
                  )
                ) : (
                  <div className="p-8">
                    <EmptyState
                      variant="search"
                      size="sm"
                      title="No markets found"
                    />
                  </div>
                )}
              </div>
            </div>

            {/* Top Movers Sidebar */}
            {topMovers.length > 0 && (
              <div className="border-border hidden w-64 overflow-auto border-l p-4 lg:block">
                <TopMoversPanel
                  movers={topMovers}
                  onMoverClick={(ticker) => {
                    const market = marketsInEvent.find((m) => m.market_ticker === ticker);
                    if (market) handleMarketSelect(market);
                  }}
                />
              </div>
            )}
          </div>
        )}
      </div>
    </div>
  );
}
