import { cn } from '@/lib/utils';
import { Badge } from '@/components/atoms/Badge';
import { ProbabilityBar } from '@/components/atoms/ProbabilityBar';
import { formatVolume, formatAmericanOdds, type TransformedMarket } from '@/lib/marketTransform';

interface MarketRowProps {
  market: TransformedMarket;
  onClick: () => void;
}

export function MarketRow({ market, onClick }: MarketRowProps) {
  const isPositive = market.change24h >= 0;
  const { sides } = market;
  const hasTwoSides = sides.hasTeams && sides.teamA && sides.teamB;

  return (
    <button
      onClick={onClick}
      className="hover:bg-muted/30 flex w-full items-center gap-4 p-4 text-left transition-colors"
    >
      {/* Left: Title and meta */}
      <div className="min-w-0 flex-1">
        <div className="mb-1 flex items-center gap-2">
          {hasTwoSides ? (
            <span className="text-sm font-medium">
              {sides.teamA} <span className="text-muted-foreground font-normal">@</span>{' '}
              {sides.teamB}
            </span>
          ) : (
            <span className="truncate text-sm font-medium">{market.title}</span>
          )}
          {market.status === 'live' && (
            <Badge
              variant="success"
              size="xs"
            >
              Live
            </Badge>
          )}
        </div>
        <div className="text-muted-foreground flex items-center gap-2 text-xs">
          <span className="font-mono">{market.ticker}</span>
          <span>•</span>
          <span>{formatVolume(market.volume24h)} vol</span>
          <span>•</span>
          <span>{market.expiresFormatted}</span>
        </div>
      </div>

      {/* Center: Probability bar (compact) */}
      <div className="hidden w-32 md:block">
        <ProbabilityBar
          probability={market.yesPrice}
          height={4}
        />
      </div>

      {/* Right: Odds */}
      <div className="flex items-center gap-4">
        {/* Change */}
        <div className="min-w-[60px] text-right">
          <span
            className={cn('text-sm font-medium', isPositive ? 'text-green-500' : 'text-red-500')}
          >
            {isPositive ? '+' : ''}
            {market.change24h.toFixed(1)}%
          </span>
          <div className="text-muted-foreground/60 text-[10px]">24h</div>
        </div>

        {/* YES side odds */}
        <div className="min-w-[70px] text-right">
          <div
            className="truncate text-[10px] text-green-400"
            title={sides.yesLabel}
          >
            {sides.yesLabel}
          </div>
          <span className="text-sm font-semibold text-green-400">
            {formatAmericanOdds(market.yesOdds)}
          </span>
        </div>

        {/* NO side odds */}
        <div className="min-w-[70px] text-right">
          <div
            className="truncate text-[10px] text-red-400"
            title={sides.noLabel}
          >
            {sides.noLabel}
          </div>
          <span className="text-sm font-semibold text-red-400">
            {formatAmericanOdds(market.noOdds)}
          </span>
        </div>
      </div>
    </button>
  );
}
