import { ChevronRight, Repeat, Tag, BarChart3 } from 'lucide-react';
import { cn } from '@/lib/utils';
import { formatVolume } from '@/lib/marketTransform';
import type { KalshiSeries } from '@/types';

interface SeriesCardProps {
  series: KalshiSeries;
  onClick: () => void;
}

export function SeriesCard({ series, onClick }: SeriesCardProps) {
  const volume = series.volume ?? (series.volume_fp ? parseFloat(series.volume_fp) : null);
  const frequency = series.frequency?.toLowerCase();
  const tags = series.tags?.slice(0, 3) || [];

  const frequencyDisplay = getFrequencyDisplay(frequency);

  return (
    <button
      onClick={onClick}
      className="bg-card border-border hover:border-primary/50 hover:bg-muted/30 group flex h-full flex-col rounded-lg border p-3 text-left transition-colors"
    >
      {/* Header: title + chevron */}
      <div className="mb-2 flex items-start justify-between gap-2">
        <h3 className="group-hover:text-primary line-clamp-2 flex-1 text-sm font-medium transition-colors">
          {series.title}
        </h3>
        <ChevronRight className="text-muted-foreground group-hover:text-primary mt-0.5 h-4 w-4 flex-shrink-0 transition-colors" />
      </div>

      {/* Meta row: frequency badge + volume */}
      <div className="mb-2 flex flex-wrap items-center gap-2">
        {frequencyDisplay && (
          <span
            className={cn(
              'inline-flex items-center gap-1 rounded px-1.5 py-0.5 text-[10px] font-medium',
              frequencyDisplay.className
            )}
          >
            <Repeat className="h-2.5 w-2.5" />
            {frequencyDisplay.label}
          </span>
        )}
        {volume !== null && volume > 0 && (
          <span className="text-muted-foreground inline-flex items-center gap-1 text-[10px]">
            <BarChart3 className="h-2.5 w-2.5" />
            {formatVolume(volume)}
          </span>
        )}
      </div>

      {/* Tags */}
      {tags.length > 0 && (
        <div className="mt-auto flex flex-wrap items-center gap-1">
          {tags.map((tag, i) => (
            <span
              key={i}
              className="bg-muted/50 text-muted-foreground inline-flex items-center gap-0.5 rounded px-1.5 py-0.5 text-[10px]"
            >
              <Tag className="h-2 w-2" />
              {tag}
            </span>
          ))}
        </div>
      )}

      {/* Ticker */}
      <div className="text-muted-foreground/60 mt-2 truncate font-mono text-[10px]">
        {series.series_ticker}
      </div>
    </button>
  );
}

function getFrequencyDisplay(
  frequency: string | undefined
): { label: string; className: string } | null {
  if (!frequency) return null;

  const frequencyMap: Record<string, { label: string; className: string }> = {
    daily: { label: 'Daily', className: 'bg-blue-500/20 text-blue-400' },
    weekly: { label: 'Weekly', className: 'bg-green-500/20 text-green-400' },
    monthly: { label: 'Monthly', className: 'bg-purple-500/20 text-purple-400' },
    quarterly: { label: 'Quarterly', className: 'bg-amber-500/20 text-amber-400' },
    yearly: { label: 'Yearly', className: 'bg-rose-500/20 text-rose-400' },
    annual: { label: 'Annual', className: 'bg-rose-500/20 text-rose-400' },
    'one-time': { label: 'One-time', className: 'bg-slate-500/20 text-slate-400' },
    once: { label: 'One-time', className: 'bg-slate-500/20 text-slate-400' },
  };

  return (
    frequencyMap[frequency] || { label: frequency, className: 'bg-slate-500/20 text-slate-400' }
  );
}
