/**
 * useMarketExplorerData - Data fetching hook for MarketExplorer
 *
 * Handles loading series, events, and markets based on URL params.
 */

import { useState, useEffect, useCallback } from 'react';
import type { KalshiApiClient } from '@/lib/kalshiApi';
import type { KalshiSeries, KalshiMarket, KalshiEvent } from '@/types';

export interface ExploreParams {
  category?: string;
  series?: string;
  event?: string;
}

type LoadingState = 'idle' | 'loading' | 'error';

interface MarketExplorerData {
  // Data
  allSeries: KalshiSeries[];
  categoryCounts: Map<string, number>;
  seriesInCategory: KalshiSeries[];
  eventsInSeries: KalshiEvent[];
  marketsInEvent: KalshiMarket[];
  selectedSeries: KalshiSeries | null;
  selectedEvent: KalshiEvent | null;

  // State
  loading: LoadingState;
  error: string | null;

  // Actions
  refresh: () => Promise<void>;
}

export function useMarketExplorerData(
  apiClient: KalshiApiClient,
  params?: ExploreParams
): MarketExplorerData {
  // Data state
  const [allSeries, setAllSeries] = useState<KalshiSeries[]>([]);
  const [categoryCounts, setCategoryCounts] = useState<Map<string, number>>(new Map());
  const [seriesInCategory, setSeriesInCategory] = useState<KalshiSeries[]>([]);
  const [eventsInSeries, setEventsInSeries] = useState<KalshiEvent[]>([]);
  const [marketsInEvent, setMarketsInEvent] = useState<KalshiMarket[]>([]);

  // Track resolved objects for breadcrumb display
  const [selectedSeries, setSelectedSeries] = useState<KalshiSeries | null>(null);
  const [selectedEvent, setSelectedEvent] = useState<KalshiEvent | null>(null);

  // Loading state
  const [loading, setLoading] = useState<LoadingState>('idle');
  const [error, setError] = useState<string | null>(null);

  // Load all series
  const loadAllSeries = useCallback(async () => {
    setLoading('loading');
    setError(null);
    try {
      const data = await apiClient.getSeries();
      setAllSeries(data);

      const counts = new Map<string, number>();
      for (const s of data) {
        const cat = s.category || 'Other';
        counts.set(cat, (counts.get(cat) || 0) + 1);
      }
      setCategoryCounts(counts);
      setLoading('idle');
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to load series');
      setLoading('error');
    }
  }, [apiClient]);

  // Load on mount
  useEffect(() => {
    loadAllSeries();
  }, [loadAllSeries]);

  // Filter series when category changes
  useEffect(() => {
    if (!params?.category) {
      setSeriesInCategory([]);
      return;
    }
    const filtered = allSeries.filter((s) => s.category === params.category);
    setSeriesInCategory(filtered);
  }, [params?.category, allSeries]);

  // Load events when series changes
  useEffect(() => {
    if (!params?.series) {
      setEventsInSeries([]);
      setSelectedSeries(null);
      return;
    }

    const series = allSeries.find((s) => s.series_ticker === params.series);
    setSelectedSeries(series ?? null);

    const loadEvents = async () => {
      setLoading('loading');
      setError(null);

      try {
        let events = await apiClient.getEvents({
          series_ticker: params.series!,
          status: 'open',
          limit: 200,
        });

        // Fallback: extract events from markets if events endpoint returns empty
        if (events.length === 0) {
          const allMarkets = await apiClient.getMarkets({
            series_ticker: params.series!,
            status: 'open',
            limit: 500,
          });
          const now = new Date();
          const markets = allMarkets.filter((m) => {
            if (!m.close_time) return true;
            return new Date(m.close_time) > now;
          });

          const eventMap = new Map<string, KalshiEvent>();
          for (const m of markets) {
            const eventTicker = m.event_ticker;
            if (!eventTicker) continue;

            if (!eventMap.has(eventTicker)) {
              eventMap.set(eventTicker, {
                event_ticker: eventTicker,
                title: extractEventTitleFromMarket(m),
                category: series?.category ?? '',
                series_ticker: params.series!,
                markets: [],
              });
            }
            eventMap.get(eventTicker)!.markets!.push(m);
          }

          events = [...eventMap.values()];
        }

        // Sort by strike_date (soonest first)
        events.sort((a, b) => {
          if (a.strike_date && b.strike_date) {
            return new Date(a.strike_date).getTime() - new Date(b.strike_date).getTime();
          }
          if (a.strike_date) return -1;
          if (b.strike_date) return 1;
          return (a.title || '').localeCompare(b.title || '');
        });

        setEventsInSeries(events);
        setLoading('idle');
      } catch (err) {
        console.error('getEvents error:', err);
        setError(err instanceof Error ? err.message : 'Failed to load events');
        setLoading('error');
      }
    };

    loadEvents();
  }, [params?.series, allSeries, apiClient]);

  // Load markets when event changes
  useEffect(() => {
    if (!params?.event) {
      setMarketsInEvent([]);
      setSelectedEvent(null);
      return;
    }

    const event = eventsInSeries.find((e) => e.event_ticker === params.event);
    setSelectedEvent(event ?? null);

    const loadMarkets = async () => {
      setLoading('loading');
      setError(null);

      try {
        const allMarkets = await apiClient.getMarkets({ event_ticker: params.event!, limit: 200 });
        const now = new Date();
        const markets = allMarkets.filter((m) => {
          if (!m.close_time) return true;
          return new Date(m.close_time) > now;
        });

        setMarketsInEvent(markets);
        setLoading('idle');
      } catch (err) {
        setError(err instanceof Error ? err.message : 'Failed to load markets');
        setLoading('error');
      }
    };

    loadMarkets();
  }, [params?.event, eventsInSeries, apiClient]);

  return {
    allSeries,
    categoryCounts,
    seriesInCategory,
    eventsInSeries,
    marketsInEvent,
    selectedSeries,
    selectedEvent,
    loading,
    error,
    refresh: loadAllSeries,
  };
}

function extractEventTitleFromMarket(market: KalshiMarket): string {
  const parts = market.title?.split(' - ') || [];
  if (parts.length > 1) {
    return parts[0].trim();
  }
  return (
    market.event_ticker?.replace(/-/g, ' ').replace(/\b\w/g, (c) => c.toUpperCase()) ||
    'Unknown Event'
  );
}
