/**
 * SweepConfirmPanel - Confirmation panel for sweeping order book liquidity
 *
 * Replaces the Quick Order form when a user Shift+clicks an order book level.
 * Shows sweep details and fires a single IOC limit order on confirm.
 */

import { cn } from '@/lib/utils';
import { Money } from '@/components/atoms/Money';
import { Zap } from 'lucide-react';

interface SweepConfirmPanelProps {
  side: 'buy' | 'sell';
  position: 'yes' | 'no';
  levels: Array<{ price: number; quantity: number }>;
  limitPrice: number;
  totalContracts: number;
  submitting: boolean;
  error?: string | null;
  onConfirm: () => void;
  onCancel: () => void;
}

export function SweepConfirmPanel({
  side,
  position,
  levels,
  limitPrice,
  totalContracts,
  submitting,
  error,
  onConfirm,
  onCancel,
}: SweepConfirmPanelProps) {
  const isBuy = side === 'buy';
  const bestPrice = levels.length > 0 ? levels[0].price : limitPrice;

  // Weighted average price across swept levels
  const weightedSum = levels.reduce((sum, l) => sum + l.price * l.quantity, 0);
  const totalQty = levels.reduce((sum, l) => sum + l.quantity, 0);
  const avgPrice = totalQty > 0 ? weightedSum / totalQty : limitPrice;

  // Total cost = sum of (price × quantity) per level / 100 → dollars
  const totalCost = levels.reduce((sum, l) => sum + (l.price * l.quantity) / 100, 0);

  // Max win = sum of ((100 - price) × quantity) per level / 100 → dollars
  const maxWin = levels.reduce((sum, l) => sum + ((100 - l.price) * l.quantity) / 100, 0);

  return (
    <div className="bg-card border-border/50 rounded-lg border p-4">
      <h3 className="mb-4 flex items-center gap-2 text-sm font-semibold">
        <Zap className={cn('h-4 w-4', isBuy ? 'text-green-500' : 'text-red-500')} />
        Book Sweep
      </h3>

      {/* Side + Position badges */}
      <div className="mb-4 flex items-center gap-2">
        <span
          className={cn(
            'rounded px-2 py-1 text-xs font-bold uppercase',
            isBuy ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400'
          )}
        >
          {side}
        </span>
        <span
          className={cn(
            'rounded px-2 py-1 text-xs font-bold uppercase',
            position === 'yes'
              ? 'bg-emerald-500/20 text-emerald-400'
              : 'bg-purple-500/20 text-purple-400'
          )}
        >
          {position}
        </span>
      </div>

      {/* Sweep details */}
      <div className="bg-muted/50 mb-4 space-y-2 rounded-lg p-3 text-xs">
        <div className="flex justify-between">
          <span className="text-muted-foreground">Price Range</span>
          <span className="font-mono">
            {bestPrice}¢ → {limitPrice}¢
          </span>
        </div>
        <div className="flex justify-between">
          <span className="text-muted-foreground">Avg Price</span>
          <span className="font-mono">{avgPrice.toFixed(1)}¢</span>
        </div>
        <div className="border-border/30 border-t pt-2">
          <div className="flex justify-between">
            <span className="text-muted-foreground">Contracts</span>
            <span className="font-mono font-medium">{totalContracts.toLocaleString()}</span>
          </div>
        </div>
        <div className="flex justify-between">
          <span className="text-muted-foreground">Total Cost</span>
          <Money
            value={totalCost}
            className="font-mono font-medium"
          />
        </div>
        <div className="flex justify-between">
          <span className="text-muted-foreground">Max Win</span>
          <Money
            value={maxWin}
            className="font-mono font-medium text-green-500"
            showDollarSign
          />
        </div>
        <div className="flex justify-between">
          <span className="text-muted-foreground">Levels</span>
          <span className="font-mono">{levels.length}</span>
        </div>
      </div>

      {/* Confirm button */}
      <button
        onClick={onConfirm}
        disabled={submitting}
        className={cn(
          'w-full rounded-lg py-3 font-semibold text-white transition-colors disabled:opacity-50',
          isBuy ? 'bg-green-500 hover:bg-green-600' : 'bg-red-500 hover:bg-red-600'
        )}
      >
        {submitting ? 'Sweeping...' : `SWEEP ${side.toUpperCase()}`}
      </button>

      {/* Cancel */}
      <button
        onClick={onCancel}
        className="text-muted-foreground hover:text-foreground mt-2 w-full py-1 text-center text-xs transition-colors"
      >
        Cancel (Esc)
      </button>

      {error && <p className="mt-2 text-center text-xs text-red-500">{error}</p>}
    </div>
  );
}

export default SweepConfirmPanel;
