/**
 * ConsolidatedMoneylines - Table showing odds comparison across platforms
 *
 * Displays NBA game moneylines from Kalshi and PolyMarket side-by-side
 * with liquidity bars for quick visual comparison.
 */

import { cn } from '@/lib/utils';
import { LiquidityBar } from '@/components/atoms/LiquidityBar';
import { Th, Td } from '@/components/atoms/TableCells';
import { SortableTh } from '@/components/atoms/SortableTh';
import { Fragment, useMemo, useState } from 'react';
import { useTableSort } from '@/hooks/useTableSort';
import type { ConsolidatedGameBooks, TeamSide, Venue } from '@/lib/nbaConsolidated/types';
import { getSeriesTicker, type Sport } from '@/lib/sportsDiscovery';
import { formatDualPriceCents } from '@/lib/nbaConsolidated/format';
import { formatStartTimePt } from '@/lib/formatters';
import { ChevronDown, ChevronUp } from 'lucide-react';
import {
  applyTakerFee,
  computeOrderPriceFromRawBid,
  getTeamWinData,
  next3NoLadders,
  next3YesLadders,
  takerFeeForVenue,
  liquidityDollars,
} from '@/lib/nbaConsolidated/pricing';
import type { OddsMode } from './DashboardHeader';
import type { SportsLoadingState } from '@/lib/sportsStream';
import { LoadingPhaseStatus } from './LoadingPhaseStatus';

export interface TradeClickRequest {
  venue: Venue;
  /** Kalshi: market_ticker. Polymarket: may be `asset:<tokenId>` for display/debug. */
  marketTicker: string;
  /** Polymarket token ID (asset_id) when venue=polymarket */
  tokenId?: string;
  /** Polymarket condition ID when venue=polymarket */
  conditionId?: string;
  tickSize?: string;
  negRisk?: boolean;
  side: 'yes' | 'no';
  /** limit price in cents */
  priceCents: number;
  /** default contracts */
  contracts: number;
  isTakerMode: boolean;
  teamLabel: string;
  gameKey: string;
  gameDate: string;
  startTimePt?: string | null;
}

interface ConsolidatedMoneylinesProps {
  games: ConsolidatedGameBooks[];
  isLoading?: boolean;
  loadingState?: SportsLoadingState;
  oddsMode: OddsMode;
  maxLiquidity?: number;
  expandedGameKeys: Set<string>;
  onToggleExpand: (gameKey: string) => void;
  onTradeClick?: (req: TradeClickRequest) => void;
  /** Navigate to MarketDetailView (primary click). Shift+click still opens trade modal. */
  onNavigateToMarket?: (marketTicker: string) => void;
  /** Sport for team name lookup and title (default: nba) */
  sport?: Sport;
  className?: string;
}

export function ConsolidatedMoneylines({
  games,
  isLoading = false,
  loadingState,
  maxLiquidity = 100000,
  oddsMode,
  expandedGameKeys,
  onToggleExpand,
  onTradeClick,
  onNavigateToMarket,
  sport = 'nba',
  className,
}: ConsolidatedMoneylinesProps) {
  const [showAllRows, setShowAllRows] = useState(true);
  const maxRows = 10;

  const sortExtractors = useMemo(
    () => ({
      rotation: (game: ConsolidatedGameBooks) => game.awayRotation ?? null,
    }),
    []
  );

  const { sorted, sortKey, sortDir, handleSort } = useTableSort(games, sortExtractors, {
    defaultSortKey: 'time',
    defaultSortDir: 'asc',
  });

  const visibleGames = useMemo(() => {
    if (showAllRows) return sorted;
    return sorted.slice(0, maxRows);
  }, [sorted, showAllRows]);

  const sportLabel = sport.toUpperCase().replace('-', ' ');
  const seriesTicker = getSeriesTicker(sport, 'moneyline');

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header with sport-prefixed title and series ticker */}
      <div className="border-border border-b px-4 py-4">
        <div className="flex items-center justify-between">
          <h2 className="text-foreground text-lg font-semibold">{sportLabel} Moneylines</h2>
          <div className="flex items-center gap-3">
            {seriesTicker && (
              <span className="text-muted-foreground font-mono text-xs">{seriesTicker}</span>
            )}
            <LoadingPhaseStatus
              isLoading={isLoading}
              loadingState={loadingState}
            />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <SortableTh
                sortKey="rotation"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
                className="w-16"
              >
                Roto
              </SortableTh>
              <SortableTh
                sortKey="date"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Game Date
              </SortableTh>
              <SortableTh
                sortKey="time"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Time PT
              </SortableTh>
              <SortableTh
                sortKey="team"
                activeSortKey={sortKey}
                activeSortDir={sortDir}
                onSort={handleSort}
              >
                Away Team
              </SortableTh>
              <Th>Home Team</Th>
              <Th className="text-right">Away Kalshi</Th>
              <Th className="text-right">Home Kalshi</Th>
              <Th className="w-16 text-right">Away K Vol</Th>
              <Th className="w-16 text-right">Home K Vol</Th>
              <Th className="w-24">Away K Liq</Th>
              <Th className="w-24">Home K Liq</Th>
              <Th className="text-right">Away Poly</Th>
              <Th className="text-right">Home Poly</Th>
              <Th className="w-16 text-right">Away P Vol</Th>
              <Th className="w-16 text-right">Home P Vol</Th>
              <Th className="w-24">Away P Liq</Th>
              <Th className="w-24">Home P Liq</Th>
            </tr>
          </thead>
          <tbody className="divide-border divide-y">
            {isLoading && games.length === 0
              ? Array.from({ length: 10 }).map((_, i) => (
                  <tr
                    key={`loading-${i}`}
                    className="animate-pulse"
                  >
                    <Td>
                      <div className="bg-muted/40 h-3 w-12 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-20 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-14 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-24 rounded" />
                    </Td>
                    <Td>
                      <div className="bg-muted/40 h-3 w-24 rounded" />
                    </Td>
                    {/* Kalshi prices */}
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    {/* Kalshi vol */}
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                    </Td>
                    {/* Kalshi liq */}
                    <Td className="p-1">
                      <div className="bg-muted/30 h-6 w-full rounded" />
                    </Td>
                    <Td className="p-1">
                      <div className="bg-muted/30 h-6 w-full rounded" />
                    </Td>
                    {/* Poly prices */}
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-10 rounded" />
                    </Td>
                    {/* Poly vol */}
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                    </Td>
                    <Td className="text-right">
                      <div className="bg-muted/40 ml-auto h-3 w-8 rounded" />
                    </Td>
                    {/* Poly liq */}
                    <Td className="p-1">
                      <div className="bg-muted/30 h-6 w-full rounded" />
                    </Td>
                    <Td className="p-1">
                      <div className="bg-muted/30 h-6 w-full rounded" />
                    </Td>
                  </tr>
                ))
              : visibleGames.map((game) => (
                  <Fragment key={game.key}>
                    {(() => {
                      const isTakerMode = oddsMode === 'taker';
                      const awayName = game.awayName ?? game.awayCode;
                      const homeName = game.homeName ?? game.homeCode;

                      const awayK = getTeamWinData({
                        game: game.kalshi ?? null,
                        venue: 'kalshi',
                        teamSide: 'away',
                        isTakerMode,
                      });
                      const homeK = getTeamWinData({
                        game: game.kalshi ?? null,
                        venue: 'kalshi',
                        teamSide: 'home',
                        isTakerMode,
                      });
                      const awayP = getTeamWinData({
                        game: game.polymarket ?? null,
                        venue: 'polymarket',
                        teamSide: 'away',
                        isTakerMode,
                      });
                      const homeP = getTeamWinData({
                        game: game.polymarket ?? null,
                        venue: 'polymarket',
                        teamSide: 'home',
                        isTakerMode,
                      });

                      const awayKDollars = liquidityDollars(awayK.liq, awayK.priceCents);
                      const homeKDollars = liquidityDollars(homeK.liq, homeK.priceCents);
                      const awayPDollars = liquidityDollars(awayP.liq, awayP.priceCents);
                      const homePDollars = liquidityDollars(homeP.liq, homeP.priceCents);

                      const awayKDisplay = formatDualPriceCents(awayK.priceCents);
                      const homeKDisplay = formatDualPriceCents(homeK.priceCents);
                      const awayPDisplay = formatDualPriceCents(awayP.priceCents);
                      const homePDisplay = formatDualPriceCents(homeP.priceCents);

                      const expanded = expandedGameKeys.has(game.key);

                      const handleTrade = (teamSide: TeamSide, e: React.MouseEvent) => {
                        const top = teamSide === 'away' ? awayK : homeK;
                        const marketTicker = top.marketTicker;
                        if (!marketTicker) return;

                        // Shift+click: open trade modal (legacy behavior)
                        if (e.shiftKey && onTradeClick) {
                          const teamLabel = teamSide === 'away' ? awayName : homeName;
                          const orderPrice = computeOrderPriceFromRawBid(
                            top.rawBidPriceCents,
                            isTakerMode
                          );
                          if (orderPrice === null) return;
                          onTradeClick({
                            venue: 'kalshi',
                            marketTicker,
                            side: 'no',
                            priceCents: orderPrice,
                            contracts: 100,
                            isTakerMode,
                            teamLabel,
                            gameKey: game.key,
                            gameDate: game.date,
                            startTimePt: game.startTimePt,
                          });
                          return;
                        }

                        // Primary click: navigate to market detail
                        if (onNavigateToMarket) {
                          onNavigateToMarket(marketTicker);
                        }
                      };

                      const handlePolyTrade = (teamSide: TeamSide, e: React.MouseEvent) => {
                        const top = teamSide === 'away' ? awayP : homeP;
                        const marketTicker = top.marketTicker;
                        if (!marketTicker) return;

                        // Shift+click: open trade modal (legacy behavior)
                        if (e.shiftKey && onTradeClick) {
                          const teamLabel = teamSide === 'away' ? awayName : homeName;
                          const orderPrice = computeOrderPriceFromRawBid(
                            top.rawBidPriceCents,
                            isTakerMode
                          );
                          if (orderPrice === null) return;

                          // Team win is expressed as NO on opponent.
                          const opponentMarket =
                            teamSide === 'away'
                              ? game.polymarket?.markets.home
                              : game.polymarket?.markets.away;

                          onTradeClick({
                            venue: 'polymarket',
                            marketTicker,
                            tokenId: opponentMarket?.tokenId,
                            conditionId: opponentMarket?.conditionId,
                            tickSize: opponentMarket?.tickSize,
                            negRisk: opponentMarket?.negRisk,
                            side: 'no',
                            priceCents: orderPrice,
                            contracts: 100,
                            isTakerMode,
                            teamLabel,
                            gameKey: game.key,
                            gameDate: game.date,
                            startTimePt: game.startTimePt,
                          });
                          return;
                        }

                        // Primary click: navigate to market detail
                        // For Polymarket, use the team's own market ticker if available
                        if (onNavigateToMarket) {
                          onNavigateToMarket(marketTicker);
                        }
                      };

                      const handleLadderTrade = (
                        teamSide: TeamSide,
                        ladderPrice: number | null,
                        marketTicker: string | null,
                        e: React.MouseEvent
                      ) => {
                        if (!marketTicker) return;

                        // Shift+click: open trade modal (legacy behavior)
                        if (e.shiftKey && onTradeClick && ladderPrice !== null) {
                          const teamLabel = teamSide === 'away' ? awayName : homeName;
                          const orderPrice = ladderPrice;
                          onTradeClick({
                            venue: 'kalshi',
                            marketTicker,
                            side: 'no',
                            priceCents: orderPrice,
                            contracts: 100,
                            isTakerMode,
                            teamLabel,
                            gameKey: game.key,
                            gameDate: game.date,
                            startTimePt: game.startTimePt,
                          });
                          return;
                        }

                        // Primary click: navigate to market detail
                        if (onNavigateToMarket) {
                          onNavigateToMarket(marketTicker);
                        }
                      };

                      const handleLadderPolyTrade = (
                        teamSide: TeamSide,
                        ladderPrice: number | null,
                        e: React.MouseEvent
                      ) => {
                        const opponentMarket =
                          teamSide === 'away'
                            ? game.polymarket?.markets.home
                            : game.polymarket?.markets.away;

                        // Shift+click: open trade modal (legacy behavior)
                        if (e.shiftKey && onTradeClick && ladderPrice !== null) {
                          const teamLabel = teamSide === 'away' ? awayName : homeName;
                          if (!opponentMarket?.tokenId) return;

                          onTradeClick({
                            venue: 'polymarket',
                            marketTicker: `asset:${opponentMarket.tokenId}`,
                            tokenId: opponentMarket.tokenId,
                            conditionId: opponentMarket.conditionId,
                            tickSize: opponentMarket.tickSize,
                            negRisk: opponentMarket.negRisk,
                            side: 'no',
                            priceCents: ladderPrice,
                            contracts: 100,
                            isTakerMode,
                            teamLabel,
                            gameKey: game.key,
                            gameDate: game.date,
                            startTimePt: game.startTimePt,
                          });
                          return;
                        }

                        // Primary click: navigate to market detail (Polymarket markets start with asset: prefix)
                        const marketTicker = opponentMarket?.tokenId
                          ? `asset:${opponentMarket.tokenId}`
                          : null;
                        if (marketTicker && onNavigateToMarket) {
                          onNavigateToMarket(marketTicker);
                        }
                      };

                      const renderLadderRow = (idx: number) => {
                        // In maker mode, ladder comes from NO bids; in taker mode, from YES bids (for inversion).
                        const awayLvls = isTakerMode
                          ? next3YesLadders(awayK.yesLevels, awayK.rawBidPriceCents)
                          : next3NoLadders(awayK.noLevels, awayK.rawBidPriceCents);
                        const homeLvls = isTakerMode
                          ? next3YesLadders(homeK.yesLevels, homeK.rawBidPriceCents)
                          : next3NoLadders(homeK.noLevels, homeK.rawBidPriceCents);
                        const awayPLvls = isTakerMode
                          ? next3YesLadders(awayP.yesLevels, awayP.rawBidPriceCents)
                          : next3NoLadders(awayP.noLevels, awayP.rawBidPriceCents);
                        const homePLvls = isTakerMode
                          ? next3YesLadders(homeP.yesLevels, homeP.rawBidPriceCents)
                          : next3NoLadders(homeP.noLevels, homeP.rawBidPriceCents);

                        const awayLvl = awayLvls[idx] ?? null;
                        const homeLvl = homeLvls[idx] ?? null;
                        const awayPLvl = awayPLvls[idx] ?? null;
                        const homePLvl = homePLvls[idx] ?? null;

                        const feeRate = takerFeeForVenue('kalshi');
                        const awayPrice = awayLvl
                          ? isTakerMode
                            ? applyTakerFee(100 - awayLvl.priceCents, feeRate)
                            : awayLvl.priceCents
                          : null;
                        const homePrice = homeLvl
                          ? isTakerMode
                            ? applyTakerFee(100 - homeLvl.priceCents, feeRate)
                            : homeLvl.priceCents
                          : null;
                        const polyFee = takerFeeForVenue('polymarket');
                        const awayPPrice = awayPLvl
                          ? isTakerMode
                            ? applyTakerFee(100 - awayPLvl.priceCents, polyFee)
                            : awayPLvl.priceCents
                          : null;
                        const homePPrice = homePLvl
                          ? isTakerMode
                            ? applyTakerFee(100 - homePLvl.priceCents, polyFee)
                            : homePLvl.priceCents
                          : null;

                        const awayDollars = awayLvl
                          ? liquidityDollars(awayLvl.size, awayPrice)
                          : null;
                        const homeDollars = homeLvl
                          ? liquidityDollars(homeLvl.size, homePrice)
                          : null;
                        const awayPD = awayPLvl
                          ? liquidityDollars(awayPLvl.size, awayPPrice)
                          : null;
                        const homePD = homePLvl
                          ? liquidityDollars(homePLvl.size, homePPrice)
                          : null;

                        return (
                          <tr
                            key={`${game.key}-ladder-${idx}`}
                            className=""
                          >
                            <Td> </Td>
                            <Td className="font-mono text-[10px]"> </Td>
                            <Td className="font-mono text-[10px]"> </Td>
                            <Td className="text-xs"> </Td>
                            <Td className="text-xs"> </Td>
                            {/* Kalshi ladder prices */}
                            <Td
                              className={cn(
                                'text-right font-mono',
                                awayPrice !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleLadderTrade('away', awayPrice, awayK.marketTicker, e);
                              }}
                              title={
                                awayK.marketTicker
                                  ? `Click to view market${awayPrice !== null ? ' (Shift+click to trade)' : ''}`
                                  : undefined
                              }
                            >
                              {formatDualPriceCents(awayPrice)}
                            </Td>
                            <Td
                              className={cn(
                                'text-right font-mono',
                                homePrice !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleLadderTrade('home', homePrice, homeK.marketTicker, e);
                              }}
                              title={
                                homeK.marketTicker
                                  ? `Click to view market${homePrice !== null ? ' (Shift+click to trade)' : ''}`
                                  : undefined
                              }
                            >
                              {formatDualPriceCents(homePrice)}
                            </Td>
                            {/* Kalshi ladder volume */}
                            <Td className="text-right font-mono text-xs">
                              {awayLvl ? awayLvl.size.toLocaleString() : ''}
                            </Td>
                            <Td className="text-right font-mono text-xs">
                              {homeLvl ? homeLvl.size.toLocaleString() : ''}
                            </Td>
                            {/* Kalshi ladder liquidity */}
                            <Td>
                              <LiquidityBar
                                value={awayDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            <Td>
                              <LiquidityBar
                                value={homeDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            {/* Poly ladder prices */}
                            <Td
                              className={cn(
                                'text-right font-mono',
                                awayPPrice !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleLadderPolyTrade('away', awayPPrice, e);
                              }}
                              title={
                                awayPPrice !== null
                                  ? `Click to view market (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {formatDualPriceCents(awayPPrice)}
                            </Td>
                            <Td
                              className={cn(
                                'text-right font-mono',
                                homePPrice !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleLadderPolyTrade('home', homePPrice, e);
                              }}
                              title={
                                homePPrice !== null
                                  ? `Click to view market (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {formatDualPriceCents(homePPrice)}
                            </Td>
                            {/* Poly ladder volume */}
                            <Td className="text-right font-mono text-xs">
                              {awayPLvl ? awayPLvl.size.toLocaleString() : ''}
                            </Td>
                            <Td className="text-right font-mono text-xs">
                              {homePLvl ? homePLvl.size.toLocaleString() : ''}
                            </Td>
                            {/* Poly ladder liquidity */}
                            <Td>
                              <LiquidityBar
                                value={awayPD ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            <Td>
                              <LiquidityBar
                                value={homePD ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                          </tr>
                        );
                      };

                      return (
                        <>
                          <tr
                            key={game.key}
                            className="hover:bg-accent/50 cursor-pointer transition-colors"
                            onClick={() => {
                              onToggleExpand(game.key);
                            }}
                          >
                            <Td className="text-muted-foreground font-mono text-xs">
                              {game.awayRotation != null && game.homeRotation != null
                                ? `${game.awayRotation}/${game.homeRotation}`
                                : '—'}
                            </Td>
                            <Td className="text-muted-foreground font-mono text-xs">{game.date}</Td>
                            <Td className="text-muted-foreground font-mono text-xs">
                              {formatStartTimePt(game.startTimePt)}
                            </Td>
                            <Td className="font-medium">{awayName}</Td>
                            <Td className="font-medium">{homeName}</Td>
                            {/* Kalshi prices */}
                            <Td
                              className={cn(
                                'text-right font-mono',
                                awayK.priceCents !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleTrade('away', e);
                              }}
                              title={
                                awayK.marketTicker
                                  ? `Click to view ${awayK.marketTicker} (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {awayKDisplay}
                            </Td>
                            <Td
                              className={cn(
                                'text-right font-mono',
                                homeK.priceCents !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handleTrade('home', e);
                              }}
                              title={
                                homeK.marketTicker
                                  ? `Click to view ${homeK.marketTicker} (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {homeKDisplay}
                            </Td>
                            {/* Kalshi volume (contracts) */}
                            <Td className="text-right font-mono text-xs">
                              {awayK.liq != null ? awayK.liq.toLocaleString() : '—'}
                            </Td>
                            <Td className="text-right font-mono text-xs">
                              {homeK.liq != null ? homeK.liq.toLocaleString() : '—'}
                            </Td>
                            {/* Kalshi liquidity ($) */}
                            <Td>
                              <LiquidityBar
                                value={awayKDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            <Td>
                              <LiquidityBar
                                value={homeKDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            {/* Poly prices */}
                            <Td
                              className={cn(
                                'text-right font-mono',
                                awayP.priceCents !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handlePolyTrade('away', e);
                              }}
                              title={
                                awayP.marketTicker
                                  ? `Click to view market (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {awayPDisplay}
                            </Td>
                            <Td
                              className={cn(
                                'text-right font-mono',
                                homeP.priceCents !== null && 'hover:underline'
                              )}
                              onClick={(e: React.MouseEvent<HTMLTableCellElement>) => {
                                e.stopPropagation();
                                handlePolyTrade('home', e);
                              }}
                              title={
                                homeP.marketTicker
                                  ? `Click to view market (Shift+click to trade)`
                                  : undefined
                              }
                            >
                              {homePDisplay}
                            </Td>
                            {/* Poly volume (contracts) */}
                            <Td className="text-right font-mono text-xs">
                              {awayP.liq != null ? awayP.liq.toLocaleString() : '—'}
                            </Td>
                            <Td className="text-right font-mono text-xs">
                              {homeP.liq != null ? homeP.liq.toLocaleString() : '—'}
                            </Td>
                            {/* Poly liquidity ($) */}
                            <Td>
                              <LiquidityBar
                                value={awayPDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                            <Td>
                              <LiquidityBar
                                value={homePDollars ?? 0}
                                maxValue={maxLiquidity}
                              />
                            </Td>
                          </tr>

                          {expanded && (
                            <>
                              {renderLadderRow(0)}
                              {renderLadderRow(1)}
                              {renderLadderRow(2)}
                            </>
                          )}
                        </>
                      );
                    })()}
                  </Fragment>
                ))}
          </tbody>
        </table>
      </div>

      {/* Expand / minimize (show more rows) */}
      {games.length > maxRows && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
            aria-label={showAllRows ? 'Show fewer games' : 'Show more games'}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${games.length}`
                : `Showing ${Math.min(maxRows, games.length)} of ${games.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}

      {/* Empty state */}
      {games.length === 0 && !isLoading && (
        <div className="text-muted-foreground p-8 text-center">No games available</div>
      )}
    </div>
  );
}

export default ConsolidatedMoneylines;
