/**
 * GameDetailPanel - Right-side panel showing exchange-style order books
 *
 * Pops up when a game is clicked in the consolidated moneylines table.
 * Shows Kalshi + Polymarket order books for the selected team with
 * sweep depth, cost/payout, and quick trade buttons.
 */

import { useState, useMemo } from 'react';
import { X, BookOpen } from 'lucide-react';
import { cn } from '@/lib/utils';
import { Money } from '@/components/atoms/Money';
import { ExchangeOrderBook, type SweepInfo } from '@/components/organisms/ExchangeOrderBook';
import type {
  ConsolidatedGameBooks,
  TeamSide,
  Venue,
  BookLevel,
} from '@/lib/nbaConsolidated/types';
import {
  getTeamWinData,
  computeOrderPriceFromRawBid,
  liquidityDollars,
} from '@/lib/nbaConsolidated/pricing';
import { formatDualPriceCents } from '@/lib/nbaConsolidated/format';
import { formatStartTimePt } from '@/lib/formatters';
import type { OddsMode } from './DashboardHeader';
import type { TradeClickRequest } from './ConsolidatedMoneylines';

interface GameDetailPanelProps {
  game: ConsolidatedGameBooks;
  oddsMode: OddsMode;
  onTradeClick?: (req: TradeClickRequest) => void;
  onClose: () => void;
  className?: string;
}

/** Derive YES asks from NO bids via inversion */
function deriveAsks(noBids: BookLevel[]): BookLevel[] {
  return noBids.map((l) => ({
    priceCents: 100 - l.priceCents,
    size: l.size,
  }));
}

/** Compute spread between best bid and best ask */
function computeSpread(bids: BookLevel[], asks: BookLevel[]): number | null {
  if (bids.length === 0 || asks.length === 0) return null;
  const bestBid = Math.max(...bids.map((b) => b.priceCents));
  const bestAsk = Math.min(...asks.map((a) => a.priceCents));
  return bestAsk - bestBid;
}

/** Compute last price (midpoint of best bid/ask) */
function computeLastPrice(bids: BookLevel[], asks: BookLevel[]): number | null {
  if (bids.length === 0 && asks.length === 0) return null;
  const bestBid = bids.length > 0 ? Math.max(...bids.map((b) => b.priceCents)) : null;
  const bestAsk = asks.length > 0 ? Math.min(...asks.map((a) => a.priceCents)) : null;
  if (bestBid !== null && bestAsk !== null) return (bestBid + bestAsk) / 2;
  return bestBid ?? bestAsk;
}

export function GameDetailPanel({
  game,
  oddsMode,
  onTradeClick,
  onClose,
  className,
}: GameDetailPanelProps) {
  const [selectedTeam, setSelectedTeam] = useState<TeamSide>('away');
  const isTakerMode = oddsMode === 'taker';

  const awayName = game.awayName ?? game.awayCode;
  const homeName = game.homeName ?? game.homeCode;
  const selectedTeamName = selectedTeam === 'away' ? awayName : homeName;

  // Compute books for the selected team
  const books = useMemo(() => {
    const teamSide = selectedTeam;

    // Kalshi book
    const kalshiMarket =
      teamSide === 'away' ? game.kalshi?.markets.away : game.kalshi?.markets.home;
    const kalshiYesBids = kalshiMarket?.yes ?? [];
    const kalshiNoBids = kalshiMarket?.no ?? [];
    const kalshiAsks = deriveAsks(kalshiNoBids);
    const kalshiSpread = computeSpread(kalshiYesBids, kalshiAsks);
    const kalshiLast = computeLastPrice(kalshiYesBids, kalshiAsks);

    // Polymarket book
    const polyMarket =
      teamSide === 'away' ? game.polymarket?.markets.away : game.polymarket?.markets.home;
    const polyYesBids = polyMarket?.yes ?? [];
    const polyNoBids = polyMarket?.no ?? [];
    const polyAsks = deriveAsks(polyNoBids);
    const polySpread = computeSpread(polyYesBids, polyAsks);
    const polyLast = computeLastPrice(polyYesBids, polyAsks);

    return {
      kalshi: {
        bids: kalshiYesBids,
        asks: kalshiAsks,
        spread: kalshiSpread,
        lastPrice: kalshiLast,
        hasData: kalshiYesBids.length > 0 || kalshiNoBids.length > 0,
      },
      poly: {
        bids: polyYesBids,
        asks: polyAsks,
        spread: polySpread,
        lastPrice: polyLast,
        hasData: polyYesBids.length > 0 || polyNoBids.length > 0,
      },
    };
  }, [game, selectedTeam]);

  // Get top prices for the price summary
  const awayK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'away',
    isTakerMode,
  });
  const homeK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'home',
    isTakerMode,
  });
  const awayP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'away',
    isTakerMode,
  });
  const homeP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'home',
    isTakerMode,
  });

  // Trade handler
  const handleTrade = (teamSide: TeamSide, venue: Venue) => {
    if (!onTradeClick) return;

    const top =
      venue === 'kalshi'
        ? teamSide === 'away'
          ? awayK
          : homeK
        : teamSide === 'away'
          ? awayP
          : homeP;

    const teamLabel = teamSide === 'away' ? awayName : homeName;
    const marketTicker = top.marketTicker;
    const orderPrice = computeOrderPriceFromRawBid(top.rawBidPriceCents, isTakerMode);
    if (!marketTicker || orderPrice === null) return;

    if (venue === 'kalshi') {
      onTradeClick({
        venue: 'kalshi',
        marketTicker,
        side: 'no',
        priceCents: orderPrice,
        contracts: 100,
        isTakerMode,
        teamLabel,
        gameKey: game.key,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
    } else {
      const opponentMarket =
        teamSide === 'away' ? game.polymarket?.markets.home : game.polymarket?.markets.away;
      onTradeClick({
        venue: 'polymarket',
        marketTicker,
        tokenId: opponentMarket?.tokenId,
        conditionId: opponentMarket?.conditionId,
        tickSize: opponentMarket?.tickSize,
        negRisk: opponentMarket?.negRisk,
        side: 'no',
        priceCents: orderPrice,
        contracts: 100,
        isTakerMode,
        teamLabel,
        gameKey: game.key,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
    }
  };

  // Sweep click handler (from ExchangeOrderBook)
  const handleSweepClick = (info: SweepInfo, venue: Venue) => {
    // For now, log sweep info. Could pre-fill the trade modal with sweep quantities.
    console.log(`[GameDetailPanel] Sweep ${venue}:`, info);
    handleTrade(selectedTeam, venue);
  };

  return (
    <div
      className={cn(
        'bg-card border-border flex h-full w-[380px] flex-shrink-0 flex-col overflow-hidden rounded-lg border',
        className
      )}
    >
      {/* Header */}
      <div className="border-border flex items-center justify-between border-b px-4 py-3">
        <div className="min-w-0 flex-1">
          <div className="flex items-center gap-2">
            <BookOpen className="text-primary h-4 w-4 flex-shrink-0" />
            <h3 className="truncate text-sm font-semibold">
              {awayName} @ {homeName}
            </h3>
          </div>
          <div className="text-muted-foreground mt-0.5 flex items-center gap-2 text-xs">
            <span>{game.date}</span>
            {game.startTimePt && (
              <>
                <span>·</span>
                <span>{formatStartTimePt(game.startTimePt)} PT</span>
              </>
            )}
          </div>
        </div>
        <button
          onClick={onClose}
          className="hover:bg-muted text-muted-foreground hover:text-foreground ml-2 flex-shrink-0 rounded-lg p-1.5 transition-colors"
          title="Close (Esc)"
        >
          <X className="h-4 w-4" />
        </button>
      </div>

      {/* Team tabs */}
      <div className="border-border border-b px-4 py-2">
        <div className="bg-muted flex rounded-lg p-0.5">
          <button
            onClick={() => setSelectedTeam('away')}
            className={cn(
              'flex-1 rounded-md px-3 py-1.5 text-xs font-medium transition-colors',
              selectedTeam === 'away'
                ? 'bg-primary text-primary-foreground'
                : 'text-muted-foreground hover:text-foreground'
            )}
          >
            {awayName}
          </button>
          <button
            onClick={() => setSelectedTeam('home')}
            className={cn(
              'flex-1 rounded-md px-3 py-1.5 text-xs font-medium transition-colors',
              selectedTeam === 'home'
                ? 'bg-primary text-primary-foreground'
                : 'text-muted-foreground hover:text-foreground'
            )}
          >
            {homeName}
          </button>
        </div>
      </div>

      {/* Scrollable content */}
      <div className="min-h-0 flex-1 space-y-3 overflow-y-auto p-3">
        {/* Price summary */}
        <div className="grid grid-cols-2 gap-2">
          <div className="bg-muted/30 rounded-lg px-3 py-2">
            <div className="text-muted-foreground mb-1 text-[10px] font-medium uppercase">
              {awayName}
            </div>
            <div className="space-y-0.5 text-xs">
              {awayK.priceCents !== null && (
                <div className="flex justify-between">
                  <span className="text-blue-400">K</span>
                  <span className="font-mono">{formatDualPriceCents(awayK.priceCents)}</span>
                </div>
              )}
              {awayP.priceCents !== null && (
                <div className="flex justify-between">
                  <span className="text-purple-400">P</span>
                  <span className="font-mono">{formatDualPriceCents(awayP.priceCents)}</span>
                </div>
              )}
            </div>
          </div>
          <div className="bg-muted/30 rounded-lg px-3 py-2">
            <div className="text-muted-foreground mb-1 text-[10px] font-medium uppercase">
              {homeName}
            </div>
            <div className="space-y-0.5 text-xs">
              {homeK.priceCents !== null && (
                <div className="flex justify-between">
                  <span className="text-blue-400">K</span>
                  <span className="font-mono">{formatDualPriceCents(homeK.priceCents)}</span>
                </div>
              )}
              {homeP.priceCents !== null && (
                <div className="flex justify-between">
                  <span className="text-purple-400">P</span>
                  <span className="font-mono">{formatDualPriceCents(homeP.priceCents)}</span>
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Kalshi Order Book */}
        {books.kalshi.hasData && (
          <ExchangeOrderBook
            bids={books.kalshi.bids}
            asks={books.kalshi.asks}
            lastPrice={books.kalshi.lastPrice}
            spread={books.kalshi.spread}
            venue="kalshi"
            teamLabel={`${selectedTeamName} Wins`}
            onLevelClick={(info) => handleSweepClick(info, 'kalshi')}
          />
        )}

        {/* Polymarket Order Book */}
        {books.poly.hasData && (
          <ExchangeOrderBook
            bids={books.poly.bids}
            asks={books.poly.asks}
            lastPrice={books.poly.lastPrice}
            spread={books.poly.spread}
            venue="polymarket"
            teamLabel={`${selectedTeamName} Wins`}
            onLevelClick={(info) => handleSweepClick(info, 'polymarket')}
          />
        )}

        {/* No data state */}
        {!books.kalshi.hasData && !books.poly.hasData && (
          <div className="text-muted-foreground py-8 text-center text-sm">
            No order book data available
          </div>
        )}

        {/* Quick Trade Buttons */}
        <div>
          <div className="text-muted-foreground mb-2 text-[10px] font-medium uppercase tracking-wider">
            Quick Trade — {selectedTeamName}
          </div>
          <div className="grid grid-cols-2 gap-2">
            {/* Kalshi */}
            {(() => {
              const top = selectedTeam === 'away' ? awayK : homeK;
              const liqDollars = liquidityDollars(top.liq, top.priceCents);
              return (
                <button
                  onClick={() => handleTrade(selectedTeam, 'kalshi')}
                  disabled={top.priceCents === null}
                  className={cn(
                    'flex flex-col items-center gap-1 rounded-lg border px-3 py-2.5 text-sm transition-colors',
                    top.priceCents !== null
                      ? 'cursor-pointer border-blue-500/30 hover:bg-blue-500/10'
                      : 'border-border/30 cursor-not-allowed opacity-50'
                  )}
                >
                  <span className="rounded bg-blue-500/20 px-1.5 py-0.5 text-[10px] font-bold text-blue-400">
                    Kalshi
                  </span>
                  {top.priceCents !== null && (
                    <span className="font-mono text-xs font-medium">
                      {formatDualPriceCents(top.priceCents)}
                    </span>
                  )}
                  {liqDollars !== null && (
                    <Money
                      value={liqDollars}
                      className="text-muted-foreground text-[10px]"
                    />
                  )}
                </button>
              );
            })()}

            {/* Polymarket */}
            {(() => {
              const top = selectedTeam === 'away' ? awayP : homeP;
              const liqDollars = liquidityDollars(top.liq, top.priceCents);
              return (
                <button
                  onClick={() => handleTrade(selectedTeam, 'polymarket')}
                  disabled={top.priceCents === null}
                  className={cn(
                    'flex flex-col items-center gap-1 rounded-lg border px-3 py-2.5 text-sm transition-colors',
                    top.priceCents !== null
                      ? 'cursor-pointer border-purple-500/30 hover:bg-purple-500/10'
                      : 'border-border/30 cursor-not-allowed opacity-50'
                  )}
                >
                  <span className="rounded bg-purple-500/20 px-1.5 py-0.5 text-[10px] font-bold text-purple-400">
                    Poly
                  </span>
                  {top.priceCents !== null && (
                    <span className="font-mono text-xs font-medium">
                      {formatDualPriceCents(top.priceCents)}
                    </span>
                  )}
                  {liqDollars !== null && (
                    <Money
                      value={liqDollars}
                      className="text-muted-foreground text-[10px]"
                    />
                  )}
                </button>
              );
            })()}
          </div>
        </div>
      </div>
    </div>
  );
}
