/**
 * InlineBookPanel - Inline order book + quick trade panel for the Value Dashboard
 *
 * Shows a compact view of the orderbook for a selected game, with quick-trade
 * buttons that hook into the existing PlaceOrderModal flow. Designed to appear
 * as a right pane displacing the odds table width when a game is selected.
 *
 * Data comes from the ConsolidatedGameBooks already available in the dashboard,
 * so no additional API calls or streaming connections are needed.
 */

import { useMemo } from 'react';
import { X, BookOpen } from 'lucide-react';
import { cn } from '@/lib/utils';
import { Money } from '@/components/atoms/Money';
import type {
  ConsolidatedGameBooks,
  TeamSide,
  Venue,
  BookLevel,
} from '@/lib/nbaConsolidated/types';
import {
  getTeamWinData,
  computeOrderPriceFromRawBid,
  liquidityDollars,
} from '@/lib/nbaConsolidated/pricing';
import { formatDualPriceCents } from '@/lib/nbaConsolidated/format';
import { formatStartTimePt } from '@/lib/formatters';
import type { OddsMode } from './DashboardHeader';
import type { TradeClickRequest } from './ConsolidatedMoneylines';

interface InlineBookPanelProps {
  /** The selected game to display */
  game: ConsolidatedGameBooks;
  /** Current odds mode (maker/taker) */
  oddsMode: OddsMode;
  /** Callback to fire a trade click (opens PlaceOrderModal) */
  onTradeClick?: (req: TradeClickRequest) => void;
  /** Callback when the panel is dismissed */
  onClose: () => void;
  className?: string;
}

/** Render a compact orderbook ladder for one venue + team side */
function MiniBook({
  levels,
  side,
  maxSize,
  label,
}: {
  levels: BookLevel[];
  side: 'bid' | 'ask';
  maxSize: number;
  label: string;
}) {
  const isBid = side === 'bid';
  const colorClass = isBid ? 'text-green-400' : 'text-red-400';
  const bgClass = isBid ? 'bg-green-500/10' : 'bg-red-500/10';

  return (
    <div className="space-y-0.5">
      <div className="text-muted-foreground text-[10px] font-medium uppercase tracking-wider">
        {label}
      </div>
      {levels.length === 0 ? (
        <div className="text-muted-foreground py-2 text-center text-xs">No data</div>
      ) : (
        levels.slice(0, 5).map((lvl, i) => {
          const barWidth = maxSize > 0 ? (lvl.size / maxSize) * 100 : 0;
          return (
            <div
              key={`${lvl.priceCents}-${i}`}
              className="relative flex items-center justify-between rounded px-1.5 py-0.5 text-xs"
            >
              <div
                className={cn('absolute inset-0 rounded opacity-30', bgClass)}
                style={{ width: `${Math.min(barWidth, 100)}%` }}
              />
              <span className={cn('relative z-10 font-mono font-medium', colorClass)}>
                {Math.round(lvl.priceCents)}¢
              </span>
              <span className="text-muted-foreground relative z-10 font-mono">
                {lvl.size.toLocaleString()}
              </span>
            </div>
          );
        })
      )}
    </div>
  );
}

/** Quick trade button for a specific team + venue */
function QuickTradeButton({
  teamLabel,
  venue,
  priceCents,
  liq,
  onClick,
}: {
  teamLabel: string;
  venue: Venue;
  priceCents: number | null;
  liq: number | null;
  onClick: () => void;
}) {
  const liqDollars = liquidityDollars(liq, priceCents);
  const isKalshi = venue === 'kalshi';

  return (
    <button
      onClick={onClick}
      disabled={priceCents === null}
      className={cn(
        'flex w-full items-center justify-between rounded-lg border px-3 py-2 text-sm transition-colors',
        priceCents !== null
          ? 'border-border hover:bg-muted/50 cursor-pointer'
          : 'border-border/30 cursor-not-allowed opacity-50'
      )}
    >
      <div className="flex items-center gap-2">
        <span
          className={cn(
            'rounded px-1.5 py-0.5 text-[10px] font-bold uppercase',
            isKalshi ? 'bg-blue-500/20 text-blue-400' : 'bg-purple-500/20 text-purple-400'
          )}
        >
          {isKalshi ? 'K' : 'P'}
        </span>
        <span className="font-medium">{teamLabel}</span>
      </div>
      <div className="flex items-center gap-3">
        {priceCents !== null && (
          <span className="font-mono text-xs">{formatDualPriceCents(priceCents)}</span>
        )}
        {liqDollars !== null && (
          <span className="text-muted-foreground text-xs">
            <Money
              value={liqDollars}
              className="text-xs"
            />
          </span>
        )}
      </div>
    </button>
  );
}

export function InlineBookPanel({
  game,
  oddsMode,
  onTradeClick,
  onClose,
  className,
}: InlineBookPanelProps) {
  const isTakerMode = oddsMode === 'taker';
  const awayName = game.awayName ?? game.awayCode;
  const homeName = game.homeName ?? game.homeCode;

  // Compute team win data for all venues
  const awayK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'away',
    isTakerMode,
  });
  const homeK = getTeamWinData({
    game: game.kalshi ?? null,
    venue: 'kalshi',
    teamSide: 'home',
    isTakerMode,
  });
  const awayP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'away',
    isTakerMode,
  });
  const homeP = getTeamWinData({
    game: game.polymarket ?? null,
    venue: 'polymarket',
    teamSide: 'home',
    isTakerMode,
  });

  // Build orderbook levels for display
  const { kalshiAwayBids, kalshiAwayAsks, kalshiHomeBids, kalshiHomeAsks, maxBookSize } =
    useMemo(() => {
      const awayYesBids = game.kalshi?.markets.away.yes ?? [];
      const awayNoBids = game.kalshi?.markets.away.no ?? [];
      const homeYesBids = game.kalshi?.markets.home.yes ?? [];
      const homeNoBids = game.kalshi?.markets.home.no ?? [];

      // Derive asks by inverting the opposite side
      const awayAsks: BookLevel[] = awayNoBids.map((l) => ({
        priceCents: 100 - l.priceCents,
        size: l.size,
      }));
      const homeAsks: BookLevel[] = homeNoBids.map((l) => ({
        priceCents: 100 - l.priceCents,
        size: l.size,
      }));

      // Sort bids descending, asks ascending
      const sortedAwayBids = [...awayYesBids].sort((a, b) => b.priceCents - a.priceCents);
      const sortedAwayAsks = [...awayAsks].sort((a, b) => a.priceCents - b.priceCents);
      const sortedHomeBids = [...homeYesBids].sort((a, b) => b.priceCents - a.priceCents);
      const sortedHomeAsks = [...homeAsks].sort((a, b) => a.priceCents - b.priceCents);

      // Max size for bar width normalization
      const allLevels = [
        ...sortedAwayBids,
        ...sortedAwayAsks,
        ...sortedHomeBids,
        ...sortedHomeAsks,
      ];
      const maxSz = allLevels.reduce((max, l) => Math.max(max, l.size), 1);

      return {
        kalshiAwayBids: sortedAwayBids,
        kalshiAwayAsks: sortedAwayAsks,
        kalshiHomeBids: sortedHomeBids,
        kalshiHomeAsks: sortedHomeAsks,
        maxBookSize: maxSz,
      };
    }, [game.kalshi]);

  // Trade handlers
  const handleTrade = (teamSide: TeamSide, venue: Venue) => {
    if (!onTradeClick) return;

    if (venue === 'kalshi') {
      const top = teamSide === 'away' ? awayK : homeK;
      const teamLabel = teamSide === 'away' ? awayName : homeName;
      const marketTicker = top.marketTicker;
      const orderPrice = computeOrderPriceFromRawBid(top.rawBidPriceCents, isTakerMode);
      if (!marketTicker || orderPrice === null) return;
      onTradeClick({
        venue: 'kalshi',
        marketTicker,
        side: 'no',
        priceCents: orderPrice,
        contracts: 100,
        isTakerMode,
        teamLabel,
        gameKey: game.key,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
    } else {
      const top = teamSide === 'away' ? awayP : homeP;
      const teamLabel = teamSide === 'away' ? awayName : homeName;
      const marketTicker = top.marketTicker;
      const orderPrice = computeOrderPriceFromRawBid(top.rawBidPriceCents, isTakerMode);
      if (!marketTicker || orderPrice === null) return;

      const opponentMarket =
        teamSide === 'away' ? game.polymarket?.markets.home : game.polymarket?.markets.away;

      onTradeClick({
        venue: 'polymarket',
        marketTicker,
        tokenId: opponentMarket?.tokenId,
        conditionId: opponentMarket?.conditionId,
        tickSize: opponentMarket?.tickSize,
        negRisk: opponentMarket?.negRisk,
        side: 'no',
        priceCents: orderPrice,
        contracts: 100,
        isTakerMode,
        teamLabel,
        gameKey: game.key,
        gameDate: game.date,
        startTimePt: game.startTimePt,
      });
    }
  };

  return (
    <div
      className={cn(
        'bg-card border-border flex h-full flex-col overflow-hidden rounded-lg border',
        className
      )}
    >
      {/* Header */}
      <div className="border-border flex items-center justify-between border-b px-4 py-3">
        <div className="min-w-0 flex-1">
          <div className="flex items-center gap-2">
            <BookOpen className="text-primary h-4 w-4 flex-shrink-0" />
            <h3 className="truncate text-sm font-semibold">
              {awayName} @ {homeName}
            </h3>
          </div>
          <div className="text-muted-foreground mt-0.5 flex items-center gap-2 text-xs">
            <span>{game.date}</span>
            {game.startTimePt && (
              <>
                <span>·</span>
                <span>{formatStartTimePt(game.startTimePt)} PT</span>
              </>
            )}
          </div>
        </div>
        <button
          onClick={onClose}
          className="hover:bg-muted text-muted-foreground hover:text-foreground ml-2 flex-shrink-0 rounded-lg p-1.5 transition-colors"
          title="Close (Esc)"
        >
          <X className="h-4 w-4" />
        </button>
      </div>

      {/* Content - scrollable */}
      <div className="min-h-0 flex-1 space-y-4 overflow-y-auto p-4">
        {/* Price Summary */}
        <div className="grid grid-cols-2 gap-3">
          {/* Away team */}
          <div className="bg-muted/30 rounded-lg p-3">
            <div className="mb-2 text-xs font-semibold">{awayName}</div>
            <div className="space-y-1 text-xs">
              {awayK.priceCents !== null && (
                <div className="flex items-center justify-between">
                  <span className="text-blue-400">Kalshi</span>
                  <span className="font-mono">{formatDualPriceCents(awayK.priceCents)}</span>
                </div>
              )}
              {awayP.priceCents !== null && (
                <div className="flex items-center justify-between">
                  <span className="text-purple-400">Poly</span>
                  <span className="font-mono">{formatDualPriceCents(awayP.priceCents)}</span>
                </div>
              )}
              {awayK.priceCents === null && awayP.priceCents === null && (
                <div className="text-muted-foreground">No data</div>
              )}
            </div>
          </div>

          {/* Home team */}
          <div className="bg-muted/30 rounded-lg p-3">
            <div className="mb-2 text-xs font-semibold">{homeName}</div>
            <div className="space-y-1 text-xs">
              {homeK.priceCents !== null && (
                <div className="flex items-center justify-between">
                  <span className="text-blue-400">Kalshi</span>
                  <span className="font-mono">{formatDualPriceCents(homeK.priceCents)}</span>
                </div>
              )}
              {homeP.priceCents !== null && (
                <div className="flex items-center justify-between">
                  <span className="text-purple-400">Poly</span>
                  <span className="font-mono">{formatDualPriceCents(homeP.priceCents)}</span>
                </div>
              )}
              {homeK.priceCents === null && homeP.priceCents === null && (
                <div className="text-muted-foreground">No data</div>
              )}
            </div>
          </div>
        </div>

        {/* Kalshi Order Book (Away) */}
        {game.kalshi && (
          <div>
            <div className="text-muted-foreground mb-2 flex items-center gap-1.5 text-xs font-medium uppercase tracking-wider">
              <span className="rounded bg-blue-500/20 px-1 py-0.5 text-[10px] text-blue-400">
                Kalshi
              </span>
              {awayName} Book
            </div>
            <div className="grid grid-cols-2 gap-2">
              <MiniBook
                levels={kalshiAwayBids}
                side="bid"
                maxSize={maxBookSize}
                label="Bids (YES)"
              />
              <MiniBook
                levels={kalshiAwayAsks}
                side="ask"
                maxSize={maxBookSize}
                label="Asks (YES)"
              />
            </div>
          </div>
        )}

        {/* Kalshi Order Book (Home) */}
        {game.kalshi && (
          <div>
            <div className="text-muted-foreground mb-2 flex items-center gap-1.5 text-xs font-medium uppercase tracking-wider">
              <span className="rounded bg-blue-500/20 px-1 py-0.5 text-[10px] text-blue-400">
                Kalshi
              </span>
              {homeName} Book
            </div>
            <div className="grid grid-cols-2 gap-2">
              <MiniBook
                levels={kalshiHomeBids}
                side="bid"
                maxSize={maxBookSize}
                label="Bids (YES)"
              />
              <MiniBook
                levels={kalshiHomeAsks}
                side="ask"
                maxSize={maxBookSize}
                label="Asks (YES)"
              />
            </div>
          </div>
        )}

        {/* Quick Trade Buttons */}
        <div>
          <div className="text-muted-foreground mb-2 text-xs font-medium uppercase tracking-wider">
            Quick Trade
          </div>
          <div className="space-y-1.5">
            {/* Kalshi trades */}
            {awayK.priceCents !== null && (
              <QuickTradeButton
                teamLabel={awayName}
                venue="kalshi"
                priceCents={awayK.priceCents}
                liq={awayK.liq}
                onClick={() => handleTrade('away', 'kalshi')}
              />
            )}
            {homeK.priceCents !== null && (
              <QuickTradeButton
                teamLabel={homeName}
                venue="kalshi"
                priceCents={homeK.priceCents}
                liq={homeK.liq}
                onClick={() => handleTrade('home', 'kalshi')}
              />
            )}
            {/* Polymarket trades */}
            {awayP.priceCents !== null && (
              <QuickTradeButton
                teamLabel={awayName}
                venue="polymarket"
                priceCents={awayP.priceCents}
                liq={awayP.liq}
                onClick={() => handleTrade('away', 'polymarket')}
              />
            )}
            {homeP.priceCents !== null && (
              <QuickTradeButton
                teamLabel={homeName}
                venue="polymarket"
                priceCents={homeP.priceCents}
                liq={homeP.liq}
                onClick={() => handleTrade('home', 'polymarket')}
              />
            )}
            {/* No data state */}
            {awayK.priceCents === null &&
              homeK.priceCents === null &&
              awayP.priceCents === null &&
              homeP.priceCents === null && (
                <div className="text-muted-foreground py-4 text-center text-sm">
                  No trading data available for this game
                </div>
              )}
          </div>
        </div>
      </div>
    </div>
  );
}
