import { useMemo, useState } from 'react';
import { cn } from '@/lib/utils';
import type { SportsLoadingState } from '@/lib/sportsStream';

interface LoadingPhaseStatusProps {
  isLoading?: boolean;
  loadingState?: SportsLoadingState;
}

const PHASE_LABELS: Record<SportsLoadingState['phase'], string> = {
  idle: 'Idle',
  'discovering-markets': 'Discovering markets',
  'fetching-orderbooks': 'Fetching orderbooks',
  'fetching-events': 'Fetching event metadata',
  'building-rows': 'Building rows',
  'hydrating-polymarket': 'Hydrating Polymarket',
  'connecting-stream': 'Connecting live stream',
  live: 'Live',
  error: 'Error',
};

function formatTime(tsMs: number): string {
  return new Date(tsMs).toLocaleTimeString('en-US', {
    hour12: false,
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
  });
}

export function LoadingPhaseStatus({ isLoading = false, loadingState }: LoadingPhaseStatusProps) {
  const [showDetails, setShowDetails] = useState(false);

  const status = useMemo(() => {
    if (loadingState) {
      const label = PHASE_LABELS[loadingState.phase] ?? loadingState.phase;
      if (
        typeof loadingState.done === 'number' &&
        typeof loadingState.total === 'number' &&
        loadingState.total > 0
      ) {
        return `${label} ${loadingState.done}/${loadingState.total}`;
      }
      return label;
    }
    if (isLoading) return 'Loading';
    return 'Ready';
  }, [isLoading, loadingState]);

  const active = loadingState?.isLoading ?? isLoading;
  const events = loadingState?.events ?? [];

  return (
    <div className="relative w-[340px] shrink-0">
      <div className="text-muted-foreground flex min-h-6 items-center justify-end gap-2 text-xs">
        <div
          className={cn(
            'h-4 w-4 rounded-full border-2',
            active
              ? 'border-muted-foreground/30 border-t-primary animate-spin'
              : 'border-muted-foreground/30 opacity-0'
          )}
        />
        <span className="truncate">{status}</span>
        <button
          type="button"
          onClick={() => setShowDetails((prev) => !prev)}
          className="hover:text-foreground rounded px-1 py-0.5 transition-colors"
          aria-expanded={showDetails}
          aria-label="Toggle loading details"
        >
          Details
        </button>
      </div>
      {showDetails && (
        <div className="bg-popover border-border absolute right-0 top-7 z-30 w-[420px] rounded-md border p-2 shadow-lg">
          <div className="text-foreground border-border mb-2 border-b pb-1 text-xs font-semibold">
            Loading Timeline
          </div>
          {events.length === 0 ? (
            <div className="text-muted-foreground text-xs">
              No phase events recorded for this stream yet.
            </div>
          ) : (
            <div className="max-h-56 space-y-1 overflow-y-auto">
              {events.map((evt, idx) => (
                <div
                  key={`${evt.tsMs}-${evt.phase}-${idx}`}
                  className="text-muted-foreground grid grid-cols-[72px_1fr] gap-2 text-xs"
                >
                  <span className="font-mono">{formatTime(evt.tsMs)}</span>
                  <span className="truncate">
                    [{evt.level}] {PHASE_LABELS[evt.phase] ?? evt.phase}
                    {evt.message ? `: ${evt.message}` : ''}
                    {typeof evt.done === 'number' && typeof evt.total === 'number'
                      ? ` (${evt.done}/${evt.total})`
                      : ''}
                  </span>
                </div>
              ))}
            </div>
          )}
        </div>
      )}
    </div>
  );
}
