/**
 * OpenPositionsPanel - Displays open positions for the NBA Value Dashboard
 *
 * Compact view intended to sit between ConsolidatedMoneylines and OpenOrdersPanel.
 */

import { useMemo, useState } from 'react';
import { ChevronDown, ChevronUp } from 'lucide-react';
import type { NBAMarketRow, Position } from '@/types';
import { cn } from '@/lib/utils';
import { Money } from '@/components/atoms/Money';
import { Th, Td } from '@/components/atoms/TableCells';

interface OpenPositionsPanelProps {
  positions: Position[];
  markets?: NBAMarketRow[];
  titleByTicker?: Record<string, string>;
  gameDateByTicker?: Record<string, string>;
  outcomeByTicker?: Record<string, { yes?: string; no?: string }>;
  className?: string;
}

function includesInsensitive(haystack: string, needle: string) {
  return haystack.toLowerCase().includes(needle.toLowerCase());
}

function inferExposureSideLabel(args: {
  title?: string;
  awayTeam?: string;
  homeTeam?: string;
}): string {
  const { title, awayTeam, homeTeam } = args;
  if (!title) return '—';

  // For moneyline-style markets, we want a concise "team ML" label.
  // Avoid returning the full market title (which often contains both teams and looks like the "Game" column).
  const mentionsAway = !!awayTeam && includesInsensitive(title, awayTeam);
  const mentionsHome = !!homeTeam && includesInsensitive(title, homeTeam);

  if (awayTeam && mentionsAway && !mentionsHome) return `${awayTeam} ML`;
  if (homeTeam && mentionsHome && !mentionsAway) return `${homeTeam} ML`;
  if (awayTeam && mentionsAway) return `${awayTeam} ML`;
  if (homeTeam && mentionsHome) return `${homeTeam} ML`;

  return '—';
}

function formatPacificDateTime(ts: string): { date: string; time: string } | null {
  const d = new Date(ts);
  if (!Number.isFinite(d.getTime())) return null;

  // Force Pacific time display (PST/PDT depending on date).
  const timeZone = 'America/Los_Angeles';

  // Date as YYYY-MM-DD in Pacific time.
  const dateParts = new Intl.DateTimeFormat('en-US', {
    timeZone,
    year: 'numeric',
    month: '2-digit',
    day: '2-digit',
  }).formatToParts(d);
  const year = dateParts.find((p) => p.type === 'year')?.value;
  const month = dateParts.find((p) => p.type === 'month')?.value;
  const day = dateParts.find((p) => p.type === 'day')?.value;
  if (!year || !month || !day) return null;

  // Time as h:mm AM/PM with timezone abbreviation, no seconds.
  const timeParts = new Intl.DateTimeFormat('en-US', {
    timeZone,
    hour: 'numeric',
    minute: '2-digit',
    hour12: true,
    timeZoneName: 'short',
  }).formatToParts(d);
  const hour = timeParts.find((p) => p.type === 'hour')?.value;
  const minute = timeParts.find((p) => p.type === 'minute')?.value;
  const dayPeriod = timeParts.find((p) => p.type === 'dayPeriod')?.value;
  const tzName = timeParts.find((p) => p.type === 'timeZoneName')?.value ?? 'PT';

  if (!hour || !minute || !dayPeriod) return null;

  return {
    date: `${year}-${month}-${day}`,
    time: `${hour}:${minute} ${dayPeriod} ${tzName}`,
  };
}

export function OpenPositionsPanel({
  positions,
  markets,
  titleByTicker,
  gameDateByTicker,
  outcomeByTicker,
  className,
}: OpenPositionsPanelProps) {
  const [showAllRows, setShowAllRows] = useState(false);
  const maxRows = 10;

  const marketMetaByTicker = new Map<
    string,
    { game: string; gameDate?: string; awayTeam?: string; homeTeam?: string }
  >();
  (markets ?? []).forEach((m) => {
    const game = m.awayTeam && m.homeTeam ? `${m.awayTeam} @ ${m.homeTeam}` : m.market_ticker;
    marketMetaByTicker.set(m.market_ticker, {
      game,
      gameDate: (m as unknown as { gameDate?: string }).gameDate,
      awayTeam: m.awayTeam,
      homeTeam: m.homeTeam,
    });
  });

  const visiblePositions = useMemo(() => {
    if (showAllRows) return positions;
    return positions.slice(0, maxRows);
  }, [positions, showAllRows]);

  return (
    <div className={cn('bg-card border-border rounded-lg border', className)}>
      {/* Header - clean section title */}
      <div className="border-border flex items-center justify-between border-b px-4 py-4">
        <h2 className="text-foreground text-lg font-semibold">Open Positions</h2>
        <div className="text-muted-foreground text-xs">{positions.length} total</div>
      </div>

      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-muted/50 border-border border-b">
            <tr>
              <Th>Game Date</Th>
              <Th>Game</Th>
              <Th>Exposure Side</Th>
              <Th>Position Side</Th>
              <Th className="text-right">Qty</Th>
              <Th className="text-right">Avg</Th>
              <Th className="text-right">Current</Th>
              <Th className="text-right">P&L</Th>
            </tr>
          </thead>
          <tbody className="divide-border divide-y">
            {positions.length === 0 ? (
              <tr>
                <td
                  colSpan={8}
                  className="text-muted-foreground px-3 py-6 text-center"
                >
                  No open positions
                </td>
              </tr>
            ) : (
              visiblePositions.map((p, rowIndex) => {
                const meta = marketMetaByTicker.get(p.ticker);
                const title = titleByTicker?.[p.ticker];
                const fallbackExposureSide = inferExposureSideLabel({
                  title,
                  awayTeam: meta?.awayTeam,
                  homeTeam: meta?.homeTeam,
                });
                const exposureSide = outcomeByTicker?.[p.ticker]?.[p.side] ?? fallbackExposureSide;
                const gameDate = meta?.gameDate ?? gameDateByTicker?.[p.ticker] ?? '—';
                const dt = gameDate !== '—' ? formatPacificDateTime(gameDate) : null;
                const gameLabel = meta?.game ?? title ?? p.ticker;
                const zebraStyle =
                  rowIndex % 2 === 1
                    ? { backgroundColor: 'hsl(var(--muted) / var(--zebra-opacity))' }
                    : undefined;
                return (
                  <tr
                    key={`${p.ticker}-${p.side}`}
                    className="hover:bg-accent/50 transition-colors"
                    style={zebraStyle}
                  >
                    <Td className="font-mono">
                      {dt ? (
                        <>
                          <div className="text-foreground text-sm">{dt.date}</div>
                          <div className="text-muted-foreground text-xs">{dt.time}</div>
                        </>
                      ) : (
                        <div className="text-muted-foreground text-xs">{gameDate}</div>
                      )}
                    </Td>
                    <Td className="font-medium">
                      <div className="text-foreground text-sm">{gameLabel}</div>
                      <div className="text-muted-foreground font-mono text-xs">{p.ticker}</div>
                    </Td>
                    <Td className="font-medium">{exposureSide}</Td>
                    <Td>
                      <span
                        className={cn(
                          'rounded px-2 py-0.5 text-xs font-medium uppercase',
                          p.side === 'yes'
                            ? 'bg-positive/20 text-positive'
                            : 'bg-negative/20 text-negative'
                        )}
                      >
                        {p.side}
                      </span>
                    </Td>
                    <Td className="text-right font-mono">{p.quantity.toLocaleString()}</Td>
                    <Td className="text-right font-mono">
                      <Money value={p.averagePrice} />
                    </Td>
                    <Td className="text-right font-mono">
                      <Money value={p.currentPrice} />
                    </Td>
                    <Td className="text-right font-mono font-semibold">
                      <Money
                        value={p.totalPnl}
                        variant="pnl"
                      />
                    </Td>
                  </tr>
                );
              })
            )}
          </tbody>
        </table>
      </div>

      {/* Show more/less */}
      {positions.length > maxRows && (
        <div className="border-border/50 flex items-center justify-center border-t p-2">
          <button
            type="button"
            onClick={() => setShowAllRows((v) => !v)}
            className="text-muted-foreground hover:text-foreground hover:bg-muted/30 flex items-center gap-2 rounded px-3 py-1.5 text-xs transition-colors"
            aria-expanded={showAllRows}
            aria-label={showAllRows ? 'Show fewer positions' : 'Show more positions'}
          >
            <span className="tabular-nums">
              {showAllRows
                ? `Showing ${positions.length}`
                : `Showing ${Math.min(maxRows, positions.length)} of ${positions.length}`}
            </span>
            {showAllRows ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </button>
        </div>
      )}
    </div>
  );
}

export default OpenPositionsPanel;
