import { useEffect, useMemo, useState } from 'react';
import { Modal } from '@/components/atoms/Modal';
import { Dropdown } from '@/components/atoms/Dropdown';
import { NumericStepper } from '@/components/molecules/NumericStepper';
import { cn } from '@/lib/utils';
import { fmtProbFromCents } from '@/lib/nbaConsolidated/format';
import { formatStartTimePt } from '@/lib/formatters';
import {
  parseKalshiNbaMarketTicker,
  parseKalshiNbaEventTicker,
} from '@/lib/nbaConsolidated/kalshiNbaTicker';

export type ExpiryMode = 'gtc' | 'gst' | 'custom';

export interface PlaceOrderRequest {
  venue: 'kalshi' | 'polymarket';
  marketTicker: string;
  tokenId?: string;
  conditionId?: string;
  tickSize?: string;
  negRisk?: boolean;
  side: 'yes' | 'no';
  priceCents: number;
  isTakerMode: boolean;
  teamLabel: string;
  gameDate: string; // YYYY-MM-DD (best-effort; required for Kalshi expiry convenience)
  startTimePt?: string | null; // HH:MM
  defaultContracts?: number;
}

export interface PlaceOrderSubmission {
  contracts: number;
  expiryMode: ExpiryMode;
  /** UTC epoch seconds (optional) */
  expirationTs?: number;
}

function parseYmd(date: string): { y: number; m: number; d: number } | null {
  const m = String(date || '')
    .trim()
    .match(/^(\d{4})-(\d{2})-(\d{2})$/);
  if (!m) return null;
  const y = Number(m[1]);
  const mo = Number(m[2]);
  const d = Number(m[3]);
  if (!Number.isFinite(y) || !Number.isFinite(mo) || !Number.isFinite(d)) return null;
  return { y, m: mo, d };
}

function parseHhMm(hhmm: string): { hh: number; mm: number } | null {
  const m = String(hhmm || '')
    .trim()
    .match(/^(\d{1,2}):(\d{2})$/);
  if (!m) return null;
  const hh = Number(m[1]);
  const mm = Number(m[2]);
  if (!Number.isFinite(hh) || !Number.isFinite(mm)) return null;
  if (hh < 0 || hh > 23 || mm < 0 || mm > 59) return null;
  return { hh, mm };
}

/**
 * Convert a wall-clock time in America/Los_Angeles to UTC epoch seconds.
 * Best-effort without Temporal; iterates to handle DST offsets.
 */
function laWallClockToUtcSeconds(args: { dateYmd: string; timeHhMm: string }): number | null {
  const d = parseYmd(args.dateYmd);
  const t = parseHhMm(args.timeHhMm);
  if (!d || !t) return null;

  const desired = { year: d.y, month: d.m, day: d.d, hour: t.hh, minute: t.mm };

  // Start from a UTC guess with same components.
  let guessMs = Date.UTC(
    desired.year,
    desired.month - 1,
    desired.day,
    desired.hour,
    desired.minute,
    0
  );

  const fmt = new Intl.DateTimeFormat('en-US', {
    timeZone: 'America/Los_Angeles',
    year: 'numeric',
    month: '2-digit',
    day: '2-digit',
    hour: '2-digit',
    minute: '2-digit',
    hour12: false,
  });

  for (let i = 0; i < 3; i++) {
    const parts = fmt.formatToParts(new Date(guessMs));
    const get = (type: string) => parts.find((p) => p.type === type)?.value;
    const y = Number(get('year'));
    const mo = Number(get('month'));
    const da = Number(get('day'));
    const hh = Number(get('hour'));
    const mm = Number(get('minute'));
    if (![y, mo, da, hh, mm].every((n) => Number.isFinite(n))) return null;

    const deltaMinutes =
      (desired.year - y) * 525600 +
      (desired.month - mo) * 43200 +
      (desired.day - da) * 1440 +
      (desired.hour - hh) * 60 +
      (desired.minute - mm);

    if (deltaMinutes === 0) break;
    guessMs += deltaMinutes * 60_000;
  }

  return Math.floor(guessMs / 1000);
}

/**
 * Format YYYY-MM-DD to a readable date string like "Jan 28, 2026"
 */
function formatGameDate(dateStr: string): string {
  const parsed = parseYmd(dateStr);
  if (!parsed) return dateStr;

  const months = [
    'Jan',
    'Feb',
    'Mar',
    'Apr',
    'May',
    'Jun',
    'Jul',
    'Aug',
    'Sep',
    'Oct',
    'Nov',
    'Dec',
  ];
  return `${months[parsed.m - 1]} ${parsed.d}, ${parsed.y}`;
}

/**
 * Parse game context from the market ticker for display
 */
function OrderContextHeader({ request }: { request: PlaceOrderRequest }) {
  // Try to parse Kalshi ticker for rich display
  const marketInfo = parseKalshiNbaMarketTicker(request.marketTicker);
  const eventInfo = marketInfo ? parseKalshiNbaEventTicker(marketInfo.eventTicker) : null;

  // Format the matchup if we have event info
  const matchup = eventInfo ? `${eventInfo.awayCode} @ ${eventInfo.homeCode}` : null;
  const formattedDate = formatGameDate(request.gameDate);

  return (
    <div className="space-y-1">
      {/* Line 1: Team + all badges */}
      <div className="flex flex-wrap items-center gap-2">
        <span className="text-base font-semibold">{request.teamLabel}</span>
        <span className="rounded bg-green-500/20 px-1.5 py-0.5 text-xs font-bold uppercase text-green-400">
          Win
        </span>
        <span
          className={cn(
            'rounded px-1.5 py-0.5 text-xs font-medium uppercase',
            request.venue === 'kalshi'
              ? 'bg-blue-500/20 text-blue-400'
              : 'bg-purple-500/20 text-purple-400'
          )}
        >
          {request.venue}
        </span>
        <span
          className={cn(
            'rounded px-1.5 py-0.5 text-xs font-medium uppercase',
            request.isTakerMode
              ? 'bg-orange-500/20 text-orange-400'
              : 'bg-cyan-500/20 text-cyan-400'
          )}
        >
          {request.isTakerMode ? 'Taker' : 'Maker'}
        </span>
      </div>

      {/* Line 2: Matchup, date, and ticker */}
      <div className="text-muted-foreground font-mono text-xs">
        {matchup && <span>{matchup}</span>}
        {matchup && <span className="mx-1.5">·</span>}
        <span>{formattedDate}</span>
        <span className="mx-1.5">·</span>
        <span>{request.marketTicker}</span>
      </div>
    </div>
  );
}

export function PlaceOrderModal(props: {
  isOpen: boolean;
  request: PlaceOrderRequest | null;
  submitting?: boolean;
  error?: string | null;
  onClose: () => void;
  onSubmit: (submission: PlaceOrderSubmission) => void;
}) {
  const { isOpen, request, submitting = false, error, onClose, onSubmit } = props;

  const [contracts, setContracts] = useState<number>(request?.defaultContracts ?? 100);
  const [expiryMode, setExpiryMode] = useState<ExpiryMode>('custom');
  const [customExpiry, setCustomExpiry] = useState<string>(''); // YYYY-MM-DD HH:MM (local time)

  const priceDisplay = useMemo(() => {
    if (!request) return '—';
    return fmtProbFromCents(request.priceCents);
  }, [request]);

  const computedSubmission = useMemo<PlaceOrderSubmission | null>(() => {
    if (!request) return null;
    if (request.venue === 'polymarket') return { contracts, expiryMode: 'gtc' };
    if (expiryMode === 'gtc') return { contracts, expiryMode };
    if (expiryMode === 'gst') {
      if (!request.startTimePt) return { contracts, expiryMode }; // will error on submit
      const ts = laWallClockToUtcSeconds({
        dateYmd: request.gameDate,
        timeHhMm: request.startTimePt,
      });
      return ts ? { contracts, expiryMode, expirationTs: ts } : { contracts, expiryMode };
    }
    // custom
    if (!customExpiry.trim()) return { contracts, expiryMode };
    const m = customExpiry.trim().match(/^(\d{4}-\d{2}-\d{2})\s+(\d{1,2}:\d{2})$/);
    if (!m) return { contracts, expiryMode };
    const ts = laWallClockToUtcSeconds({ dateYmd: m[1], timeHhMm: m[2] });
    return ts ? { contracts, expiryMode, expirationTs: ts } : { contracts, expiryMode };
  }, [contracts, customExpiry, expiryMode, request]);

  // Reset when request changes (new cell clicked)
  useEffect(() => {
    setContracts(request?.defaultContracts ?? 100);
    // Default to game start time expiry for Kalshi if start time is available
    if (request?.venue === 'kalshi') {
      setExpiryMode(request?.startTimePt ? 'gst' : 'custom');
    } else {
      setExpiryMode('gtc');
    }
    setCustomExpiry('');
  }, [
    request?.marketTicker,
    request?.priceCents,
    request?.isTakerMode,
    request?.startTimePt,
    request?.venue,
  ]);

  if (!isOpen || !request) return null;

  const footer = (
    <div className="flex items-center justify-end gap-2">
      <button
        onClick={onClose}
        disabled={submitting}
        className="text-muted-foreground hover:text-foreground px-4 py-2 text-sm transition-colors disabled:opacity-50"
      >
        Cancel
      </button>
      <button
        onClick={() => {
          if (!computedSubmission) return;
          onSubmit(computedSubmission);
        }}
        disabled={submitting}
        className={cn(
          'rounded-lg px-4 py-2 text-sm font-medium transition-colors',
          'bg-primary text-primary-foreground hover:bg-primary/90',
          submitting && 'cursor-not-allowed opacity-50'
        )}
      >
        {submitting ? 'Submitting…' : 'Place Order'}
      </button>
    </div>
  );

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="PLACE ORDER"
      size="md"
      footer={footer}
    >
      <div className="space-y-4">
        {/* Order context header */}
        <OrderContextHeader request={request} />

        <div className="border-border bg-muted/30 rounded-lg border p-3 text-sm">
          <div className="flex items-center justify-between">
            <span className="text-muted-foreground">Side</span>
            <span className="font-mono font-medium uppercase">{request.side}</span>
          </div>
          <div className="mt-1 flex items-center justify-between">
            <span className="text-muted-foreground">Limit</span>
            <span className="font-mono font-medium">{priceDisplay}</span>
          </div>
        </div>

        <NumericStepper
          value={contracts}
          onChange={setContracts}
          min={1}
          step={100}
          label="Contracts"
        />

        {request.venue === 'kalshi' && (
          <div className="space-y-2">
            <Dropdown
              label="Expiry"
              value={expiryMode}
              onChange={(value) => setExpiryMode(value as ExpiryMode)}
              disabled={submitting}
              options={[
                {
                  value: 'gst',
                  label: request.startTimePt
                    ? `Game Start (${formatStartTimePt(request.startTimePt)} PT on ${formatGameDate(request.gameDate)})`
                    : 'Game Start Time',
                  disabled: !request.startTimePt,
                },
                { value: 'custom', label: 'Custom' },
                { value: 'gtc', label: 'Good Til Canceled' },
              ]}
            />

            {expiryMode === 'custom' && (
              <input
                value={customExpiry}
                onChange={(e) => setCustomExpiry(e.target.value)}
                placeholder="YYYY-MM-DD HH:MM"
                className="border-border bg-background h-10 w-full rounded-lg border px-3 font-mono text-sm"
                disabled={submitting}
              />
            )}
          </div>
        )}

        {request.venue === 'polymarket' && (
          <div className="border-border bg-muted/30 text-muted-foreground rounded-lg border p-3 text-xs">
            <div>
              Token: <span className="text-foreground/90 font-mono">{request.tokenId ?? '—'}</span>
            </div>
            <div className="mt-1">
              tickSize:{' '}
              <span className="text-foreground/90 font-mono">{request.tickSize ?? '—'}</span> ·
              negRisk:{' '}
              <span className="text-foreground/90 font-mono">
                {String(Boolean(request.negRisk))}
              </span>
            </div>
          </div>
        )}

        {error && <div className="text-sm text-red-400">{error}</div>}
      </div>
    </Modal>
  );
}
