/**
 * CandlestickChart - Interactive price chart using TradingView Lightweight Charts
 */

import { useEffect, useRef, useCallback } from 'react';
import {
  createChart,
  IChartApi,
  ISeriesApi,
  CandlestickData,
  Time,
  CrosshairMode,
  CandlestickSeries,
  HistogramSeries,
} from 'lightweight-charts';
import { cn } from '@/lib/utils';
import type { MockCandle } from '@/mock/candlesticks';

export interface CandlestickChartProps {
  data: MockCandle[];
  title?: string;
  height?: number;
  showVolume?: boolean;
  showToolbar?: boolean;
  onCrosshairMove?: (price: number | null, time: number | null) => void;
  className?: string;
}

export function CandlestickChart({
  data,
  title,
  height = 300,
  showVolume = true,
  showToolbar = true,
  onCrosshairMove,
  className,
}: CandlestickChartProps) {
  const containerRef = useRef<HTMLDivElement>(null);
  const chartRef = useRef<IChartApi | null>(null);
  const candleSeriesRef = useRef<ISeriesApi<'Candlestick'> | null>(null);
  const volumeSeriesRef = useRef<ISeriesApi<'Histogram'> | null>(null);

  // Initialize chart
  useEffect(() => {
    if (!containerRef.current) return;

    const chart = createChart(containerRef.current, {
      width: containerRef.current.clientWidth,
      height: showVolume ? height : height,
      layout: {
        background: { color: 'transparent' },
        textColor: '#9ca3af',
        fontSize: 11,
      },
      grid: {
        vertLines: { color: 'rgba(255, 255, 255, 0.04)' },
        horzLines: { color: 'rgba(255, 255, 255, 0.04)' },
      },
      crosshair: {
        mode: CrosshairMode.Normal,
        vertLine: {
          color: 'rgba(255, 255, 255, 0.2)',
          width: 1,
          style: 2,
          labelBackgroundColor: '#374151',
        },
        horzLine: {
          color: 'rgba(255, 255, 255, 0.2)',
          width: 1,
          style: 2,
          labelBackgroundColor: '#374151',
        },
      },
      rightPriceScale: {
        borderColor: 'rgba(255, 255, 255, 0.1)',
        scaleMargins: {
          top: 0.1,
          bottom: showVolume ? 0.25 : 0.1,
        },
      },
      timeScale: {
        borderColor: 'rgba(255, 255, 255, 0.1)',
        timeVisible: true,
        secondsVisible: false,
      },
    });

    chartRef.current = chart;

    // Add candlestick series
    const candleSeries = chart.addSeries(CandlestickSeries, {
      upColor: '#22c55e',
      downColor: '#ef4444',
      borderVisible: false,
      wickUpColor: '#22c55e',
      wickDownColor: '#ef4444',
    });
    candleSeriesRef.current = candleSeries;

    // Add volume series if enabled
    if (showVolume) {
      const volumeSeries = chart.addSeries(HistogramSeries, {
        color: '#3b82f6',
        priceFormat: {
          type: 'volume',
        },
        priceScaleId: '',
      });

      volumeSeries.priceScale().applyOptions({
        scaleMargins: {
          top: 0.8,
          bottom: 0,
        },
      });

      volumeSeriesRef.current = volumeSeries;
    }

    // Handle crosshair
    if (onCrosshairMove) {
      chart.subscribeCrosshairMove((param) => {
        if (!param.time || !param.point) {
          onCrosshairMove(null, null);
          return;
        }
        const price = param.seriesData.get(candleSeries);
        if (price && 'close' in price) {
          onCrosshairMove(price.close, param.time as number);
        }
      });
    }

    // Handle resize
    const handleResize = () => {
      if (containerRef.current && chartRef.current) {
        chartRef.current.applyOptions({
          width: containerRef.current.clientWidth,
        });
      }
    };

    window.addEventListener('resize', handleResize);

    return () => {
      window.removeEventListener('resize', handleResize);
      chart.remove();
      chartRef.current = null;
      candleSeriesRef.current = null;
      volumeSeriesRef.current = null;
    };
  }, [height, showVolume, onCrosshairMove]);

  // Update data
  useEffect(() => {
    if (!candleSeriesRef.current || data.length === 0) return;

    const candleData: CandlestickData[] = data.map((candle) => ({
      time: (candle.timestamp / 1000) as Time,
      open: candle.open,
      high: candle.high,
      low: candle.low,
      close: candle.close,
    }));

    candleSeriesRef.current.setData(candleData);

    if (volumeSeriesRef.current) {
      const volumeData = data.map((candle) => ({
        time: (candle.timestamp / 1000) as Time,
        value: candle.volume,
        color: candle.close >= candle.open ? 'rgba(34, 197, 94, 0.3)' : 'rgba(239, 68, 68, 0.3)',
      }));
      volumeSeriesRef.current.setData(volumeData);
    }

    // Fit content
    chartRef.current?.timeScale().fitContent();
  }, [data]);

  const handleFitContent = useCallback(() => {
    chartRef.current?.timeScale().fitContent();
  }, []);

  const handleResetZoom = useCallback(() => {
    chartRef.current?.timeScale().resetTimeScale();
    chartRef.current?.priceScale('right').applyOptions({ autoScale: true });
  }, []);

  return (
    <div className={cn('bg-card border-border/50 flex flex-col rounded-lg border', className)}>
      {/* Header */}
      {(title || showToolbar) && (
        <div className="border-border/50 flex items-center justify-between border-b px-3 py-2">
          {title && <span className="text-foreground text-sm font-medium">{title}</span>}
          {showToolbar && (
            <div className="flex items-center gap-1">
              <button
                onClick={handleFitContent}
                className="text-muted-foreground hover:text-foreground hover:bg-muted/50 rounded px-2 py-1 text-xs transition-colors"
              >
                Fit
              </button>
              <button
                onClick={handleResetZoom}
                className="text-muted-foreground hover:text-foreground hover:bg-muted/50 rounded px-2 py-1 text-xs transition-colors"
              >
                Reset
              </button>
            </div>
          )}
        </div>
      )}

      {/* Chart */}
      <div
        ref={containerRef}
        className="flex-1"
        style={{ minHeight: height }}
      />
    </div>
  );
}
