/**
 * ExchangeOrderBook - Kalshi/Polymarket-style vertical order book ladder
 *
 * Layout: Asks on top (worst→best), spread divider, Bids below (best→worst)
 * Columns: Price | Contracts/Shares | Total (cumulative $)
 * Features: Depth bars, sweep cost, click-to-trade
 */

import { useMemo, useState } from 'react';
import { cn } from '@/lib/utils';
import { Money } from '@/components/atoms/Money';
import type { BookLevel, Venue } from '@/lib/nbaConsolidated/types';

/** Computed level with cumulative depth for display */
interface ComputedLevel {
  priceCents: number;
  size: number;
  /** Dollar cost at this single level: price * size / 100 */
  levelDollars: number;
  /** Running cumulative $ from best price outward */
  cumulativeDollars: number;
  /** Running cumulative contracts from best price outward */
  cumulativeContracts: number;
}

export interface SweepInfo {
  /** Price level clicked */
  priceCents: number;
  /** Total contracts in the sweep */
  totalContracts: number;
  /** Total dollar cost of the sweep */
  totalDollars: number;
  side: 'bid' | 'ask';
}

interface ExchangeOrderBookProps {
  /** YES bids (descending by price) */
  bids: BookLevel[];
  /** YES asks derived from NO bids (ascending by price expected, will be sorted) */
  asks: BookLevel[];
  /** Last traded price in cents */
  lastPrice?: number | null;
  /** Spread in cents */
  spread?: number | null;
  /** Venue for column label ('Contracts' vs 'Shares') */
  venue: Venue;
  /** Header label (team name or market name) */
  teamLabel: string;
  /** Max price levels per side (default 8) */
  maxLevels?: number;
  /** Click handler with sweep info */
  onLevelClick?: (info: SweepInfo) => void;
  /** Highlighted sweep level (hovered) */
  className?: string;
}

function computeLevels(levels: BookLevel[], side: 'bid' | 'ask'): ComputedLevel[] {
  // Bids: descending (best = highest first). Accumulate from top.
  // Asks: ascending (best = lowest first). Accumulate from bottom displayed (which is best = lowest).
  // For asks display, we show worst (highest) at top, best (lowest) at bottom.
  // So we sort ascending, then reverse for display, but accumulate in ascending order.

  if (side === 'bid') {
    // Bids: sorted descending. Best bid at top. Accumulate from top.
    const sorted = [...levels].sort((a, b) => b.priceCents - a.priceCents);
    let cumDollars = 0;
    let cumContracts = 0;
    return sorted.map((lvl) => {
      const levelDollars = (lvl.priceCents * lvl.size) / 100;
      cumDollars += levelDollars;
      cumContracts += lvl.size;
      return {
        priceCents: lvl.priceCents,
        size: lvl.size,
        levelDollars,
        cumulativeDollars: cumDollars,
        cumulativeContracts: cumContracts,
      };
    });
  } else {
    // Asks: sort ascending first (best ask = lowest price first)
    const sorted = [...levels].sort((a, b) => a.priceCents - b.priceCents);
    let cumDollars = 0;
    let cumContracts = 0;
    const withCumulative = sorted.map((lvl) => {
      const levelDollars = (lvl.priceCents * lvl.size) / 100;
      cumDollars += levelDollars;
      cumContracts += lvl.size;
      return {
        priceCents: lvl.priceCents,
        size: lvl.size,
        levelDollars,
        cumulativeDollars: cumDollars,
        cumulativeContracts: cumContracts,
      };
    });
    // Reverse for display: worst ask (highest price) at top
    return withCumulative.reverse();
  }
}

function formatCompactDollars(dollars: number): string {
  if (dollars >= 1_000_000) return `$${(dollars / 1_000_000).toFixed(1)}M`;
  if (dollars >= 10_000) return `$${(dollars / 1_000).toFixed(0)}K`;
  if (dollars >= 1_000) return `$${(dollars / 1_000).toFixed(1)}K`;
  return `$${dollars.toFixed(2)}`;
}

function formatContracts(n: number): string {
  if (n >= 1_000_000) return `${(n / 1_000_000).toFixed(1)}M`;
  if (n >= 100_000) return `${(n / 1_000).toFixed(0)}K`;
  return n.toLocaleString();
}

export function ExchangeOrderBook({
  bids,
  asks,
  lastPrice,
  spread,
  venue,
  teamLabel,
  maxLevels = 8,
  onLevelClick,
  className,
}: ExchangeOrderBookProps) {
  const [hoveredLevel, setHoveredLevel] = useState<{
    price: number;
    side: 'bid' | 'ask';
  } | null>(null);

  const contractsLabel = venue === 'kalshi' ? 'Contracts' : 'Shares';
  const venueBadge = venue === 'kalshi' ? 'K' : 'P';
  const venueBadgeClass =
    venue === 'kalshi' ? 'bg-blue-500/20 text-blue-400' : 'bg-purple-500/20 text-purple-400';

  const computedAsks = useMemo(() => computeLevels(asks, 'ask'), [asks]);
  const computedBids = useMemo(() => computeLevels(bids, 'bid'), [bids]);

  const displayAsks = computedAsks.slice(0, maxLevels);
  const displayBids = computedBids.slice(0, maxLevels);

  // Max cumulative for depth bar scaling (use the larger of ask/bid max)
  const maxCumDollars = Math.max(
    displayAsks.length > 0 ? displayAsks[0].cumulativeDollars : 0, // worst ask has largest cumulative (first in display)
    displayBids.length > 0 ? (displayBids[displayBids.length - 1]?.cumulativeDollars ?? 0) : 0
  );

  const handleLevelClick = (level: ComputedLevel, side: 'bid' | 'ask') => {
    onLevelClick?.({
      priceCents: level.priceCents,
      totalContracts: level.cumulativeContracts,
      totalDollars: level.cumulativeDollars,
      side,
    });
  };

  // Determine if a level is within the hover sweep range
  const isInSweep = (level: ComputedLevel, side: 'bid' | 'ask'): boolean => {
    if (!hoveredLevel || hoveredLevel.side !== side) return false;
    if (side === 'ask') {
      // Sweep asks from best (lowest) to hovered. In display order (reversed),
      // hovered is somewhere above; all levels below it (including it) are in sweep.
      return level.priceCents <= hoveredLevel.price;
    } else {
      // Sweep bids from best (highest) to hovered. All levels above it (including it) are in sweep.
      return level.priceCents >= hoveredLevel.price;
    }
  };

  return (
    <div className={cn('bg-card border-border overflow-hidden rounded-lg border', className)}>
      {/* Header */}
      <div className="border-border flex items-center justify-between border-b px-3 py-2">
        <div className="flex items-center gap-2">
          <span
            className={cn('rounded px-1.5 py-0.5 text-[10px] font-bold uppercase', venueBadgeClass)}
          >
            {venueBadge}
          </span>
          <span className="text-sm font-medium">{teamLabel}</span>
        </div>
      </div>

      {/* Column headers */}
      <div className="text-muted-foreground border-border grid grid-cols-3 border-b px-3 py-1 text-[10px] uppercase tracking-wider">
        <span>Price</span>
        <span className="text-right">{contractsLabel}</span>
        <span className="text-right">Total</span>
      </div>

      {/* Asks (worst at top, best at bottom → toward spread) */}
      <div className="relative">
        {displayAsks.length === 0 ? (
          <div className="text-muted-foreground py-3 text-center text-xs">No asks</div>
        ) : (
          displayAsks.map((level) => {
            const depthPct =
              maxCumDollars > 0 ? (level.cumulativeDollars / maxCumDollars) * 100 : 0;
            const swept = isInSweep(level, 'ask');
            return (
              <div
                key={`ask-${level.priceCents}`}
                className={cn(
                  'group relative grid cursor-pointer grid-cols-3 px-3 py-1 font-mono text-xs transition-all',
                  swept ? 'bg-red-500/20' : 'hover:bg-red-500/10'
                )}
                onClick={() => handleLevelClick(level, 'ask')}
                onMouseEnter={() => setHoveredLevel({ price: level.priceCents, side: 'ask' })}
                onMouseLeave={() => setHoveredLevel(null)}
              >
                {/* Depth bar */}
                <div
                  className={cn(
                    'pointer-events-none absolute inset-y-0 right-0 transition-colors',
                    swept ? 'bg-red-500/25' : 'bg-red-500/10'
                  )}
                  style={{ width: `${Math.min(depthPct, 100)}%` }}
                />
                <span className="relative z-10 font-semibold text-red-400">
                  {Math.round(level.priceCents)}¢
                </span>
                <span className="text-muted-foreground relative z-10 text-right">
                  {formatContracts(level.size)}
                </span>
                <span className="text-muted-foreground relative z-10 text-right">
                  {formatCompactDollars(level.cumulativeDollars)}
                </span>
              </div>
            );
          })
        )}
      </div>

      {/* Spread divider */}
      <div className="border-border bg-muted/40 flex items-center justify-between border-y px-3 py-1.5">
        <div className="flex items-center gap-2">
          <span className="text-[10px] font-semibold uppercase text-red-400">Asks</span>
          {lastPrice != null && (
            <span className="font-mono text-xs font-medium">Last {Math.round(lastPrice)}¢</span>
          )}
          <span className="text-[10px] font-semibold uppercase text-green-400">Bids</span>
        </div>
        {spread != null && (
          <span className="text-muted-foreground text-[10px]">Spread: {Math.round(spread)}¢</span>
        )}
      </div>

      {/* Bids (best at top, worst at bottom → away from spread) */}
      <div className="relative">
        {displayBids.length === 0 ? (
          <div className="text-muted-foreground py-3 text-center text-xs">No bids</div>
        ) : (
          displayBids.map((level) => {
            const depthPct =
              maxCumDollars > 0 ? (level.cumulativeDollars / maxCumDollars) * 100 : 0;
            const swept = isInSweep(level, 'bid');
            return (
              <div
                key={`bid-${level.priceCents}`}
                className={cn(
                  'group relative grid cursor-pointer grid-cols-3 px-3 py-1 font-mono text-xs transition-all',
                  swept ? 'bg-green-500/20' : 'hover:bg-green-500/10'
                )}
                onClick={() => handleLevelClick(level, 'bid')}
                onMouseEnter={() => setHoveredLevel({ price: level.priceCents, side: 'bid' })}
                onMouseLeave={() => setHoveredLevel(null)}
              >
                {/* Depth bar */}
                <div
                  className={cn(
                    'pointer-events-none absolute inset-y-0 left-0 transition-colors',
                    swept ? 'bg-green-500/25' : 'bg-green-500/10'
                  )}
                  style={{ width: `${Math.min(depthPct, 100)}%` }}
                />
                <span className="relative z-10 font-semibold text-green-400">
                  {Math.round(level.priceCents)}¢
                </span>
                <span className="text-muted-foreground relative z-10 text-right">
                  {formatContracts(level.size)}
                </span>
                <span className="text-muted-foreground relative z-10 text-right">
                  {formatCompactDollars(level.cumulativeDollars)}
                </span>
              </div>
            );
          })
        )}
      </div>

      {/* Sweep info footer (shows when hovering a level) */}
      {hoveredLevel && (
        <div className="border-border bg-muted/50 border-t px-3 py-1.5">
          {(() => {
            const levels = hoveredLevel.side === 'ask' ? displayAsks : displayBids;
            const hovered = levels.find((l) => l.priceCents === hoveredLevel.price);
            if (!hovered) return null;
            const payout = hovered.cumulativeContracts; // $1 per contract if YES
            const cost = hovered.cumulativeDollars;
            const profit = payout - cost;
            return (
              <div className="flex items-center justify-between text-[10px]">
                <span className="text-muted-foreground">
                  Sweep {hovered.cumulativeContracts.toLocaleString()}{' '}
                  {contractsLabel.toLowerCase()}
                </span>
                <div className="flex items-center gap-3">
                  <span className="text-muted-foreground">
                    Cost:{' '}
                    <Money
                      value={cost}
                      className="text-[10px]"
                    />
                  </span>
                  {hoveredLevel.side === 'ask' && profit > 0 && (
                    <span className="text-green-400">
                      Payout:{' '}
                      <Money
                        value={payout}
                        className="text-[10px]"
                      />
                    </span>
                  )}
                </div>
              </div>
            );
          })()}
        </div>
      )}
    </div>
  );
}
