/**
 * OrderBook - Full order book with bids and asks
 *
 * Supports click-to-trade: clicking a bid sets up a sell, clicking an ask sets up a buy.
 */

import { cn } from '@/lib/utils';
import { OrderBookLevel } from '../molecules/OrderBookLevel';
import { useFlashOrderbookChanges } from '@/hooks/useFlash';

interface OrderBookProps {
  bids: Array<{ price: number; quantity: number; cumulative: number }>;
  asks: Array<{ price: number; quantity: number; cumulative: number }>;
  spread?: number;
  lastPrice?: number;
  className?: string;
  /** Called when a level is clicked. Bid click = sell opportunity, Ask click = buy opportunity */
  onLevelClick?: (
    price: number,
    quantity: number,
    cumulative: number,
    side: 'bid' | 'ask',
    shiftKey: boolean
  ) => void;
}

export function OrderBook({
  bids,
  asks,
  spread,
  lastPrice,
  className,
  onLevelClick,
}: OrderBookProps) {
  const maxBidCumulative = bids.length > 0 ? bids[bids.length - 1].cumulative : 1;
  const maxAskCumulative = asks.length > 0 ? asks[asks.length - 1].cumulative : 1;

  // Track which price levels have changed and should flash
  const flashingBidPrices = useFlashOrderbookChanges(bids);
  const flashingAskPrices = useFlashOrderbookChanges(asks);

  return (
    <div
      className={cn(
        'bg-card border-border/50 flex flex-col overflow-hidden rounded-lg border',
        className
      )}
    >
      {/* Header */}
      <div className="border-border/50 bg-muted/30 flex items-center justify-between border-b px-3 py-2">
        <h3 className="text-sm font-medium">Order Book</h3>
        {spread !== undefined && (
          <span className="text-muted-foreground text-xs">
            Spread: <span className="font-mono">{spread}¢</span>
          </span>
        )}
      </div>

      {/* Column headers */}
      <div className="text-muted-foreground border-border/30 grid grid-cols-2 border-b px-2 py-1 text-[10px] uppercase tracking-wider">
        <div className="flex justify-between">
          <span>Price</span>
          <span>Size</span>
          <span>Cost</span>
          <span>Depth $</span>
        </div>
        <div className="flex justify-between">
          <span>Depth $</span>
          <span>Cost</span>
          <span>Size</span>
          <span>Price</span>
        </div>
      </div>

      {/* Book content */}
      <div className="divide-border/30 grid min-h-0 flex-1 grid-cols-2 divide-x">
        {/* Bids (left) */}
        <div className="flex flex-col overflow-y-auto">
          {bids.map((level) => (
            <OrderBookLevel
              key={`bid-${level.price}`}
              price={level.price}
              quantity={level.quantity}
              cumulative={level.cumulative}
              maxCumulative={maxBidCumulative}
              side="bid"
              isFlashing={flashingBidPrices.has(level.price)}
              onClick={onLevelClick}
            />
          ))}
        </div>

        {/* Asks (right) */}
        <div className="flex flex-col overflow-y-auto">
          {asks.map((level) => (
            <OrderBookLevel
              key={`ask-${level.price}`}
              price={level.price}
              quantity={level.quantity}
              cumulative={level.cumulative}
              maxCumulative={maxAskCumulative}
              side="ask"
              isFlashing={flashingAskPrices.has(level.price)}
              onClick={onLevelClick}
            />
          ))}
        </div>
      </div>

      {/* Footer with last price */}
      {lastPrice !== undefined && (
        <div className="border-border/50 bg-muted/30 flex items-center justify-center border-t px-3 py-2">
          <span className="font-mono text-sm">
            Last: <span className="font-bold">{lastPrice}¢</span>
          </span>
        </div>
      )}
    </div>
  );
}

export default OrderBook;
