/**
 * DiagnosticsView — In-app health check page
 *
 * Runs diagnostic checks against the current environment and displays results.
 * Accessible via Bug icon in the sidebar.
 */

import { useState, useMemo } from 'react';
import {
  CheckCircle2,
  XCircle,
  AlertTriangle,
  Loader2,
  Clock,
  Play,
  ChevronDown,
  ChevronRight,
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { useDiagnostics } from '@/hooks/useDiagnostics';
import type { DiagnosticContext, DiagnosticResult, DiagnosticCategory } from '@/lib/diagnostics';
import { DIAGNOSTIC_CHECKS } from '@/lib/diagnostics';
import type { KalshiApiClient } from '@/types';
import type { ConsolidatedGameBooks } from '@/lib/nbaConsolidated/types';
import type { GameData } from '@/lib/sportsStream';

interface DiagnosticsViewProps {
  apiClientRef: React.RefObject<KalshiApiClient | null>;
  credentials: { accessKeyId: string } | null;
  consolidatedGames: ConsolidatedGameBooks[];
  sportsGames: Map<string, GameData[]>;
  feedStatus: 'live' | 'offline';
  lastUpdateTime: string;
}

const CATEGORY_LABELS: Record<DiagnosticCategory, string> = {
  connectivity: 'Connectivity',
  discovery: 'Discovery',
  stream: 'Stream Health',
  data: 'Data',
  date: 'Date Sanity',
};

const CATEGORY_ORDER: DiagnosticCategory[] = [
  'connectivity',
  'discovery',
  'stream',
  'data',
  'date',
];

function StatusIcon({ status }: { status: DiagnosticResult['status'] | 'pending' }) {
  switch (status) {
    case 'pass':
      return <CheckCircle2 className="h-4 w-4 text-green-500" />;
    case 'fail':
      return <XCircle className="h-4 w-4 text-red-500" />;
    case 'warn':
      return <AlertTriangle className="h-4 w-4 text-yellow-500" />;
    case 'running':
      return <Loader2 className="h-4 w-4 animate-spin text-blue-500" />;
    case 'skip':
      return <Clock className="h-4 w-4 text-gray-400" />;
    default:
      return <Clock className="h-4 w-4 text-gray-500" />;
  }
}

function statusLabel(status: DiagnosticResult['status']): string {
  switch (status) {
    case 'pass':
      return 'OK';
    case 'fail':
      return 'FAIL';
    case 'warn':
      return 'WARN';
    case 'skip':
      return 'SKIP';
    case 'running':
      return '...';
    default:
      return '—';
  }
}

export function DiagnosticsView({
  apiClientRef,
  credentials,
  consolidatedGames,
  sportsGames,
  feedStatus,
  lastUpdateTime,
}: DiagnosticsViewProps) {
  const ctx = useMemo<DiagnosticContext>(
    () => ({
      apiClient: apiClientRef.current,
      credentials,
      consolidatedGames,
      sportsGames,
      feedStatus,
      lastUpdateTime,
    }),
    [apiClientRef, credentials, consolidatedGames, sportsGames, feedStatus, lastUpdateTime]
  );

  const { results, isRunning, lastRunAt, runAll } = useDiagnostics(ctx);
  const [expandedIds, setExpandedIds] = useState<Set<string>>(new Set());

  const toggleExpand = (id: string) => {
    setExpandedIds((prev) => {
      const next = new Set(prev);
      if (next.has(id)) next.delete(id);
      else next.add(id);
      return next;
    });
  };

  const checksByCategory = useMemo(() => {
    const map = new Map<DiagnosticCategory, typeof DIAGNOSTIC_CHECKS>();
    for (const check of DIAGNOSTIC_CHECKS) {
      const arr = map.get(check.category) ?? [];
      arr.push(check);
      map.set(check.category, arr);
    }
    return map;
  }, []);

  // Count summary
  const summary = useMemo(() => {
    let pass = 0;
    let fail = 0;
    let warn = 0;
    let skip = 0;
    for (const r of results.values()) {
      if (r.status === 'pass') pass++;
      else if (r.status === 'fail') fail++;
      else if (r.status === 'warn') warn++;
      else if (r.status === 'skip') skip++;
    }
    return { pass, fail, warn, skip, total: results.size };
  }, [results]);

  return (
    <div className="mx-auto max-w-3xl space-y-6 p-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-foreground text-lg font-semibold">Diagnostics</h2>
          <p className="text-muted-foreground text-sm">
            Run health checks against the current environment
          </p>
        </div>
        <div className="flex items-center gap-4">
          {lastRunAt && (
            <span className="text-muted-foreground text-xs">
              Last run: {new Date(lastRunAt).toLocaleTimeString()}
            </span>
          )}
          <button
            onClick={runAll}
            disabled={isRunning}
            className={cn(
              'flex items-center gap-2 rounded-md px-4 py-2 text-sm font-medium transition-colors',
              isRunning
                ? 'bg-muted text-muted-foreground cursor-not-allowed'
                : 'bg-primary text-primary-foreground hover:bg-primary/90'
            )}
          >
            {isRunning ? (
              <Loader2 className="h-4 w-4 animate-spin" />
            ) : (
              <Play className="h-4 w-4" />
            )}
            {isRunning ? 'Running...' : 'Run All'}
          </button>
        </div>
      </div>

      {/* Summary bar */}
      {summary.total > 0 && (
        <div className="bg-muted/50 flex items-center gap-4 rounded-lg px-4 py-2 text-sm">
          <span className="font-medium text-green-500">{summary.pass} pass</span>
          {summary.fail > 0 && (
            <span className="font-medium text-red-500">{summary.fail} fail</span>
          )}
          {summary.warn > 0 && (
            <span className="font-medium text-yellow-500">{summary.warn} warn</span>
          )}
          {summary.skip > 0 && <span className="text-muted-foreground">{summary.skip} skip</span>}
        </div>
      )}

      {/* Categories */}
      {CATEGORY_ORDER.map((category) => {
        const checks = checksByCategory.get(category);
        if (!checks || checks.length === 0) return null;

        return (
          <div key={category}>
            <h3 className="text-muted-foreground mb-2 text-xs font-semibold uppercase tracking-wider">
              {CATEGORY_LABELS[category]}
            </h3>
            <div className="border-border divide-border divide-y rounded-lg border">
              {checks.map((check) => {
                const r = results.get(check.id);
                const isExpanded = expandedIds.has(check.id);
                const hasDetails = r?.details && Object.keys(r.details).length > 0;

                return (
                  <div key={check.id}>
                    <button
                      onClick={() => hasDetails && toggleExpand(check.id)}
                      className={cn(
                        'flex w-full items-center gap-3 px-4 py-3 text-left transition-colors',
                        hasDetails && 'hover:bg-muted/50 cursor-pointer',
                        !hasDetails && 'cursor-default'
                      )}
                    >
                      <StatusIcon status={r?.status ?? 'pending'} />
                      <span className="text-foreground min-w-0 flex-1 truncate text-sm font-medium">
                        {check.name}
                      </span>
                      {r && (
                        <>
                          <span
                            className={cn(
                              'min-w-0 flex-shrink text-sm',
                              r.status === 'pass' && 'text-muted-foreground',
                              r.status === 'fail' && 'text-red-400',
                              r.status === 'warn' && 'text-yellow-400',
                              r.status === 'skip' && 'text-muted-foreground'
                            )}
                          >
                            {r.message}
                          </span>
                          <span className="text-muted-foreground flex-shrink-0 text-xs tabular-nums">
                            {r.durationMs}ms
                          </span>
                          <span
                            className={cn(
                              'w-10 flex-shrink-0 text-right text-xs font-bold',
                              r.status === 'pass' && 'text-green-500',
                              r.status === 'fail' && 'text-red-500',
                              r.status === 'warn' && 'text-yellow-500',
                              r.status === 'skip' && 'text-gray-400'
                            )}
                          >
                            {statusLabel(r.status)}
                          </span>
                        </>
                      )}
                      {hasDetails && (
                        <span className="text-muted-foreground flex-shrink-0">
                          {isExpanded ? (
                            <ChevronDown className="h-3 w-3" />
                          ) : (
                            <ChevronRight className="h-3 w-3" />
                          )}
                        </span>
                      )}
                    </button>

                    {/* Expanded details */}
                    {isExpanded && r?.details && (
                      <div className="bg-muted/30 border-border border-t px-4 py-3">
                        <pre className="text-muted-foreground overflow-x-auto text-xs">
                          {JSON.stringify(r.details, null, 2)}
                        </pre>
                      </div>
                    )}
                  </div>
                );
              })}
            </div>
          </div>
        );
      })}
    </div>
  );
}

export default DiagnosticsView;
