/**
 * ValueDashboardView - Multi-sport value dashboard
 *
 * Features:
 * - Sport selector at top
 * - Tabs for moneyline/spread/total (based on selected sport)
 * - Existing NBA moneyline view for backwards compatibility
 * - Grid view for other sports/market types
 */

import { useState, useCallback, useMemo, useEffect, useRef } from 'react';
import { ChevronLeft, ChevronRight } from 'lucide-react';
import { useHotkeys } from '@/lib/hotkeys';
import { cn } from '@/lib/utils';
import { getTodayStr, findNearestAvailableDate } from '@/lib/dateUtils';
import { type Sport, type MarketType, SPORT_CONFIGS, getSportConfig } from '@/lib/sportsDiscovery';
import type { GameData, SportsLoadingState } from '@/lib/sportsStream';
import type { ConsolidatedGameBooks } from '@/lib/nbaConsolidated/types';
import { useRotationNumbers } from '@/hooks/useRotationNumbers';
import type { NBAMarketRow, Position } from '@/types';
import type { KalshiOrder } from '@/lib/kalshiApi';
import type { PolyOpenOrder } from '@/lib/polymarket/types';

// Existing NBA components
import {
  DashboardHeader,
  type OddsMode,
  type ViewMode,
} from '@/components/nba-value-dashboard/DashboardHeader';
import {
  ConsolidatedMoneylines,
  type TradeClickRequest,
} from '@/components/nba-value-dashboard/ConsolidatedMoneylines';
import { TwoRowMoneylines } from '@/components/nba-value-dashboard/TwoRowMoneylines';
import { OpenOrdersPanel } from '@/components/nba-value-dashboard/OpenOrdersPanel';
import { OpenPositionsPanel } from '@/components/nba-value-dashboard/OpenPositionsPanel';
import { CombinedOrdersPositionsPanel } from '@/components/nba-value-dashboard/CombinedOrdersPositionsPanel';
import {
  PlaceOrderModal,
  type PlaceOrderRequest,
  type PlaceOrderSubmission,
} from '@/components/nba-value-dashboard/PlaceOrderModal';
import { GameDetailPanel } from '@/components/nba-value-dashboard/GameDetailPanel';
import { getTeamWinData, liquidityDollars } from '@/lib/nbaConsolidated/pricing';
import {
  type VolumeWindow,
  initVolumeSettings,
  toggleWindow,
} from '@/lib/nbaConsolidated/volumeSettings';
import {
  getViewMode,
  setViewMode as persistViewMode,
} from '@/lib/nbaConsolidated/viewModeSettings';

// New sports components
import { TwoRowTotals, TwoRowSpreads } from '@/components/sports';
import {
  type TotalGameData,
  type SpreadGameData,
  type MoneylineGameData,
  moneylineToConsolidated,
} from '@/lib/sportsStream';

/** Sport tab options */
const SPORT_TABS = SPORT_CONFIGS.map((config) => ({
  value: config.sport,
  label: config.shortLabel,
}));

/** Stable empty defaults to avoid creating new Map() on every render */
const EMPTY_MAP = new Map() as Map<string, never>;
const EMPTY_BOOL_MAP = new Map() as Map<string, boolean>;
const EMPTY_STATE_MAP = new Map() as Map<string, SportsLoadingState>;

/** Market type labels */
const MARKET_TYPE_LABELS: Record<MarketType, string> = {
  moneyline: 'Moneyline',
  spread: 'Spread',
  total: 'Total',
};

interface ValueDashboardViewProps {
  /** NBA markets data from stream */
  markets?: NBAMarketRow[];
  /** Open orders */
  orders?: KalshiOrder[];
  /** Consolidated NBA games (for NBA moneyline view) */
  consolidatedGames?: ConsolidatedGameBooks[];
  consolidatedLoading?: boolean;
  feedStatus?: 'live' | 'offline';
  lastUpdateTime?: string;
  polyOpenOrders?: PolyOpenOrder[];
  polyStatus?: 'disconnected' | 'connected' | 'error';
  /** Open positions */
  positions?: Position[];
  /** Optional mapping from ticker -> market title */
  positionTitleByTicker?: Record<string, string>;
  /** Optional mapping from ticker -> market close/expiration time */
  positionGameDateByTicker?: Record<string, string>;
  /** Optional mapping from ticker -> { yes/no subtitle } */
  positionOutcomeByTicker?: Record<string, { yes?: string; no?: string }>;
  /** Handler for canceling an order */
  onCancelOrder?: (orderId: string) => void;
  /** Handler for decreasing order quantity (queue-safe) */
  onDecreaseOrder?: (orderId: string, nextQuantity: number) => Promise<void> | void;
  /** Handler for killing all orders */
  onKillAll?: () => void;
  /** Handler for canceling a Polymarket order */
  onCancelPolyOrder?: (orderId: string) => void;
  /** Handler for killing all Polymarket orders */
  onKillAllPolyOrders?: () => void;
  /** Place a Kalshi order */
  onPlaceKalshiOrder?: (args: {
    marketTicker: string;
    side: 'yes' | 'no';
    priceCents: number;
    contracts: number;
    isTakerMode: boolean;
    expirationTs?: number;
  }) => Promise<void>;
  onPlacePolymarketOrder?: (args: {
    tokenId: string;
    conditionId?: string;
    action: 'buy' | 'sell';
    priceCents: number;
    contracts: number;
    isTakerMode: boolean;
    tickSize?: string;
    negRisk?: boolean;
  }) => Promise<void>;
  /** Multi-sport games data (keyed by "{sport}-{marketType}") */
  sportsGames?: Map<string, GameData[]>;
  sportsLoading?: Map<string, boolean>;
  sportsLoadingStates?: Map<string, SportsLoadingState>;
  /** Callback when sport/market type changes */
  onSportMarketChange?: (sport: Sport, marketType: MarketType) => void;
}

export function ValueDashboardView({
  markets,
  orders: _orders,
  consolidatedGames = [],
  consolidatedLoading = false,
  feedStatus: _feedStatus = 'offline',
  lastUpdateTime: _lastUpdateTime = '--:--:--',
  polyOpenOrders = [],
  polyStatus = 'disconnected',
  positions,
  positionTitleByTicker,
  positionGameDateByTicker,
  positionOutcomeByTicker,
  onCancelOrder,
  onDecreaseOrder,
  onKillAll,
  onCancelPolyOrder,
  onKillAllPolyOrders,
  onPlaceKalshiOrder,
  onPlacePolymarketOrder,
  sportsGames = EMPTY_MAP,
  sportsLoading = EMPTY_BOOL_MAP,
  sportsLoadingStates = EMPTY_STATE_MAP,
  onSportMarketChange,
}: ValueDashboardViewProps) {
  // Sport and market type selection
  const [selectedSport, setSelectedSport] = useState<Sport>('nba');
  const [selectedMarketType, setSelectedMarketType] = useState<MarketType>('moneyline');

  // Date filter — null means "all dates"
  const [selectedDate, setSelectedDate] = useState<string | null>(() => getTodayStr());

  // Existing NBA dashboard state
  const [oddsMode, setOddsMode] = useState<OddsMode>('maker');
  const [viewMode, setViewMode] = useState<ViewMode>(() => getViewMode());
  const [expandedGameKeys, setExpandedGameKeys] = useState<Set<string>>(new Set());
  const [tradeReq, setTradeReq] = useState<TradeClickRequest | null>(null);
  const [tradeSubmitting, setTradeSubmitting] = useState(false);
  const [tradeError, setTradeError] = useState<string | null>(null);
  const [volumeWindows, setVolumeWindows] = useState<VolumeWindow[]>([]);

  // Initialize volume settings from localStorage
  useEffect(() => {
    setVolumeWindows(initVolumeSettings());
  }, []);

  // Get available market types for selected sport
  const availableMarketTypes = useMemo((): MarketType[] => {
    const config = getSportConfig(selectedSport);
    return config?.marketTypes ?? (['moneyline'] as MarketType[]);
  }, [selectedSport]);

  // Ensure selected market type is valid for selected sport
  useEffect(() => {
    if (!availableMarketTypes.includes(selectedMarketType)) {
      const firstType = availableMarketTypes[0];
      if (firstType) {
        setSelectedMarketType(firstType);
      }
    }
  }, [availableMarketTypes, selectedMarketType]);

  // Notify parent when sport/market type changes
  useEffect(() => {
    onSportMarketChange?.(selectedSport, selectedMarketType);
  }, [selectedSport, selectedMarketType, onSportMarketChange]);

  // Keyboard shortcuts for quick navigation (via hotkey system)
  useHotkeys({
    'sport-nba': () => setSelectedSport('nba'),
    'sport-nfl': () => setSelectedSport('nfl'),
    'sport-nhl': () => setSelectedSport('nhl'),
    'sport-cbb': () => setSelectedSport('cbb'),
    'sport-atp': () => setSelectedSport('tennis-atp'),
    'sport-wta': () => setSelectedSport('tennis-wta'),
    'market-moneyline': () => {
      if (availableMarketTypes.includes('moneyline')) {
        setSelectedMarketType('moneyline');
      }
    },
    'market-spread': () => {
      if (availableMarketTypes.includes('spread')) {
        setSelectedMarketType('spread');
      }
    },
    'market-total': () => {
      if (availableMarketTypes.includes('total')) {
        setSelectedMarketType('total');
      }
    },
  });

  // Use moneyline view for all sports now (unified component)
  const useMoneylineView = selectedMarketType === 'moneyline';

  // Get games for current selection
  const currentGamesKey = `${selectedSport}-${selectedMarketType}`;
  const currentGames = sportsGames.get(currentGamesKey) ?? [];
  const currentLoading = sportsLoading.get(currentGamesKey) ?? false;
  const currentLoadingState = sportsLoadingStates.get(currentGamesKey);

  // For non-NBA moneylines, convert to consolidated format
  const convertedMoneylineGames = useMemo(() => {
    if (selectedSport === 'nba' || selectedMarketType !== 'moneyline') {
      return []; // NBA uses consolidatedGames directly
    }
    return (currentGames as MoneylineGameData[]).map(moneylineToConsolidated);
  }, [currentGames, selectedSport, selectedMarketType]);

  // Determine which games to show for moneyline view
  const moneylineGamesRaw = selectedSport === 'nba' ? consolidatedGames : convertedMoneylineGames;
  const moneylineLoading = selectedSport === 'nba' ? consolidatedLoading : currentLoading;

  // Enrich moneyline games with rotation numbers from The Odds API (free endpoint)
  const moneylineGamesWithRotation = useRotationNumbers(selectedSport, moneylineGamesRaw);

  // Collect unique sorted dates from ALL games (moneyline + current sport games)
  const availableDates = useMemo(() => {
    const dateSet = new Set<string>();
    for (const g of moneylineGamesWithRotation) if (g.date) dateSet.add(g.date);
    for (const g of currentGames) if (g.date) dateSet.add(g.date);
    return Array.from(dateSet).sort();
  }, [moneylineGamesWithRotation, currentGames]);

  // Auto-snap: if selected date has no games, jump to nearest available date.
  // Guard: don't snap while data is still loading — partial availableDates can mis-snap.
  useEffect(() => {
    if (moneylineLoading || currentLoading) return;
    if (!selectedDate || availableDates.length === 0) return;
    const snapped = findNearestAvailableDate(selectedDate, availableDates);
    if (snapped !== null) setSelectedDate(snapped);
  }, [selectedDate, availableDates, moneylineLoading, currentLoading]);

  // Filter games by selected date
  const moneylineGamesForView = useMemo(() => {
    if (!selectedDate) return moneylineGamesWithRotation;
    return moneylineGamesWithRotation.filter((g) => g.date === selectedDate);
  }, [moneylineGamesWithRotation, selectedDate]);

  const filteredCurrentGames = useMemo(() => {
    if (!selectedDate) return currentGames;
    return currentGames.filter((g) => g.date === selectedDate);
  }, [currentGames, selectedDate]);

  // Date navigation
  const handleDatePrev = useCallback(() => {
    if (!selectedDate || availableDates.length === 0) return;
    const idx = availableDates.indexOf(selectedDate);
    if (idx > 0) setSelectedDate(availableDates[idx - 1]!);
  }, [selectedDate, availableDates]);

  const handleDateNext = useCallback(() => {
    if (!selectedDate || availableDates.length === 0) return;
    const idx = availableDates.indexOf(selectedDate);
    if (idx < availableDates.length - 1) setSelectedDate(availableDates[idx + 1]!);
  }, [selectedDate, availableDates]);

  const handleDateToday = useCallback(() => {
    setSelectedDate(getTodayStr());
  }, []);

  const handleDateAll = useCallback(() => {
    setSelectedDate(null);
  }, []);

  // Handlers
  const handleSportChange = useCallback((value: Sport) => {
    setSelectedSport(value);
  }, []);

  const handleMarketTypeChange = useCallback((type: MarketType) => {
    setSelectedMarketType(type);
  }, []);

  const handleToggleVolumeWindow = useCallback((window: VolumeWindow) => {
    setVolumeWindows(toggleWindow(window));
  }, []);

  const handleModeChange = useCallback((mode: OddsMode) => {
    setOddsMode(mode);
  }, []);

  const handleViewModeChange = useCallback((mode: ViewMode) => {
    setViewMode(mode);
    persistViewMode(mode);
  }, []);

  const handleCancelOrder = useCallback(
    (orderId: string) => {
      onCancelOrder?.(orderId);
    },
    [onCancelOrder]
  );

  const handleKillAll = useCallback(() => {
    onKillAll?.();
  }, [onKillAll]);

  const handleCancelPolyOrder = useCallback(
    (orderId: string) => {
      onCancelPolyOrder?.(orderId);
    },
    [onCancelPolyOrder]
  );

  const handleKillAllPolyOrders = useCallback(() => {
    onKillAllPolyOrders?.();
  }, [onKillAllPolyOrders]);

  const handleDecreaseOrder = useCallback(
    async (orderId: string, nextQuantity: number) => {
      await onDecreaseOrder?.(orderId, nextQuantity);
    },
    [onDecreaseOrder]
  );

  const handleToggleExpand = useCallback((gameKey: string) => {
    setExpandedGameKeys((prev) => {
      if (prev.has(gameKey)) {
        // Deselect: clear all
        return new Set();
      }
      // Single-select: replace with just this game
      return new Set([gameKey]);
    });
  }, []);

  // Map market tickers → game keys so odds cell clicks toggle expansion
  const tickerToGameKey = useMemo(() => {
    const map = new Map<string, string>();
    for (const game of moneylineGamesForView) {
      if (game.kalshi) {
        map.set(game.kalshi.markets.away.marketTicker, game.key);
        map.set(game.kalshi.markets.home.marketTicker, game.key);
      }
      if (game.polymarket) {
        const awayToken = game.polymarket.markets.away.tokenId;
        const homeToken = game.polymarket.markets.home.tokenId;
        if (awayToken) map.set(`asset:${awayToken}`, game.key);
        if (homeToken) map.set(`asset:${homeToken}`, game.key);
      }
    }
    return map;
  }, [moneylineGamesForView]);

  // Primary click on odds cells → toggle row expansion (not navigate away)
  const handleNavigateToMarket = useCallback(
    (marketTicker: string) => {
      const gameKey = tickerToGameKey.get(marketTicker);
      if (gameKey) {
        handleToggleExpand(gameKey);
      }
    },
    [tickerToGameKey, handleToggleExpand]
  );

  // Max liquidity calculation for moneyline view (all sports)
  const maxLiquidity = useMemo(() => {
    if (!useMoneylineView) return 1000;

    const isTakerMode = oddsMode === 'taker';
    let max = 1000;
    for (const g of moneylineGamesForView) {
      const awayK = getTeamWinData({
        game: g.kalshi ?? null,
        venue: 'kalshi',
        teamSide: 'away',
        isTakerMode,
      });
      const homeK = getTeamWinData({
        game: g.kalshi ?? null,
        venue: 'kalshi',
        teamSide: 'home',
        isTakerMode,
      });
      const awayP = getTeamWinData({
        game: g.polymarket ?? null,
        venue: 'polymarket',
        teamSide: 'away',
        isTakerMode,
      });
      const homeP = getTeamWinData({
        game: g.polymarket ?? null,
        venue: 'polymarket',
        teamSide: 'home',
        isTakerMode,
      });
      const a = liquidityDollars(awayK.liq, awayK.priceCents) ?? 0;
      const h = liquidityDollars(homeK.liq, homeK.priceCents) ?? 0;
      const ap = liquidityDollars(awayP.liq, awayP.priceCents) ?? 0;
      const hp = liquidityDollars(homeP.liq, homeP.priceCents) ?? 0;
      max = Math.max(max, a, h, ap, hp);
    }
    return max;
  }, [moneylineGamesForView, oddsMode, useMoneylineView]);

  // Orderbook mapping for positions panel
  const orderbookByTicker = useMemo(() => {
    const result: Record<
      string,
      {
        yesBestBid: number | null;
        yesBestAsk: number | null;
        noBestBid: number | null;
        noBestAsk: number | null;
      }
    > = {};

    for (const game of consolidatedGames) {
      if (game.kalshi) {
        const markets = [game.kalshi.markets.away, game.kalshi.markets.home];
        for (const market of markets) {
          const yesBestBid = market.yes[0]?.priceCents ?? null;
          const noBestBid = market.no[0]?.priceCents ?? null;
          const yesBestAsk = noBestBid !== null ? 100 - noBestBid : null;
          const noBestAsk = yesBestBid !== null ? 100 - yesBestBid : null;
          result[market.marketTicker] = { yesBestBid, yesBestAsk, noBestBid, noBestAsk };
        }
      }
    }

    return result;
  }, [consolidatedGames]);

  // Stable structural fingerprint for sportsGames — only changes when game
  // structure (keys, team codes, token IDs) changes, NOT on price updates.
  // This prevents polyTokenLabels from recomputing on every WS tick.
  const sportsGamesRef = useRef(sportsGames);
  sportsGamesRef.current = sportsGames;

  // Stable structural fingerprint for sportsGames — only changes when game
  // structure (event tickers, team codes) changes, NOT on price updates.
  // This prevents polyTokenLabels from recomputing on every WS tick.
  const sportsGamesStructureKey = useMemo(() => {
    const parts: string[] = [];
    for (const [key, games] of sportsGames) {
      parts.push(key);
      for (const g of games) {
        parts.push(g.eventTicker);
      }
    }
    return parts.join('|');
  }, [sportsGames]);

  // Build Polymarket tokenId -> human-readable label mapping.
  // Uses sportsGamesRef so the dependency is on sportsGamesStructureKey (stable)
  // rather than the sportsGames Map object (changes identity on every price update).
  const polyTokenLabels = useMemo(() => {
    const labels: Record<string, string> = {};
    const sg = sportsGamesRef.current;

    // Moneylines from consolidatedGames
    for (const game of consolidatedGames) {
      if (!game.polymarket) continue;
      const away = game.awayCode;
      const home = game.homeCode;
      const awayToken = game.polymarket.markets.away.tokenId;
      const homeToken = game.polymarket.markets.home.tokenId;
      if (awayToken) labels[awayToken] = `${away} ML`;
      if (homeToken) labels[homeToken] = `${home} ML`;
    }

    // Moneylines, spreads, totals from sportsGames
    for (const [, games] of sg) {
      for (const game of games) {
        const away = game.awayCode;
        const home = game.homeCode;

        if (game.marketType === 'moneyline') {
          const ml = game as MoneylineGameData;
          const awayToken = ml.polymarket?.markets.away.tokenId;
          const homeToken = ml.polymarket?.markets.home.tokenId;
          if (awayToken) labels[awayToken] = `${away} ML`;
          if (homeToken) labels[homeToken] = `${home} ML`;
        }

        if (game.marketType === 'spread') {
          const sd = game as SpreadGameData;
          if (sd.polymarketMain) {
            const favCode = sd.polymarketMain.favoriteSide === 'away' ? away : home;
            const dogCode = sd.polymarketMain.favoriteSide === 'away' ? home : away;
            const line = sd.polymarketMain.line;
            const favToken = sd.polymarketMain.markets.favorite.tokenId;
            const dogToken = sd.polymarketMain.markets.underdog.tokenId;
            if (favToken) labels[favToken] = `${favCode} -${line}`;
            if (dogToken) labels[dogToken] = `${dogCode} +${line}`;
          }
        }

        if (game.marketType === 'total') {
          const td = game as TotalGameData;
          if (td.polymarketMain) {
            const line = td.polymarketMain.line;
            const overToken = td.polymarketMain.markets.over.tokenId;
            const underToken = td.polymarketMain.markets.under.tokenId;
            if (overToken) labels[overToken] = `${away}@${home} O${line}`;
            if (underToken) labels[underToken] = `${away}@${home} U${line}`;
          }
        }
      }
    }

    return labels;
  }, [consolidatedGames, sportsGamesStructureKey]); // sportsGamesRef is stable, only recompute when structure changes

  // Trade modal request
  const placeOrderRequest: PlaceOrderRequest | null = useMemo(() => {
    if (!tradeReq) return null;
    return {
      venue: tradeReq.venue === 'polymarket' ? 'polymarket' : 'kalshi',
      marketTicker: tradeReq.marketTicker,
      tokenId: tradeReq.tokenId,
      conditionId: tradeReq.conditionId,
      tickSize: tradeReq.tickSize,
      negRisk: tradeReq.negRisk,
      side: tradeReq.side,
      priceCents: tradeReq.priceCents,
      isTakerMode: tradeReq.isTakerMode,
      teamLabel: tradeReq.teamLabel,
      gameDate: tradeReq.gameDate,
      startTimePt: tradeReq.startTimePt,
      defaultContracts: tradeReq.contracts,
    };
  }, [tradeReq]);

  const handleSubmitTrade = useCallback(
    async (submission: PlaceOrderSubmission) => {
      if (!tradeReq) return;
      setTradeError(null);
      setTradeSubmitting(true);
      try {
        if (tradeReq.venue === 'kalshi') {
          if (!onPlaceKalshiOrder) throw new Error('Kalshi trading not available');
          await onPlaceKalshiOrder({
            marketTicker: tradeReq.marketTicker,
            side: tradeReq.side,
            priceCents: tradeReq.priceCents,
            contracts: submission.contracts,
            isTakerMode: tradeReq.isTakerMode,
            expirationTs: submission.expirationTs,
          });
        } else {
          if (!onPlacePolymarketOrder) throw new Error('Polymarket trading not available');
          if (!tradeReq.tokenId) throw new Error('Missing tokenId for Polymarket order');
          await onPlacePolymarketOrder({
            tokenId: tradeReq.tokenId,
            conditionId: tradeReq.conditionId,
            action: 'buy',
            priceCents: tradeReq.priceCents,
            contracts: submission.contracts,
            isTakerMode: tradeReq.isTakerMode,
            tickSize: tradeReq.tickSize,
            negRisk: tradeReq.negRisk,
          });
        }
        setTradeReq(null);
      } catch (err) {
        const message = err instanceof Error ? err.message : String(err);
        setTradeError(message);
      } finally {
        setTradeSubmitting(false);
      }
    },
    [tradeReq, onPlaceKalshiOrder, onPlacePolymarketOrder]
  );

  // Selected game for the right-side detail panel
  const selectedGameKey = useMemo(() => {
    const keys = Array.from(expandedGameKeys);
    return keys[0] ?? null;
  }, [expandedGameKeys]);

  const selectedGame = useMemo(() => {
    if (!selectedGameKey) return null;
    return moneylineGamesForView.find((g) => g.key === selectedGameKey) ?? null;
  }, [selectedGameKey, moneylineGamesForView]);

  return (
    <div className="space-y-6 p-4">
      {/* Single row with all controls */}
      <DashboardHeader
        oddsMode={oddsMode}
        viewMode={viewMode}
        onModeChange={handleModeChange}
        onViewModeChange={handleViewModeChange}
        volumeWindows={volumeWindows}
        onToggleVolumeWindow={handleToggleVolumeWindow}
        leftContent={
          <>
            {/* Sport tabs */}
            <div className="bg-muted flex rounded-lg p-1">
              {SPORT_TABS.map((tab) => (
                <button
                  key={tab.value}
                  onClick={() => handleSportChange(tab.value)}
                  className={cn(
                    'rounded-md px-3 py-1.5 text-sm font-medium transition-colors',
                    selectedSport === tab.value
                      ? 'bg-primary text-primary-foreground'
                      : 'text-muted-foreground hover:text-foreground'
                  )}
                >
                  {tab.label}
                </button>
              ))}
            </div>
            {/* Market type tabs */}
            <div className="bg-muted flex rounded-lg p-1">
              {availableMarketTypes.map((type) => (
                <button
                  key={type}
                  onClick={() => handleMarketTypeChange(type)}
                  className={cn(
                    'rounded-md px-4 py-1.5 text-sm font-medium transition-colors',
                    selectedMarketType === type
                      ? 'bg-primary text-primary-foreground'
                      : 'text-muted-foreground hover:text-foreground'
                  )}
                >
                  {MARKET_TYPE_LABELS[type]}
                </button>
              ))}
            </div>

            {/* Date navigator */}
            <div className="border-border flex items-center gap-1 rounded border px-1 py-0.5">
              <button
                onClick={handleDatePrev}
                disabled={!selectedDate || availableDates.indexOf(selectedDate) <= 0}
                className="text-muted-foreground hover:text-foreground rounded p-0.5 transition-colors disabled:opacity-30"
                title="Previous date"
              >
                <ChevronLeft className="h-4 w-4" />
              </button>
              <input
                type="date"
                value={selectedDate ?? ''}
                onChange={(e) => setSelectedDate(e.target.value || null)}
                className="text-foreground w-[7rem] cursor-pointer border-none bg-transparent px-1 py-0.5 text-xs font-medium outline-none [color-scheme:dark]"
                title="Click to pick a date"
              />
              <button
                onClick={handleDateNext}
                disabled={
                  !selectedDate || availableDates.indexOf(selectedDate) >= availableDates.length - 1
                }
                className="text-muted-foreground hover:text-foreground rounded p-0.5 transition-colors disabled:opacity-30"
                title="Next date"
              >
                <ChevronRight className="h-4 w-4" />
              </button>
              <button
                onClick={handleDateToday}
                className={cn(
                  'rounded px-2 py-0.5 text-xs font-medium transition-colors',
                  selectedDate === getTodayStr()
                    ? 'bg-primary text-primary-foreground'
                    : 'text-muted-foreground hover:text-foreground'
                )}
                title="Jump to today"
              >
                Today
              </button>
              <button
                onClick={handleDateAll}
                className={cn(
                  'rounded px-2 py-0.5 text-xs font-medium transition-colors',
                  selectedDate === null
                    ? 'bg-primary text-primary-foreground'
                    : 'text-muted-foreground hover:text-foreground'
                )}
                title="Show all dates"
              >
                All
              </button>
            </div>
          </>
        }
      />

      {/* Main content: table + optional right-side detail panel */}
      <div className="flex gap-4">
        {/* Table area (grows to fill available space) */}
        <div className="min-w-0 flex-1">
          {/* Moneyline View */}
          {useMoneylineView &&
            (viewMode === 'two-row' ? (
              <TwoRowMoneylines
                games={moneylineGamesForView}
                isLoading={moneylineLoading}
                loadingState={currentLoadingState}
                oddsMode={oddsMode}
                maxLiquidity={maxLiquidity}
                expandedGameKeys={expandedGameKeys}
                onToggleExpand={handleToggleExpand}
                onTradeClick={(req) => {
                  setTradeError(null);
                  setTradeReq(req);
                }}
                onNavigateToMarket={handleNavigateToMarket}
                sport={selectedSport}
              />
            ) : (
              <ConsolidatedMoneylines
                games={moneylineGamesForView}
                isLoading={moneylineLoading}
                loadingState={currentLoadingState}
                oddsMode={oddsMode}
                maxLiquidity={maxLiquidity}
                expandedGameKeys={expandedGameKeys}
                onToggleExpand={handleToggleExpand}
                onTradeClick={(req) => {
                  setTradeError(null);
                  setTradeReq(req);
                }}
                onNavigateToMarket={handleNavigateToMarket}
                sport={selectedSport}
              />
            ))}

          {/* Totals view */}
          {selectedMarketType === 'total' && (
            <TwoRowTotals
              sport={selectedSport}
              games={filteredCurrentGames as TotalGameData[]}
              isLoading={currentLoading}
              loadingState={currentLoadingState}
              volumeWindows={volumeWindows}
              onTradeClick={(req) => {
                setTradeError(null);
                setTradeReq(req);
              }}
              onNavigateToMarket={handleNavigateToMarket}
            />
          )}

          {/* Spreads view */}
          {selectedMarketType === 'spread' && (
            <TwoRowSpreads
              sport={selectedSport}
              games={filteredCurrentGames as SpreadGameData[]}
              isLoading={currentLoading}
              loadingState={currentLoadingState}
              volumeWindows={volumeWindows}
              onTradeClick={(req) => {
                setTradeError(null);
                setTradeReq(req);
              }}
              onNavigateToMarket={handleNavigateToMarket}
            />
          )}
        </div>

        {/* Right-side game detail panel */}
        {selectedGame && useMoneylineView && (
          <GameDetailPanel
            game={selectedGame}
            oddsMode={oddsMode}
            onTradeClick={(req) => {
              setTradeError(null);
              setTradeReq(req);
            }}
            onClose={() => setExpandedGameKeys(new Set())}
          />
        )}
      </div>

      {/* Positions panel (always shown) */}
      <OpenPositionsPanel
        positions={positions ?? []}
        markets={markets}
        titleByTicker={positionTitleByTicker}
        gameDateByTicker={positionGameDateByTicker}
        outcomeByTicker={positionOutcomeByTicker}
      />

      {/* Orders panel (always shown) */}
      <OpenOrdersPanel
        kalshiOrders={_orders ?? []}
        polyOrders={polyOpenOrders}
        polyStatus={polyStatus}
        titleByTicker={positionTitleByTicker}
        polyTokenLabels={polyTokenLabels}
        onCancelOrder={handleCancelOrder}
        onDecreaseOrder={handleDecreaseOrder}
        onKillAll={handleKillAll}
        onCancelPolyOrder={handleCancelPolyOrder}
        onKillAllPolyOrders={handleKillAllPolyOrders}
      />

      {/* Combined Orders & Positions panel */}
      <CombinedOrdersPositionsPanel
        kalshiOrders={_orders ?? []}
        positions={positions ?? []}
        titleByTicker={positionTitleByTicker}
        gameDateByTicker={positionGameDateByTicker}
        outcomeByTicker={positionOutcomeByTicker}
        orderbookByTicker={orderbookByTicker}
        onCancelOrder={handleCancelOrder}
        onKillAll={handleKillAll}
      />

      {/* Trade Modal */}
      <PlaceOrderModal
        isOpen={tradeReq !== null}
        request={placeOrderRequest}
        submitting={tradeSubmitting}
        error={tradeError}
        onClose={() => setTradeReq(null)}
        onSubmit={handleSubmitTrade}
      />
    </div>
  );
}

export default ValueDashboardView;
