/**
 * useDevModeData — Fetches all Kalshi sports series + markets for dev mode.
 *
 * Only active when dev mode is enabled. Fetches series on mount,
 * then markets + orderbooks for the selected series with 30s auto-refresh.
 */

import { useState, useEffect, useCallback, useRef } from 'react';
import type { KalshiApiClient } from '@/lib/kalshiApi';
import type { KalshiSeries, KalshiMarket, KalshiOrderbook } from '@/types';

interface DevModeData {
  series: KalshiSeries[];
  selectedTicker: string | null;
  markets: KalshiMarket[];
  orderbooks: Map<string, KalshiOrderbook>;
  loading: boolean;
  seriesLoading: boolean;
  selectSeries: (ticker: string) => void;
  refresh: () => void;
}

const REFRESH_INTERVAL_MS = 30_000;

export function useDevModeData(
  apiClientRef: React.RefObject<KalshiApiClient | null>,
  enabled: boolean
): DevModeData {
  const [series, setSeries] = useState<KalshiSeries[]>([]);
  const [selectedTicker, setSelectedTicker] = useState<string | null>(null);
  const [markets, setMarkets] = useState<KalshiMarket[]>([]);
  const [orderbooks, setOrderbooks] = useState<Map<string, KalshiOrderbook>>(new Map());
  const [loading, setLoading] = useState(false);
  const [seriesLoading, setSeriesLoading] = useState(false);
  const refreshTimerRef = useRef<ReturnType<typeof setInterval> | null>(null);

  // Fetch all sports series
  const fetchSeries = useCallback(async () => {
    const api = apiClientRef.current;
    if (!api) return;

    setSeriesLoading(true);
    try {
      const result = await api.getSeries({ category: 'Sports' });
      setSeries(result);
    } catch (err) {
      console.error('[DevMode] Failed to fetch series:', err);
    } finally {
      setSeriesLoading(false);
    }
  }, [apiClientRef]);

  // Fetch markets + orderbooks for selected series
  const fetchMarkets = useCallback(async () => {
    const api = apiClientRef.current;
    if (!api || !selectedTicker) return;

    setLoading(true);
    try {
      const result = await api.getMarkets({
        series_ticker: selectedTicker,
        status: 'open',
        limit: 200,
      });
      setMarkets(result);

      // Fetch orderbooks in parallel batches of 5 (balances speed vs rate limits)
      const tickersToFetch = result.slice(0, 20);
      const obMap = new Map<string, KalshiOrderbook>();
      const BATCH_SIZE = 5;
      for (let i = 0; i < tickersToFetch.length; i += BATCH_SIZE) {
        const batch = tickersToFetch.slice(i, i + BATCH_SIZE);
        const results = await Promise.allSettled(
          batch.map((market) => api.getOrderbook(market.market_ticker))
        );
        for (let j = 0; j < results.length; j++) {
          const r = results[j];
          if (r && r.status === 'fulfilled' && r.value) {
            obMap.set(batch[j]!.market_ticker, r.value);
          }
        }
      }
      setOrderbooks(obMap);
    } catch (err) {
      console.error('[DevMode] Failed to fetch markets:', err);
    } finally {
      setLoading(false);
    }
  }, [apiClientRef, selectedTicker]);

  // Fetch series when enabled
  useEffect(() => {
    if (enabled) {
      fetchSeries();
    } else {
      setSeries([]);
      setSelectedTicker(null);
      setMarkets([]);
      setOrderbooks(new Map());
    }
  }, [enabled, fetchSeries]);

  // Fetch markets when series selection changes
  useEffect(() => {
    if (enabled && selectedTicker) {
      fetchMarkets();
    } else {
      setMarkets([]);
      setOrderbooks(new Map());
    }
  }, [enabled, selectedTicker, fetchMarkets]);

  // Auto-refresh on interval
  useEffect(() => {
    if (!enabled || !selectedTicker) return;

    refreshTimerRef.current = setInterval(fetchMarkets, REFRESH_INTERVAL_MS);
    return () => {
      if (refreshTimerRef.current) {
        clearInterval(refreshTimerRef.current);
        refreshTimerRef.current = null;
      }
    };
  }, [enabled, selectedTicker, fetchMarkets]);

  const selectSeries = useCallback((ticker: string) => {
    setSelectedTicker(ticker);
  }, []);

  const refresh = useCallback(() => {
    if (selectedTicker) {
      fetchMarkets();
    } else {
      fetchSeries();
    }
  }, [selectedTicker, fetchMarkets, fetchSeries]);

  return {
    series,
    selectedTicker,
    markets,
    orderbooks,
    loading,
    seriesLoading,
    selectSeries,
    refresh,
  };
}
