/**
 * useDiagnostics Hook
 *
 * Wraps the diagnostics runner. Manages results state, exposes runAll/runOne.
 * Handles cleanup on unmount and guards against concurrent runs.
 */

import { useState, useCallback, useRef, useEffect } from 'react';
import type { DiagnosticResult, DiagnosticContext } from '@/lib/diagnostics';
import { runAll, runOne } from '@/lib/diagnostics';

export function useDiagnostics(ctx: DiagnosticContext) {
  const [results, setResults] = useState<Map<string, DiagnosticResult>>(new Map());
  const [isRunning, setIsRunning] = useState(false);
  const [lastRunAt, setLastRunAt] = useState<number | null>(null);
  const abortRef = useRef(false);
  const runningRef = useRef(false);

  // Abort on unmount to prevent state updates after teardown
  useEffect(() => {
    return () => {
      abortRef.current = true;
    };
  }, []);

  const handleRunAll = useCallback(async () => {
    // Guard against concurrent runs
    if (runningRef.current) return;
    runningRef.current = true;
    setIsRunning(true);
    setResults(new Map());
    abortRef.current = false;

    try {
      const gen = runAll(ctx);
      for await (const result of gen) {
        if (abortRef.current) break;
        setResults((prev) => {
          const next = new Map(prev);
          next.set(result.checkId, result);
          return next;
        });
      }
    } finally {
      runningRef.current = false;
      if (!abortRef.current) {
        setLastRunAt(Date.now());
        setIsRunning(false);
      }
    }
  }, [ctx]);

  const handleRunOne = useCallback(
    async (checkId: string) => {
      const result = await runOne(checkId, ctx);
      if (!abortRef.current) {
        setResults((prev) => {
          const next = new Map(prev);
          next.set(result.checkId, result);
          return next;
        });
      }
    },
    [ctx]
  );

  const stop = useCallback(() => {
    abortRef.current = true;
    runningRef.current = false;
    setIsRunning(false);
  }, []);

  return {
    results,
    isRunning,
    lastRunAt,
    runAll: handleRunAll,
    runOne: handleRunOne,
    stop,
  };
}
