import { useState, useEffect, useRef, useCallback } from 'react';
import { createKalshiClient } from '../lib/kalshiApi';
import {
  ensureKalshiProfilesStore,
  getActiveKalshiProfile,
  saveKalshiProfilesStore,
  upsertKalshiProfile,
  upsertKalshiProfileByAccessKey,
  setActiveKalshiProfile,
  deleteKalshiProfile,
  type KalshiProfilesStore,
} from '../lib/kalshiProfiles';
import { primeSeriesCache, clearSeriesCache } from '../lib/nba';
import type { KalshiCredentials } from '../types';

export interface KalshiConnectionParams {
  navigate: (to: string, opts?: { replace?: boolean }) => void;
  location: { pathname: string };
}

export interface KalshiConnectionReturn {
  isConnected: boolean;
  isAutoConnecting: boolean;
  connectionStatus: string;
  errorMessage: string | null;
  environment: 'prod' | 'demo';
  credentials: KalshiCredentials | null;
  kalshiStore: KalshiProfilesStore;
  apiClientRef: React.MutableRefObject<Awaited<ReturnType<typeof createKalshiClient>> | null>;
  setIsConnected: React.Dispatch<React.SetStateAction<boolean>>;
  setConnectionStatus: React.Dispatch<React.SetStateAction<string>>;
  setErrorMessage: React.Dispatch<React.SetStateAction<string | null>>;
  handleConnect: (
    accessKey: string,
    privateKey: string,
    env: 'prod' | 'demo',
    useRelay: boolean,
    remember: boolean
  ) => Promise<void>;
  /** Thin disconnect: resets connection state + apiClient. Caller composes full cleanup. */
  disconnect: () => void;
  handleKalshiReconnect: (args: {
    id?: string;
    name?: string;
    accessKey: string;
    privateKey: string;
    remember: boolean;
  }) => Promise<void>;
  handleKalshiSetActive: (profileId: string) => void;
  handleKalshiDeleteProfile: (profileId: string) => void;
  handleKalshiSaveProfile: (args: {
    id?: string;
    name?: string;
    accessKey: string;
    privateKey: string;
  }) => void;
  portfolioPollingEnabled: boolean;
  setPortfolioPollingEnabled: React.Dispatch<React.SetStateAction<boolean>>;
}

export function useKalshiConnection({
  navigate,
  location,
}: KalshiConnectionParams): KalshiConnectionReturn {
  const [isConnected, setIsConnected] = useState(false);
  const [isAutoConnecting, setIsAutoConnecting] = useState(false);
  const [connectionStatus, setConnectionStatus] = useState('Disconnected');
  const [errorMessage, setErrorMessage] = useState<string | null>(null);
  const [environment, setEnvironment] = useState<'prod' | 'demo'>('prod');
  const [credentials, setCredentials] = useState<KalshiCredentials | null>(null);
  const [kalshiStore, setKalshiStore] = useState<KalshiProfilesStore>(() =>
    ensureKalshiProfilesStore()
  );
  const [portfolioPollingEnabled, setPortfolioPollingEnabled] = useState(true);

  const apiClientRef = useRef<Awaited<ReturnType<typeof createKalshiClient>> | null>(null);
  const autoConnectAttemptedRef = useRef(false);

  // Auto-clear transient errors so stale messages don't linger forever.
  useEffect(() => {
    if (!errorMessage) return;
    const timeoutId = window.setTimeout(() => {
      setErrorMessage((current) => (current === errorMessage ? null : current));
    }, 7000);
    return () => window.clearTimeout(timeoutId);
  }, [errorMessage]);

  // Route protection: redirect based on auth state
  useEffect(() => {
    if (isAutoConnecting) return;

    const isAppRoute = location.pathname.startsWith('/app');

    if (isConnected && !isAppRoute) {
      navigate('/app/value-dashboard', { replace: true });
    } else if (!isConnected && isAppRoute) {
      const active = getActiveKalshiProfile(kalshiStore);
      if (active) return; // auto-connect will handle this
      navigate('/', { replace: true });
    }
  }, [isConnected, isAutoConnecting, location.pathname, navigate, kalshiStore]);

  // Handle connect — auth only. Stream/position/order init happens in downstream hooks via reactive effects.
  const handleConnect = useCallback(
    async (
      accessKey: string,
      privateKey: string,
      env: 'prod' | 'demo',
      useRelay: boolean,
      remember: boolean
    ) => {
      console.log('App: handleConnect called', { useRelay, env });
      try {
        setConnectionStatus('Connecting...');
        setErrorMessage(null);
        setPortfolioPollingEnabled(true);
        setEnvironment(env);

        setCredentials({
          accessKeyId: accessKey,
          privateKeyPem: privateKey,
          environment: env,
        });

        console.log('App: Creating Kalshi client...');
        const apiClient = await createKalshiClient(accessKey, privateKey, env, useRelay);
        apiClientRef.current = apiClient;
        console.log('App: Kalshi client created');

        // Test connection with a lightweight call (also primes the series cache)
        console.log('App: Testing connection...');
        const sportsSeries = await apiClient.getSeries({ category: 'Sports' });
        primeSeriesCache(sportsSeries);
        console.log('App: Connection test successful');

        // === SPORTS SERIES DISCOVERY ===
        console.log('\n=== SPORTS SERIES DISCOVERY ===');
        console.table(
          sportsSeries.map((s) => ({
            ticker: s.series_ticker,
            title: s.title,
            category: s.category,
            subcategory: s.subcategory,
          }))
        );
        (window as unknown as Record<string, unknown>).__sportsSeries = sportsSeries;
        (window as unknown as Record<string, unknown>).__kalshiApi = apiClient;
        console.log('Tip: Access window.__sportsSeries and window.__kalshiApi for exploration');
        console.log('=====================================\n');

        // Mark as connected — downstream hooks will react to this
        setIsConnected(true);
        setConnectionStatus('Connected');

        // Navigate to app after successful login
        if (!location.pathname.startsWith('/app')) {
          navigate('/app/value-dashboard');
        }

        if (remember) {
          setKalshiStore((prev) => {
            const next = upsertKalshiProfileByAccessKey(prev, {
              accessKey,
              privateKey,
              environment: env,
              useRelay,
            });
            saveKalshiProfilesStore(next);
            return next;
          });
        }
      } catch (error) {
        let message = 'Connection failed';
        if (error instanceof Error) {
          message = error.message;
          if (
            message.includes('CORS error') ||
            (message.includes('Failed to fetch') && !useRelay)
          ) {
            message =
              'CORS error: Direct connection blocked by browser. Please use "Connect Through Server" mode or enable CORS on Kalshi API.';
          } else if (message.includes('Relay error')) {
            message = `Cannot connect to relay server. Make sure it's running on port 8787. ${message}`;
          } else if (message.includes('Invalid PEM format')) {
            message = `Invalid private key format. ${message}`;
          } else if (message.includes('401') || message.includes('Unauthorized')) {
            if (
              !message.toLowerCase().includes('api has been moved') &&
              !message.toLowerCase().includes('moved to') &&
              !message.toLowerCase().includes('signature') &&
              !message.toLowerCase().includes('timestamp')
            ) {
              message =
                'Unauthorized (401). Please verify your Access Key ID + Private Key, and that you are using the correct environment (prod vs demo).';
            }
          } else if (message.includes('403') || message.includes('Forbidden')) {
            message = 'Access forbidden. Check your API key permissions.';
          } else if (message.includes('Failed to fetch') || message.includes('NetworkError')) {
            if (useRelay) {
              message = 'Network error. Check if relay server is running and accessible.';
            } else {
              message = 'Network error. Check your internet connection and Kalshi API status.';
            }
          }
        }
        console.error('Connection error details:', error);
        setErrorMessage(message);
        setConnectionStatus('Disconnected');
      }
    },
    [navigate, location.pathname]
  );

  // Auto-reconnect on mount if saved credentials exist
  useEffect(() => {
    if (autoConnectAttemptedRef.current) return;
    if (isConnected) return;

    try {
      const active = getActiveKalshiProfile(kalshiStore);
      if (!active) return;
      const { accessKey, privateKey, environment: savedEnv, useRelay } = active;

      if (!accessKey || !privateKey) return;

      autoConnectAttemptedRef.current = true;
      setIsAutoConnecting(true);
      setConnectionStatus('Auto-connecting...');

      handleConnect(
        accessKey,
        privateKey,
        savedEnv === 'prod' ? 'prod' : 'demo',
        useRelay !== false,
        true
      ).finally(() => {
        setIsAutoConnecting(false);
      });
    } catch (error) {
      console.error('Auto-connect failed:', error);
      setIsAutoConnecting(false);
    }
  }, [handleConnect, isConnected, kalshiStore]);

  // Thin disconnect: resets connection state only. Full cleanup composed in App.tsx.
  const disconnect = useCallback(() => {
    apiClientRef.current = null;
    clearSeriesCache();
    setIsConnected(false);
    setConnectionStatus('Disconnected');
    setErrorMessage(null);
    setCredentials(null);
    setPortfolioPollingEnabled(true);
    navigate('/');
  }, [navigate]);

  const handleKalshiReconnect = useCallback(
    async (args: {
      id?: string;
      name?: string;
      accessKey: string;
      privateKey: string;
      remember: boolean;
    }) => {
      const accessKey = String(args.accessKey || '').trim();
      const privateKey = String(args.privateKey || '').trim();
      if (!accessKey || !privateKey) {
        throw new Error('Access Key ID and Private Key are required.');
      }

      if (args.remember) {
        setKalshiStore((prev) => {
          const next = upsertKalshiProfile(prev, {
            id: args.id,
            name: args.name || accessKey,
            accessKey,
            privateKey,
            environment: 'prod',
            useRelay: true,
          });
          saveKalshiProfilesStore(next);
          return next;
        });
      }

      // Note: App.tsx composes full disconnect before reconnecting
      await handleConnect(accessKey, privateKey, 'prod', true, args.remember);
    },
    [handleConnect]
  );

  const handleKalshiSetActive = useCallback((profileId: string) => {
    setKalshiStore((prev) => {
      const next = setActiveKalshiProfile(prev, profileId);
      saveKalshiProfilesStore(next);
      return next;
    });
  }, []);

  const handleKalshiDeleteProfile = useCallback((profileId: string) => {
    setKalshiStore((prev) => {
      const next = deleteKalshiProfile(prev, profileId);
      saveKalshiProfilesStore(next);
      return next;
    });
  }, []);

  const handleKalshiSaveProfile = useCallback(
    (args: { id?: string; name?: string; accessKey: string; privateKey: string }) => {
      const accessKey = String(args.accessKey || '').trim();
      const privateKey = String(args.privateKey || '').trim();
      if (!accessKey || !privateKey) {
        throw new Error('Access Key ID and Private Key are required.');
      }

      setKalshiStore((prev) => {
        const next = upsertKalshiProfile(prev, {
          id: args.id,
          name: args.name || accessKey,
          accessKey,
          privateKey,
          environment: 'prod',
          useRelay: true,
        });
        saveKalshiProfilesStore(next);
        return next;
      });
    },
    []
  );

  return {
    isConnected,
    isAutoConnecting,
    connectionStatus,
    errorMessage,
    environment,
    credentials,
    kalshiStore,
    apiClientRef,
    setIsConnected,
    setConnectionStatus,
    setErrorMessage,
    handleConnect,
    disconnect,
    handleKalshiReconnect,
    handleKalshiSetActive,
    handleKalshiDeleteProfile,
    handleKalshiSaveProfile,
    portfolioPollingEnabled,
    setPortfolioPollingEnabled,
  };
}
