import { useState, useEffect, useRef, useCallback } from 'react';
import type { ClobClient } from '@polymarket/clob-client';
import {
  loadPolyUserConfig,
  savePolyUserConfig,
  clearPolyUserConfig,
  createOrDerivePolyApiCreds,
  createPolyTradingClient,
  placePolyOrder,
  cancelPolyOrder,
  cancelAllPolyOrders,
  type PolyUserConfig,
} from '../lib/polymarket/client';
import { createPolyUserStream } from '../lib/polymarket/userStream';
import type { PolyOpenOrder } from '../lib/polymarket/types';
import type { ConsolidatedGameBooks } from '../lib/nbaConsolidated/types';

export interface PolymarketIntegrationParams {
  isConnected: boolean;
  consolidatedGames: ConsolidatedGameBooks[];
  setErrorMessage: React.Dispatch<React.SetStateAction<string | null>>;
}

export interface PolymarketIntegrationReturn {
  polyConfig: PolyUserConfig | null;
  polyOpenOrders: PolyOpenOrder[];
  polyStatus: 'disconnected' | 'connected' | 'error';
  handlePolyConnect: (args: {
    privateKey: string;
    signatureType: 0 | 1 | 2;
    funder?: string;
  }) => Promise<void>;
  handlePolyDisconnect: () => void;
  refreshPolyOpenOrders: () => Promise<void>;
  handlePlacePolymarketDashboardOrder: (args: {
    tokenId: string;
    conditionId?: string;
    action: 'buy' | 'sell';
    priceCents: number;
    contracts: number;
    isTakerMode: boolean;
    tickSize?: string;
    negRisk?: boolean;
  }) => Promise<void>;
  handleCancelPolyOrder: (orderId: string) => Promise<void>;
  handleKillAllPolyOrders: () => Promise<void>;
  cleanupPoly: () => void;
}

export function usePolymarketIntegration({
  isConnected,
  consolidatedGames,
  setErrorMessage,
}: PolymarketIntegrationParams): PolymarketIntegrationReturn {
  const [polyConfig, setPolyConfig] = useState<PolyUserConfig | null>(() => loadPolyUserConfig());
  const [polyOpenOrders, setPolyOpenOrders] = useState<PolyOpenOrder[]>([]);
  const [polyStatus, setPolyStatus] = useState<'disconnected' | 'connected' | 'error'>(
    'disconnected'
  );

  const polyClientRef = useRef<ClobClient | null>(null);
  const polyUserStreamRef = useRef<ReturnType<typeof createPolyUserStream> | null>(null);

  const computePolyConditionIdsFromGames = useCallback(
    (games: ConsolidatedGameBooks[]): string[] => {
      const out = new Set<string>();
      for (const g of games) {
        const a = g.polymarket?.markets.away.conditionId;
        const h = g.polymarket?.markets.home.conditionId;
        if (a) out.add(String(a));
        if (h) out.add(String(h));
      }
      return Array.from(out);
    },
    []
  );

  // Keep Polymarket user channel subscribed to active NBA condition IDs.
  useEffect(() => {
    if (!isConnected) return;
    if (!polyConfig?.apiCreds) return;
    const conditionIds = computePolyConditionIdsFromGames(consolidatedGames);
    if (conditionIds.length === 0) return;

    if (!polyUserStreamRef.current) {
      const us = createPolyUserStream(polyConfig.apiCreds);
      polyUserStreamRef.current = us;
      us.onError((err) => {
        console.warn('Polymarket user stream error:', err);
        setPolyStatus('error');
      });
      us.onOrder((evt) => {
        setPolyOpenOrders((prev) => {
          const nextMap = new Map(prev.map((o) => [o.id, o]));
          const id = String(evt.id || '');
          if (!id) return prev;
          if (String(evt.type).toUpperCase() === 'CANCELLATION') {
            nextMap.delete(id);
          } else {
            const price = Number(evt.price);
            const originalSize = Number(evt.original_size ?? 0);
            const sizeMatched = Number(evt.size_matched ?? 0);
            nextMap.set(id, {
              id,
              assetId: String(evt.asset_id || ''),
              conditionId: String(evt.market || ''),
              side: String(evt.side || ''),
              price: Number.isFinite(price) ? price : 0,
              originalSize: Number.isFinite(originalSize) ? originalSize : 0,
              sizeMatched: Number.isFinite(sizeMatched) ? sizeMatched : 0,
              timestampMs: evt.timestamp ? Number(evt.timestamp) * 1000 : undefined,
            });
          }
          return Array.from(nextMap.values());
        });
      });

      void us.connect(conditionIds).then(
        () => setPolyStatus('connected'),
        () => setPolyStatus('error')
      );
      return;
    }

    polyUserStreamRef.current.subscribe(conditionIds);
  }, [isConnected, polyConfig?.apiCreds, consolidatedGames, computePolyConditionIdsFromGames]);

  const handlePolyConnect = useCallback(
    async (args: { privateKey: string; signatureType: 0 | 1 | 2; funder?: string }) => {
      const apiCreds = await createOrDerivePolyApiCreds(args.privateKey);
      const next: PolyUserConfig = {
        privateKey: args.privateKey,
        signatureType: args.signatureType,
        funder: args.funder,
        apiCreds,
      };
      savePolyUserConfig(next);
      setPolyConfig(next);

      try {
        polyClientRef.current = createPolyTradingClient(next);
        setPolyStatus('connected');
      } catch (e) {
        polyClientRef.current = null;
        setPolyStatus('error');
        throw e;
      }
    },
    []
  );

  const handlePolyDisconnect = useCallback(() => {
    clearPolyUserConfig();
    setPolyConfig(null);
    setPolyOpenOrders([]);
    setPolyStatus('disconnected');
    polyClientRef.current = null;
    if (polyUserStreamRef.current) {
      polyUserStreamRef.current.disconnect();
      polyUserStreamRef.current = null;
    }
  }, []);

  const refreshPolyOpenOrders = useCallback(async () => {
    try {
      const maybe = polyClientRef.current as unknown as {
        getOpenOrders?: () => Promise<unknown>;
      } | null;
      const raw = await maybe?.getOpenOrders?.();
      if (Array.isArray(raw)) {
        setPolyOpenOrders(
          raw.map((o) => {
            const rec: Record<string, unknown> =
              o && typeof o === 'object' ? (o as Record<string, unknown>) : {};
            const n = (v: unknown) => {
              const x = typeof v === 'number' ? v : typeof v === 'string' ? Number(v) : NaN;
              return Number.isFinite(x) ? x : 0;
            };
            const id = String(rec.id ?? rec.orderID ?? rec.orderId ?? '');
            return {
              id,
              assetId: String(rec.asset_id ?? rec.assetId ?? ''),
              conditionId: String(rec.market ?? rec.conditionId ?? ''),
              side: String(rec.side ?? ''),
              price: n(rec.price),
              originalSize: n(rec.original_size ?? rec.size ?? rec.originalSize),
              sizeMatched: n(rec.size_matched ?? rec.sizeMatched),
              timestampMs: rec.timestamp ? n(rec.timestamp) * 1000 : undefined,
            };
          })
        );
      }
    } catch {
      // ignore
    }
  }, []);

  const handlePlacePolymarketDashboardOrder = useCallback(
    async (args: {
      tokenId: string;
      conditionId?: string;
      action: 'buy' | 'sell';
      priceCents: number;
      contracts: number;
      isTakerMode: boolean;
      tickSize?: string;
      negRisk?: boolean;
    }) => {
      if (!polyClientRef.current) {
        if (!polyConfig) throw new Error('Polymarket not configured (go to Settings).');
        polyClientRef.current = createPolyTradingClient(polyConfig);
      }

      await placePolyOrder({
        client: polyClientRef.current,
        tokenId: args.tokenId,
        action: args.action,
        priceCents: args.priceCents,
        contracts: args.contracts,
        postOnly: !args.isTakerMode,
        tickSize: args.tickSize,
        negRisk: args.negRisk,
      });

      await refreshPolyOpenOrders();
    },
    [polyConfig, refreshPolyOpenOrders]
  );

  const handleCancelPolyOrder = useCallback(
    async (orderId: string) => {
      if (!polyClientRef.current) {
        if (!polyConfig) {
          setErrorMessage('Polymarket not configured (go to Settings).');
          return;
        }
        polyClientRef.current = createPolyTradingClient(polyConfig);
      }
      try {
        await cancelPolyOrder(polyClientRef.current, orderId);
        await refreshPolyOpenOrders();
      } catch (error) {
        const message =
          error instanceof Error ? error.message : 'Failed to cancel Polymarket order';
        setErrorMessage(message);
      }
    },
    [polyConfig, refreshPolyOpenOrders, setErrorMessage]
  );

  const handleKillAllPolyOrders = useCallback(async () => {
    if (!polyClientRef.current) {
      if (!polyConfig) {
        setErrorMessage('Polymarket not configured (go to Settings).');
        return;
      }
      polyClientRef.current = createPolyTradingClient(polyConfig);
    }
    try {
      await cancelAllPolyOrders(polyClientRef.current);
      await refreshPolyOpenOrders();
    } catch (error) {
      const message =
        error instanceof Error ? error.message : 'Failed to cancel all Polymarket orders';
      setErrorMessage(message);
    }
  }, [polyConfig, refreshPolyOpenOrders, setErrorMessage]);

  const cleanupPoly = useCallback(() => {
    if (polyUserStreamRef.current) {
      polyUserStreamRef.current.disconnect();
      polyUserStreamRef.current = null;
    }
    polyClientRef.current = null;
    setPolyOpenOrders([]);
    setPolyStatus('disconnected');
  }, []);

  return {
    polyConfig,
    polyOpenOrders,
    polyStatus,
    handlePolyConnect,
    handlePolyDisconnect,
    refreshPolyOpenOrders,
    handlePlacePolymarketDashboardOrder,
    handleCancelPolyOrder,
    handleKillAllPolyOrders,
    cleanupPoly,
  };
}
