import { useState, useEffect, useRef, useCallback } from 'react';
import { createKalshiClient } from '../lib/kalshiApi';
import { PositionTracker } from '../lib/positionTracker';
import { createMarketStream, type MarketStream } from '../lib/marketStream';
import { initAuth } from '../lib/kalshiAuth';
import { normalizeKalshiPriceToProbability } from '../lib/price';
import type { NBAMarketRow, Position, FillNotification, KalshiCredentials } from '../types';

export interface PositionTrackingParams {
  isConnected: boolean;
  apiClientRef: React.MutableRefObject<Awaited<ReturnType<typeof createKalshiClient>> | null>;
  credentials: KalshiCredentials | null;
  marketsDataRef: React.MutableRefObject<Map<string, NBAMarketRow>>;
  setErrorMessage: React.Dispatch<React.SetStateAction<string | null>>;
  portfolioPollingEnabled: boolean;
  setPortfolioPollingEnabled: React.Dispatch<React.SetStateAction<boolean>>;
}

export interface PositionTrackingReturn {
  positions: Position[];
  totalPnl: number;
  positionTitleByTicker: Record<string, string>;
  positionGameDateByTicker: Record<string, string>;
  positionOutcomeByTicker: Record<string, { yes?: string; no?: string }>;
  positionPriceStreamConnected: boolean;
  positionTrackerRef: React.MutableRefObject<PositionTracker>;
  positionPriceStreamRef: React.MutableRefObject<MarketStream | null>;
  subscribedPositionTickersRef: React.MutableRefObject<Set<string>>;
  loadPositions: () => Promise<void>;
  refreshPositions: () => void;
  handleClosePosition: (ticker: string, side: 'yes' | 'no', quantity: number) => Promise<void>;
  cleanupPositions: () => void;
}

export function usePositionTracking({
  isConnected,
  apiClientRef,
  credentials,
  marketsDataRef,
  setErrorMessage,
  portfolioPollingEnabled,
  setPortfolioPollingEnabled,
}: PositionTrackingParams): PositionTrackingReturn {
  const [positions, setPositions] = useState<Position[]>([]);
  const [totalPnl, setTotalPnl] = useState(0);
  const [positionTitleByTicker, setPositionTitleByTicker] = useState<Record<string, string>>({});
  const [positionGameDateByTicker, setPositionGameDateByTicker] = useState<Record<string, string>>(
    {}
  );
  const [positionOutcomeByTicker, setPositionOutcomeByTicker] = useState<
    Record<string, { yes?: string; no?: string }>
  >({});
  const [positionPriceStreamConnected, setPositionPriceStreamConnected] = useState(false);

  const positionTrackerRef = useRef<PositionTracker>(new PositionTracker());
  const positionPriceStreamRef = useRef<MarketStream | null>(null);
  const subscribedPositionTickersRef = useRef<Set<string>>(new Set());

  const refreshPositions = useCallback(() => {
    const tracker = positionTrackerRef.current;
    const positionsData = tracker.getPositions();
    setPositions(positionsData);
    setTotalPnl(tracker.getTotalPnl());
  }, []);

  const loadPositions = useCallback(async () => {
    if (!apiClientRef.current) return;
    if (!portfolioPollingEnabled) return;
    try {
      const apiPositions = await apiClientRef.current.getPositions();

      // Fetch market titles for positions (for UI labels), cached in state.
      (async () => {
        const client = apiClientRef.current;
        if (!client) return;
        for (const apiPos of apiPositions) {
          const ticker = apiPos.ticker;
          if (!ticker) continue;
          const hasTitle = !!positionTitleByTicker[ticker];
          const hasGameDate = !!positionGameDateByTicker[ticker];
          const hasOutcomes = !!positionOutcomeByTicker[ticker];
          if (hasTitle && hasGameDate && hasOutcomes) continue;
          const market = await client.getMarket(ticker);
          if (market) {
            const title = (market as unknown as { title?: unknown }).title;
            if (typeof title === 'string' && title) {
              setPositionTitleByTicker((prev) =>
                prev[ticker] ? prev : { ...prev, [ticker]: title }
              );
            }

            const time =
              (market as unknown as { close_time?: unknown }).close_time ??
              (market as unknown as { expiration_time?: unknown }).expiration_time ??
              (market as unknown as { expiry_time?: unknown }).expiry_time ??
              (market as unknown as { closeTime?: unknown }).closeTime ??
              (market as unknown as { expirationTime?: unknown }).expirationTime ??
              (market as unknown as { expiryTime?: unknown }).expiryTime ??
              null;

            if (typeof time === 'string' && time) {
              setPositionGameDateByTicker((prev) =>
                prev[ticker] ? prev : { ...prev, [ticker]: time }
              );
            }

            const yesSub = (market as unknown as { yes_sub_title?: unknown }).yes_sub_title;
            const noSub = (market as unknown as { no_sub_title?: unknown }).no_sub_title;
            const yesSub2 = (market as unknown as { yesSubTitle?: unknown }).yesSubTitle;
            const noSub2 = (market as unknown as { noSubTitle?: unknown }).noSubTitle;
            const yes =
              (typeof yesSub === 'string' && yesSub) ||
              (typeof yesSub2 === 'string' && yesSub2) ||
              undefined;
            const no =
              (typeof noSub === 'string' && noSub) ||
              (typeof noSub2 === 'string' && noSub2) ||
              undefined;
            if (yes || no) {
              setPositionOutcomeByTicker((prev) =>
                prev[ticker] ? prev : { ...prev, [ticker]: { yes, no } }
              );
            }
          }
        }
      })().catch(() => {
        // ignore title fetch errors
      });

      // Update position tracker with API positions
      for (const apiPos of apiPositions) {
        const market = marketsDataRef.current.get(apiPos.ticker);
        const yesMaybe = market?.currentPrice ?? market?.awayProb ?? market?.homeProb ?? null;
        const yes = normalizeKalshiPriceToProbability(yesMaybe);
        const currentPrice = yes === null ? null : apiPos.side === 'yes' ? yes : 1 - yes;

        const position: Position = {
          ticker: apiPos.ticker,
          side: apiPos.side,
          quantity: apiPos.position,
          averagePrice:
            normalizeKalshiPriceToProbability(apiPos.average_price) ?? apiPos.average_price,
          currentPrice: currentPrice,
          realizedPnl: apiPos.realized_pnl,
          unrealizedPnl: 0,
          totalPnl: apiPos.realized_pnl,
          pnlPercent: 0,
        };

        positionTrackerRef.current.updatePnl(position, currentPrice);
        positionTrackerRef.current.setPosition(position);
      }

      // Also load recent fills
      try {
        const recentFills = await apiClientRef.current.getFilledOrders({ limit: 100 });
        for (const fill of recentFills) {
          const market = marketsDataRef.current.get(fill.ticker);
          const currentPrice = market?.awayProb || market?.homeProb || null;

          const fillNotification: FillNotification = {
            ticker: fill.ticker,
            side: fill.side,
            action: fill.action,
            quantity: fill.count,
            price: normalizeKalshiPriceToProbability(fill.price) ?? fill.price,
            timestamp: new Date(fill.created_at).getTime(),
            orderId: fill.order_id,
          };

          positionTrackerRef.current.addFill(fillNotification, currentPrice);
        }
      } catch (error) {
        console.warn('Could not load fills:', error);
      }

      refreshPositions();
    } catch (error) {
      const msg = error instanceof Error ? error.message : String(error);
      if (
        msg.includes('API error: 401') ||
        msg.includes('"code":"authentication_error"') ||
        msg.includes('"code": "authentication_error"')
      ) {
        setPortfolioPollingEnabled(false);
        setErrorMessage(
          'Connected to market feed, but portfolio endpoints are unauthorized (401 authentication_error). ' +
            'This usually means you are using the wrong environment (prod vs demo) or your API key is not provisioned for trading/portfolio.'
        );
        return;
      }
      console.error('Failed to load positions:', error);
    }
  }, [
    apiClientRef,
    refreshPositions,
    portfolioPollingEnabled,
    positionTitleByTicker,
    positionGameDateByTicker,
    positionOutcomeByTicker,
    marketsDataRef,
    setErrorMessage,
    setPortfolioPollingEnabled,
  ]);

  // Create position price stream when connected
  useEffect(() => {
    if (!isConnected || !credentials) return;

    // Already have a stream
    if (positionPriceStreamRef.current) return;

    let cancelled = false;

    (async () => {
      try {
        const { privateKey: cryptoKey } = await initAuth(
          credentials.accessKeyId,
          credentials.privateKeyPem
        );

        if (cancelled) return;

        const ps = createMarketStream(true); // useRelay=true
        positionPriceStreamRef.current = ps;
        subscribedPositionTickersRef.current.clear();
        setPositionPriceStreamConnected(false);

        ps.onPriceUpdate((u) => {
          const m = new Map<string, number>();
          m.set(u.ticker, u.price);
          positionTrackerRef.current.updatePrices(m);
          refreshPositions();
        });
        ps.onError((e) => console.warn('Position price stream error:', e));

        await ps.connect(credentials.accessKeyId, cryptoKey, credentials.environment);
        if (!cancelled) {
          setPositionPriceStreamConnected(true);
        }
      } catch (err) {
        console.warn('Position price stream unavailable (positions current may be stale):', err);
        if (!cancelled) {
          setPositionPriceStreamConnected(false);
        }
      }
    })();

    return () => {
      cancelled = true;
    };
  }, [isConnected, credentials, refreshPositions]);

  // Keep the position-price stream subscribed to all open position tickers.
  useEffect(() => {
    const ps = positionPriceStreamRef.current;
    if (!isConnected || !positionPriceStreamConnected || !ps || !ps.isConnected()) return;

    const prev = subscribedPositionTickersRef.current;
    const next = new Set<string>();
    for (const p of positions) {
      if (p?.ticker) next.add(p.ticker);
    }

    const toAdd: string[] = [];
    next.forEach((t) => {
      if (!prev.has(t)) toAdd.push(t);
    });

    if (toAdd.length === 0) return;

    try {
      ps.subscribe(toAdd);
      ps.subscribeOrderbook(toAdd);
      toAdd.forEach((t) => prev.add(t));
    } catch (err) {
      console.warn('Failed to subscribe position tickers:', err);
    }
  }, [isConnected, positionPriceStreamConnected, positions]);

  const handleClosePosition = useCallback(
    async (ticker: string, side: 'yes' | 'no', quantity: number) => {
      if (!apiClientRef.current) return;

      try {
        await apiClientRef.current.placeOrder({
          ticker,
          side,
          action: 'sell',
          type: 'market',
          count: quantity,
        });

        await loadPositions();
      } catch (error) {
        const message = error instanceof Error ? error.message : 'Failed to close position';
        setErrorMessage(message);
      }
    },
    [apiClientRef, loadPositions, setErrorMessage]
  );

  const cleanupPositions = useCallback(() => {
    if (positionPriceStreamRef.current) {
      positionPriceStreamRef.current.disconnect();
      positionPriceStreamRef.current = null;
      subscribedPositionTickersRef.current.clear();
      setPositionPriceStreamConnected(false);
    }
    setPositions([]);
    setTotalPnl(0);
  }, []);

  return {
    positions,
    totalPnl,
    positionTitleByTicker,
    positionGameDateByTicker,
    positionOutcomeByTicker,
    positionPriceStreamConnected,
    positionTrackerRef,
    positionPriceStreamRef,
    subscribedPositionTickersRef,
    loadPositions,
    refreshPositions,
    handleClosePosition,
    cleanupPositions,
  };
}
