/**
 * Hook to fetch and attach rotation numbers from The Odds API to consolidated games.
 *
 * Uses the free /events endpoint (0 credits) with includeRotationNumbers=true.
 * Gracefully returns original games unchanged if no API key or on errors.
 */

import { useState, useEffect, useMemo } from 'react';
import type { Sport } from '@/lib/sportsDiscovery';
import type { ConsolidatedGameBooks } from '@/lib/nbaConsolidated/types';
import {
  fetchEventsForSport,
  matchOddsToGame,
  getRotationNumbers,
  SPORT_KEY_MAP,
  type OddsApiEvent,
} from '@/lib/oddsApi';

export function useRotationNumbers(
  sport: Sport,
  games: ConsolidatedGameBooks[]
): ConsolidatedGameBooks[] {
  const [events, setEvents] = useState<OddsApiEvent[]>([]);

  const sportKey = SPORT_KEY_MAP[sport] ?? null;

  useEffect(() => {
    if (!sportKey) return;

    let cancelled = false;
    fetchEventsForSport(sportKey).then((result) => {
      if (!cancelled) setEvents(result);
    });

    return () => {
      cancelled = true;
    };
  }, [sportKey]);

  return useMemo(() => {
    if (!events.length || !sportKey) return games;

    return games.map((game) => {
      const matched = matchOddsToGame(events, game.date, game.homeCode, game.awayCode);
      if (!matched) return game;

      const { home, away } = getRotationNumbers(matched);
      return { ...game, awayRotation: away, homeRotation: home };
    });
  }, [games, events, sportKey]);
}
