import { useState, useEffect, useRef, useCallback } from 'react';
import { createKalshiClient } from '../lib/kalshiApi';
import { createNbaStream, createConsolidatedNbaStream, getMarketState } from '../lib/stream';
import {
  createSportsStream,
  type SportsStream,
  type GameData,
  type SportsLoadingState,
} from '../lib/sportsStream';
import { type Sport, type MarketType } from '../lib/sportsDiscovery';
import { initAuth } from '../lib/kalshiAuth';
import { createMarketChart, type ChartManager } from '../lib/chartManager';
import { PositionTracker } from '../lib/positionTracker';
import type { DataMode } from '../lib/dataMode';
import type { NBAMarketRow, StreamUpdate, KalshiCredentials } from '../types';
import type { ConsolidatedGameBooks } from '../lib/nbaConsolidated/types';

export interface StreamManagementParams {
  isConnected: boolean;
  apiClientRef: React.MutableRefObject<Awaited<ReturnType<typeof createKalshiClient>> | null>;
  credentials: KalshiCredentials | null;
  dataMode: DataMode;
  chartPeriod: '1m' | '5m' | '15m' | '1h' | '1d';
  positionTrackerRef: React.MutableRefObject<PositionTracker>;
  refreshPositions: () => void;
  marketsDataRef: React.MutableRefObject<Map<string, NBAMarketRow>>;
  setIsConnected: React.Dispatch<React.SetStateAction<boolean>>;
  setConnectionStatus: React.Dispatch<React.SetStateAction<string>>;
}

export interface StreamManagementReturn {
  markets: NBAMarketRow[];
  consolidatedGames: ConsolidatedGameBooks[];
  consolidatedLoading: boolean;
  sportsGames: Map<string, GameData[]>;
  sportsLoading: Map<string, boolean>;
  sportsLoadingStates: Map<string, SportsLoadingState>;
  feedStatus: 'live' | 'offline';
  lastUpdateTime: string;
  handleSportMarketChange: (sport: Sport, marketType: MarketType) => Promise<void>;
  cleanupStreams: () => void;
}

export function useStreamManagement({
  isConnected,
  apiClientRef,
  credentials,
  dataMode,
  chartPeriod,
  positionTrackerRef,
  refreshPositions,
  marketsDataRef,
  setIsConnected,
  setConnectionStatus,
}: StreamManagementParams): StreamManagementReturn {
  const [markets, setMarkets] = useState<NBAMarketRow[]>([]);
  const [consolidatedGames, setConsolidatedGames] = useState<ConsolidatedGameBooks[]>([]);
  const [consolidatedLoading, setConsolidatedLoading] = useState(false);
  const [sportsGames, setSportsGames] = useState<Map<string, GameData[]>>(new Map());
  const [sportsLoading, setSportsLoading] = useState<Map<string, boolean>>(new Map());
  const [sportsLoadingStates, setSportsLoadingStates] = useState<Map<string, SportsLoadingState>>(
    new Map()
  );
  const [feedStatus, setFeedStatus] = useState<'live' | 'offline'>('offline');
  const [lastUpdateTime, setLastUpdateTime] = useState('--:--:--');

  const streamRef = useRef<ReturnType<typeof createNbaStream> | null>(null);
  const consolidatedStreamRef = useRef<ReturnType<typeof createConsolidatedNbaStream> | null>(null);
  const marketChartsRef = useRef<Map<string, ChartManager>>(new Map());
  const marketChartsContainerRef = useRef<HTMLDivElement>(null);
  const sportsStreamRef = useRef<SportsStream | null>(null);
  const sportsStreamKeyRef = useRef<string>('');

  // Update market charts
  const updateMarketCharts = useCallback(() => {
    const marketState = getMarketState();
    if (!marketState || !marketChartsContainerRef.current) return;

    markets.forEach((market) => {
      let chart = marketChartsRef.current.get(market.market_ticker);
      if (!chart) {
        const container = document.createElement('div');
        container.className = 'market-chart-item';
        marketChartsContainerRef.current!.appendChild(container);
        chart = createMarketChart(container, true);
        chart.init();
        marketChartsRef.current.set(market.market_ticker, chart);
      }

      if (chart) {
        const candles = marketState.getCandles(market.market_ticker, chartPeriod, 100);
        if (candles.length > 0) {
          chart.updateData(candles);
        }
      }
    });
  }, [markets, chartPeriod]);

  // Handle stream updates
  const handleStreamUpdate = useCallback(
    (update: StreamUpdate) => {
      if (update.type === 'snapshot') {
        setMarkets(update.rows || []);
        const dataMap = new Map<string, NBAMarketRow>();
        (update.rows || []).forEach((row) => {
          dataMap.set(row.market_ticker, row);
        });
        marketsDataRef.current = dataMap;
        refreshPositions();
        updateMarketCharts();
      } else if (update.type === 'update') {
        setMarkets((prev) => {
          const updated = new Map(prev.map((m) => [m.market_ticker, m]));
          (update.rows || []).forEach((row) => {
            updated.set(row.market_ticker, row);
          });
          return Array.from(updated.values());
        });
      } else if (update.type === 'price_update' && update.priceUpdate) {
        setMarkets((prev) =>
          prev.map((m) => {
            if (m.market_ticker === update.priceUpdate!.ticker) {
              const updated = { ...m, currentPrice: update.priceUpdate!.price };
              marketsDataRef.current.set(m.market_ticker, updated);
              return updated;
            }
            return m;
          })
        );

        const priceMap = new Map<string, number>();
        priceMap.set(update.priceUpdate.ticker, update.priceUpdate.price);
        positionTrackerRef.current.updatePrices(priceMap);
        refreshPositions();
        updateMarketCharts();
      }

      setIsConnected(true);
      setConnectionStatus('Connected');
      setFeedStatus('live');

      const now = new Date();
      setLastUpdateTime(
        now.toLocaleTimeString('en-US', {
          hour12: false,
          hour: '2-digit',
          minute: '2-digit',
          second: '2-digit',
        })
      );
    },
    [
      refreshPositions,
      updateMarketCharts,
      marketsDataRef,
      positionTrackerRef,
      setIsConnected,
      setConnectionStatus,
    ]
  );

  // Init streams when connected
  useEffect(() => {
    if (!isConnected || !credentials || !apiClientRef.current) return;

    // Already have streams
    if (streamRef.current || consolidatedStreamRef.current) return;

    let cancelled = false;
    const apiClient = apiClientRef.current;

    (async () => {
      try {
        const { privateKey: cryptoKey } = await initAuth(
          credentials.accessKeyId,
          credentials.privateKeyPem
        );

        if (cancelled) return;

        // Create NBA stream
        const stream = createNbaStream({
          api: apiClient,
          accessKeyId: credentials.accessKeyId,
          privateKey: cryptoKey,
          environment: credentials.environment,
          useRelay: true,
          useMock: false,
        });

        streamRef.current = stream;
        stream.onUpdate((update: StreamUpdate) => {
          handleStreamUpdate(update);
        });

        try {
          stream.start();
        } catch (err) {
          console.error('Stream start failed:', err);
        }

        // Create consolidated stream
        try {
          setConsolidatedLoading(true);
          const consolidated = createConsolidatedNbaStream({
            api: apiClient,
            accessKeyId: credentials.accessKeyId,
            privateKey: cryptoKey,
            environment: credentials.environment,
            useRelay: true,
            useMock: false,
            dataMode,
          });
          consolidatedStreamRef.current = consolidated;
          consolidated.onUpdate(({ games, lastUpdateMs }) => {
            setConsolidatedLoading(false);
            setConsolidatedGames(games);
            setFeedStatus('live');
            const now = new Date(lastUpdateMs || Date.now());
            setLastUpdateTime(
              now.toLocaleTimeString('en-US', {
                hour12: false,
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
              })
            );
          });
          consolidated.start();
        } catch (err) {
          console.warn('Consolidated NBA stream unavailable:', err);
          setConsolidatedGames([]);
          setConsolidatedLoading(false);
        }
      } catch (err) {
        console.error('Stream auth init failed:', err);
      }
    })();

    return () => {
      cancelled = true;
    };
  }, [isConnected, credentials, apiClientRef, dataMode, handleStreamUpdate]);

  // Update charts when period or markets change
  useEffect(() => {
    if (isConnected) {
      updateMarketCharts();
    }
  }, [chartPeriod, markets, isConnected, updateMarketCharts]);

  // Handler for sport/market type changes
  const handleSportMarketChange = useCallback(
    async (sport: Sport, marketType: MarketType) => {
      if (sport === 'nba' && marketType === 'moneyline') return;

      const streamKey = `${sport}-${marketType}`;
      if (sportsStreamKeyRef.current === streamKey && sportsStreamRef.current) return;

      if (sportsStreamRef.current) {
        sportsStreamRef.current.stop();
        sportsStreamRef.current = null;
      }

      if (!apiClientRef.current || !credentials) return;

      setSportsLoading((prev) => new Map(prev).set(streamKey, true));
      setSportsLoadingStates((prev) => {
        const next = new Map(prev);
        const now = Date.now();
        next.set(streamKey, {
          isLoading: true,
          phase: 'discovering-markets',
          startedAtMs: now,
          updatedAtMs: now,
          events: [
            {
              tsMs: now,
              phase: 'discovering-markets',
              level: 'info',
              message: `Starting ${sport.toUpperCase()} ${marketType} stream`,
            },
          ],
        });
        return next;
      });

      try {
        const { privateKey: cryptoKey } = await initAuth(
          credentials.accessKeyId,
          credentials.privateKeyPem
        );

        const stream = createSportsStream(
          {
            api: apiClientRef.current,
            accessKeyId: credentials.accessKeyId,
            privateKey: cryptoKey,
            environment: credentials.environment,
            useRelay: true,
            useMock: false,
          },
          {
            sport,
            marketType,
            daysAhead: 14,
          }
        );

        sportsStreamRef.current = stream;
        sportsStreamKeyRef.current = streamKey;

        stream.onUpdate((update) => {
          setSportsGames((prev) => {
            const next = new Map(prev);
            next.set(streamKey, update.games);
            return next;
          });
          setSportsLoadingStates((prev) => {
            const next = new Map(prev);
            next.set(streamKey, update.loadingState);
            return next;
          });
          setSportsLoading((prev) => {
            const next = new Map(prev);
            next.set(streamKey, update.loadingState.isLoading);
            return next;
          });
        });

        await stream.start();
      } catch (err) {
        console.error(`Failed to start ${streamKey} stream:`, err);
        setSportsLoadingStates((prev) => {
          const next = new Map(prev);
          const now = Date.now();
          const message = err instanceof Error ? err.message : String(err);
          next.set(streamKey, {
            isLoading: false,
            phase: 'error',
            startedAtMs: now,
            updatedAtMs: now,
            events: [
              {
                tsMs: now,
                phase: 'error',
                level: 'error',
                message,
              },
            ],
          });
          return next;
        });
        setSportsLoading((prev) => {
          const next = new Map(prev);
          next.set(streamKey, false);
          return next;
        });
      }
    },
    [credentials, apiClientRef]
  );

  const cleanupStreams = useCallback(() => {
    if (streamRef.current) {
      streamRef.current.stop();
      streamRef.current = null;
    }
    if (consolidatedStreamRef.current) {
      consolidatedStreamRef.current.stop();
      consolidatedStreamRef.current = null;
    }
    if (sportsStreamRef.current) {
      sportsStreamRef.current.stop();
      sportsStreamRef.current = null;
    }
    sportsStreamKeyRef.current = '';

    // Clean up charts
    marketChartsRef.current.forEach((chart) => chart.destroy());
    marketChartsRef.current.clear();

    setMarkets([]);
    setConsolidatedGames([]);
    setConsolidatedLoading(false);
    setSportsGames(new Map());
    setSportsLoading(new Map());
    setSportsLoadingStates(new Map());
    setFeedStatus('offline');
    setLastUpdateTime('--:--:--');
  }, []);

  return {
    markets,
    consolidatedGames,
    consolidatedLoading,
    sportsGames,
    sportsLoading,
    sportsLoadingStates,
    feedStatus,
    lastUpdateTime,
    handleSportMarketChange,
    cleanupStreams,
  };
}
