/**
 * Shared sorting hook for sports table components.
 *
 * Cycles through: ascending → descending → back to default sort.
 *
 * Supports custom value extractors for sorting by arbitrary columns
 * (e.g. odds, liquidity, spread lines). Null values always sort to the bottom.
 */

import { useState, useMemo, useCallback } from 'react';
import type { SortDirection } from '@/components/atoms/SortableTh';

interface Sortable {
  date: string;
  startTimePt?: string | null;
  awayCode: string;
  homeCode: string;
  awayName?: string;
  homeName?: string;
}

export type ValueExtractor<T> = (item: T) => number | string | null;

export interface TableSortOptions {
  /** Sort key to use on initial render and when user cycles back to "default" */
  defaultSortKey?: string;
  /** Sort direction for the default sort */
  defaultSortDir?: SortDirection;
}

export function useTableSort<T extends Sortable>(
  games: T[],
  extractors?: Record<string, ValueExtractor<T>>,
  options?: TableSortOptions
) {
  const defaultSortKey = options?.defaultSortKey ?? null;
  const defaultSortDir = options?.defaultSortDir ?? null;

  const [sortKey, setSortKey] = useState<string | null>(defaultSortKey);
  const [sortDir, setSortDir] = useState<SortDirection>(defaultSortDir);

  const handleSort = useCallback(
    (key: string) => {
      if (sortKey === key) {
        // Cycle: asc → desc → back to default
        if (sortDir === 'asc') setSortDir('desc');
        else if (sortDir === 'desc') {
          setSortKey(defaultSortKey);
          setSortDir(defaultSortDir);
        }
      } else {
        setSortKey(key);
        setSortDir('asc');
      }
    },
    [sortKey, sortDir, defaultSortKey, defaultSortDir]
  );

  const sorted = useMemo(() => {
    if (!sortKey || !sortDir) return games;

    return [...games].sort((a, b) => {
      let cmp = 0;

      // Check custom extractors first
      const extractor = extractors?.[sortKey];
      if (extractor) {
        const aVal = extractor(a);
        const bVal = extractor(b);
        // Null values sort to bottom regardless of direction
        if (aVal === null && bVal === null) return 0;
        if (aVal === null) return 1;
        if (bVal === null) return -1;
        if (typeof aVal === 'string' && typeof bVal === 'string') {
          cmp = aVal.localeCompare(bVal);
        } else {
          cmp = (aVal as number) - (bVal as number);
        }
        return sortDir === 'desc' ? -cmp : cmp;
      }

      switch (sortKey) {
        case 'date':
          cmp = a.date.localeCompare(b.date);
          break;
        case 'time':
          // Composite: date first (chronological), then time within each date
          cmp = a.date.localeCompare(b.date);
          if (cmp === 0) {
            cmp = (a.startTimePt ?? '').localeCompare(b.startTimePt ?? '');
          }
          break;
        case 'team': {
          const aName = a.awayName ?? a.awayCode;
          const bName = b.awayName ?? b.awayCode;
          cmp = aName.localeCompare(bName);
          break;
        }
        default:
          cmp = 0;
      }

      return sortDir === 'desc' ? -cmp : cmp;
    });
  }, [games, sortKey, sortDir, extractors]);

  return { sorted, sortKey, sortDir, handleSort };
}
