/**
 * useVersionCheck - Polls /version.json to detect new deploys.
 *
 * When the deployed version changes while the tab is open, sets `updateAvailable`
 * to true so the UI can prompt a reload.
 */

import { useState, useEffect, useRef, useCallback } from 'react';

const POLL_INTERVAL_MS = 60_000; // 1 minute
const VERSION_URL = '/version.json';

interface VersionInfo {
  version: string;
  deployedAt: string;
}

export function useVersionCheck() {
  const [updateAvailable, setUpdateAvailable] = useState(false);
  const initialVersion = useRef<string | null>(null);

  const checkVersion = useCallback(async () => {
    try {
      const resp = await fetch(VERSION_URL, { cache: 'no-store' });
      if (!resp.ok) return;
      const data = (await resp.json()) as VersionInfo;
      if (!data.version) return;

      if (initialVersion.current === null) {
        // First check — record the baseline
        initialVersion.current = data.version;
      } else if (data.version !== initialVersion.current) {
        setUpdateAvailable(true);
      }
    } catch {
      // Network error — skip silently
    }
  }, []);

  useEffect(() => {
    // Check immediately on mount
    checkVersion();

    const interval = setInterval(checkVersion, POLL_INTERVAL_MS);
    return () => clearInterval(interval);
  }, [checkVersion]);

  const reload = useCallback(() => {
    window.location.reload();
  }, []);

  return { updateAvailable, reload };
}
