/**
 * Candlestick Aggregator
 *
 * Aggregates candlesticks from multiple markets into portfolio-level candlesticks.
 */

import type { Candlestick, CandlestickPeriod, Portfolio } from '../types';

export class CandlestickAggregator {
  /**
   * Aggregate portfolio candlesticks (weighted average)
   */
  aggregatePortfolio(
    portfolio: Portfolio,
    period: CandlestickPeriod,
    marketCandles: Map<string, Candlestick[]>
  ): Candlestick[] {
    if (portfolio.markets.length === 0) {
      return [];
    }

    // Get all unique timestamps across all markets
    const timestamps = new Set<number>();
    for (const market of portfolio.markets) {
      const candles = marketCandles.get(market.ticker) || [];
      for (const candle of candles) {
        if (candle.period === period) {
          timestamps.add(candle.timestamp);
        }
      }
    }

    if (timestamps.size === 0) {
      return [];
    }

    // Aggregate candles at each timestamp
    const portfolioCandles: Candlestick[] = [];
    const sortedTimestamps = Array.from(timestamps).sort((a, b) => a - b);

    for (const timestamp of sortedTimestamps) {
      const candlesAtTime: Array<{ candle: Candlestick; weight: number }> = [];

      // Collect candles from all markets at this timestamp
      for (const market of portfolio.markets) {
        const candles = marketCandles.get(market.ticker) || [];
        const candle = candles.find((c) => c.timestamp === timestamp && c.period === period);

        if (candle) {
          const weight = market.weight || 1 / portfolio.markets.length;
          candlesAtTime.push({ candle, weight });
        }
      }

      if (candlesAtTime.length === 0) continue;

      // Calculate weighted average OHLC
      let totalWeight = 0;
      let open = 0;
      let high = 0;
      let low = Infinity;
      let close = 0;
      let volume = 0;

      for (const { candle, weight } of candlesAtTime) {
        totalWeight += weight;
        open += candle.open * weight;
        high = Math.max(high, candle.high);
        low = Math.min(low, candle.low);
        close += candle.close * weight;
        volume += candle.volume;
      }

      if (totalWeight > 0) {
        portfolioCandles.push({
          open: open / totalWeight,
          high,
          low: low === Infinity ? 0 : low,
          close: close / totalWeight,
          volume,
          timestamp,
          period,
          ticker: portfolio.id, // Use portfolio ID as ticker
        });
      }
    }

    return portfolioCandles;
  }

  /**
   * Update an existing candle with a new tick
   * (For incremental updates - more efficient than full aggregation)
   */
  updateCandle(candle: Candlestick, price: number, volume: number = 0): Candlestick {
    return {
      ...candle,
      high: Math.max(candle.high, price),
      low: Math.min(candle.low, price),
      close: price,
      volume: candle.volume + volume,
    };
  }
}
