/**
 * Data Mode — localStorage-backed setting for data source selection.
 *
 * 'proxy'        — Current behavior: dashboard makes individual REST/WS calls via relay proxy
 * 'smart-relay'  — New: relay caches all public data server-side, streams to dashboard
 */

export type DataMode = 'proxy' | 'smart-relay';

const STORAGE_KEY = 'data_mode';
let currentMode: DataMode = 'proxy';

/** Initialize from localStorage. Call on mount. */
export function initDataMode(): DataMode {
  const saved = localStorage.getItem(STORAGE_KEY);
  if (saved === 'proxy' || saved === 'smart-relay') {
    currentMode = saved;
  }
  return currentMode;
}

/** Get current data mode. */
export function getDataMode(): DataMode {
  return currentMode;
}

/** Set data mode and persist to localStorage. */
export function setDataMode(mode: DataMode): void {
  currentMode = mode;
  localStorage.setItem(STORAGE_KEY, mode);
}

/** Toggle between proxy and smart-relay. */
export function toggleDataMode(): DataMode {
  const next = currentMode === 'proxy' ? 'smart-relay' : 'proxy';
  setDataMode(next);
  return next;
}
