import { describe, expect, it } from 'vitest';
import { getTodayStr, findNearestAvailableDate } from './dateUtils';

describe('getTodayStr', () => {
  it('returns YYYY-MM-DD format', () => {
    const result = getTodayStr();
    expect(result).toMatch(/^\d{4}-\d{2}-\d{2}$/);
  });

  it('matches current local date components', () => {
    const now = new Date();
    const result = getTodayStr();
    const [year, month, day] = result.split('-').map(Number);
    expect(year).toBe(now.getFullYear());
    expect(month).toBe(now.getMonth() + 1);
    expect(day).toBe(now.getDate());
  });
});

describe('findNearestAvailableDate', () => {
  const today = '2026-02-19';

  it('returns null when availableDates is empty', () => {
    expect(findNearestAvailableDate('2026-02-19', [], today)).toBeNull();
  });

  it('returns null when selectedDate is already in availableDates', () => {
    expect(
      findNearestAvailableDate('2026-02-19', ['2026-02-18', '2026-02-19', '2026-02-20'], today)
    ).toBeNull();
  });

  it('snaps to nearest future date when today is available', () => {
    const available = ['2026-02-18', '2026-02-20', '2026-02-21'];
    // selectedDate is 2026-02-19 (not in list), today is 2026-02-19 (also not in list)
    // futureOrToday = [2026-02-20, 2026-02-21]
    expect(findNearestAvailableDate('2026-02-19', available, today)).toBe('2026-02-20');
  });

  it('prefers today when today is in the list', () => {
    const available = ['2026-02-17', '2026-02-19', '2026-02-22'];
    // selectedDate is 2026-02-16 (not in list), today=2026-02-19 is in list
    // futureOrToday = [2026-02-19, 2026-02-22] → picks first = 2026-02-19
    expect(findNearestAvailableDate('2026-02-16', available, today)).toBe('2026-02-19');
  });

  it('snaps to most recent past when all dates are past', () => {
    const available = ['2026-02-10', '2026-02-15', '2026-02-17'];
    // selectedDate is 2026-02-19 (not in list), today=2026-02-19
    // futureOrToday = [] → fallback to last = 2026-02-17
    expect(findNearestAvailableDate('2026-02-19', available, today)).toBe('2026-02-17');
  });

  it('snaps when selectedDate is before all available dates', () => {
    const available = ['2026-02-20', '2026-02-21', '2026-02-22'];
    // selectedDate=2026-02-10, today=2026-02-19
    // futureOrToday = [2026-02-20, 2026-02-21, 2026-02-22] → picks 2026-02-20
    expect(findNearestAvailableDate('2026-02-10', available, today)).toBe('2026-02-20');
  });

  it('snaps when selectedDate is after all available dates', () => {
    const available = ['2026-02-10', '2026-02-11'];
    // selectedDate=2026-02-25, today=2026-02-19
    // futureOrToday = [] → fallback to last = 2026-02-11
    expect(findNearestAvailableDate('2026-02-25', available, today)).toBe('2026-02-11');
  });

  it('handles single available date that matches today', () => {
    expect(findNearestAvailableDate('2026-02-18', ['2026-02-19'], today)).toBe('2026-02-19');
  });

  it('handles single available date in the past', () => {
    expect(findNearestAvailableDate('2026-02-19', ['2026-02-10'], today)).toBe('2026-02-10');
  });

  it('uses real today when no today parameter provided', () => {
    // Without explicit today, uses getTodayStr() internally
    const result = findNearestAvailableDate('9999-12-31', ['2026-02-10']);
    // All dates are past relative to 9999, but the function checks vs getTodayStr()
    // which is today. If today > 2026-02-10, result should be '2026-02-10' (most recent past).
    expect(result).toBe('2026-02-10');
  });
});
