/**
 * Diagnostic Check Implementations
 *
 * Each check is a pure function: (context) => Promise<DiagnosticResult>
 */

import type {
  DiagnosticCheck,
  DiagnosticResult,
  DiagnosticContext,
  DiagnosticStatus,
} from './types';
import { getTodayStr, findNearestAvailableDate } from '../dateUtils';

/** Registry of all diagnostic checks */
export const DIAGNOSTIC_CHECKS: DiagnosticCheck[] = [
  // Connectivity
  {
    id: 'relay-http',
    category: 'connectivity',
    name: 'Relay HTTP',
    description: 'POST to relay server and check for a 2xx response',
  },
  {
    id: 'kalshi-rest',
    category: 'connectivity',
    name: 'Kalshi REST API',
    description: 'Call Kalshi exchange status endpoint',
  },
  {
    id: 'poly-gamma',
    category: 'connectivity',
    name: 'Polymarket Gamma',
    description: 'GET Gamma API with limit=1',
  },
  {
    id: 'odds-api',
    category: 'connectivity',
    name: 'Odds API',
    description: 'Check if Odds API key is configured',
  },
  // Discovery
  {
    id: 'kalshi-markets',
    category: 'discovery',
    name: 'Kalshi Markets',
    description: 'Check if NBA markets are discovered and loaded',
  },
  {
    id: 'poly-slug-resolve',
    category: 'discovery',
    name: 'Polymarket Slugs',
    description: 'Check if any Polymarket games are resolved',
  },
  // Stream Health
  {
    id: 'consolidated-status',
    category: 'stream',
    name: 'Consolidated Stream',
    description: 'Check consolidated NBA stream status and last update age',
  },
  {
    id: 'ws-connected',
    category: 'stream',
    name: 'WebSocket Feed',
    description: 'Check WebSocket connection state',
  },
  // Data
  {
    id: 'data-freshness',
    category: 'data',
    name: 'Data Freshness',
    description: 'Seconds since last data update (warn if >60s)',
  },
  {
    id: 'game-count',
    category: 'data',
    name: 'Game Count',
    description: 'Count of loaded games, breakdown by sport',
  },
  // Date
  {
    id: 'available-dates',
    category: 'date',
    name: 'Available Dates',
    description: 'List dates that have games, flag if today has none',
  },
  {
    id: 'autosnap-test',
    category: 'date',
    name: 'Auto-snap Test',
    description: 'Run snap logic against current data and report what would happen',
  },
];

function result(
  checkId: string,
  status: DiagnosticStatus,
  message: string,
  startMs: number,
  details?: Record<string, unknown>
): DiagnosticResult {
  return {
    checkId,
    status,
    message,
    durationMs: Date.now() - startMs,
    timestamp: Date.now(),
    details,
  };
}

/** Run a single check by ID */
export async function runCheck(checkId: string, ctx: DiagnosticContext): Promise<DiagnosticResult> {
  const startMs = Date.now();

  try {
    switch (checkId) {
      case 'relay-http':
        return await checkRelayHttp(startMs);
      case 'kalshi-rest':
        return await checkKalshiRest(ctx, startMs);
      case 'poly-gamma':
        return await checkPolyGamma(startMs);
      case 'odds-api':
        return checkOddsApi(startMs);
      case 'kalshi-markets':
        return checkKalshiMarkets(ctx, startMs);
      case 'poly-slug-resolve':
        return checkPolySlugResolve(ctx, startMs);
      case 'consolidated-status':
        return checkConsolidatedStatus(ctx, startMs);
      case 'ws-connected':
        return checkWsConnected(ctx, startMs);
      case 'data-freshness':
        return checkDataFreshness(ctx, startMs);
      case 'game-count':
        return checkGameCount(ctx, startMs);
      case 'available-dates':
        return checkAvailableDates(ctx, startMs);
      case 'autosnap-test':
        return checkAutosnapTest(ctx, startMs);
      default:
        return result(checkId, 'skip', `Unknown check: ${checkId}`, startMs);
    }
  } catch (err) {
    const message = err instanceof Error ? err.message : String(err);
    return result(checkId, 'fail', `Exception: ${message}`, startMs);
  }
}

// === Check implementations ===

async function checkRelayHttp(startMs: number): Promise<DiagnosticResult> {
  try {
    const relayUrl =
      (import.meta.env.VITE_RELAY_HTTP_URL as string | undefined)?.trim() || '/relay/http';
    const resp = await fetch(relayUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        method: 'GET',
        url: 'https://api.elections.kalshi.com/v1',
        headers: {},
      }),
    });
    if (resp.ok) {
      return result('relay-http', 'pass', `HTTP ${resp.status}`, startMs);
    }
    return result('relay-http', 'warn', `HTTP ${resp.status} ${resp.statusText}`, startMs);
  } catch (err) {
    const message = err instanceof Error ? err.message : String(err);
    return result('relay-http', 'fail', message, startMs);
  }
}

async function checkKalshiRest(ctx: DiagnosticContext, startMs: number): Promise<DiagnosticResult> {
  if (!ctx.apiClient) {
    return result('kalshi-rest', 'skip', 'Not connected', startMs);
  }
  try {
    // Use a lightweight call — fetch series with limit 1 to verify API connectivity
    // Race against a 10s timeout to avoid indefinite hangs
    const seriesPromise = ctx.apiClient.getSeries({ include_volume: false });
    const timeoutPromise = new Promise<never>((_, reject) =>
      setTimeout(() => reject(new Error('Timeout after 10s')), 10_000)
    );
    const series = await Promise.race([seriesPromise, timeoutPromise]);
    return result('kalshi-rest', 'pass', `Connected (${series.length} series)`, startMs, {
      seriesCount: series.length,
    });
  } catch (err) {
    const message = err instanceof Error ? err.message : String(err);
    return result('kalshi-rest', 'fail', message, startMs);
  }
}

async function checkPolyGamma(startMs: number): Promise<DiagnosticResult> {
  const controller = new AbortController();
  const timer = setTimeout(() => controller.abort(), 5000);
  try {
    const resp = await fetch('https://gamma-api.polymarket.com/events?limit=1&active=true', {
      signal: controller.signal,
    });
    if (resp.ok) {
      const data = (await resp.json()) as unknown[];
      return result('poly-gamma', 'pass', `OK (${data.length} event)`, startMs);
    }
    return result('poly-gamma', 'warn', `HTTP ${resp.status}`, startMs);
  } catch (err) {
    const message = err instanceof Error ? err.message : String(err);
    return result('poly-gamma', 'fail', message, startMs);
  } finally {
    clearTimeout(timer);
  }
}

function checkOddsApi(startMs: number): DiagnosticResult {
  const key = localStorage.getItem('galactus_odds_api_key');
  if (key && key.trim().length > 0) {
    return result('odds-api', 'pass', 'Key configured', startMs);
  }
  return result('odds-api', 'warn', 'Not configured', startMs);
}

function checkKalshiMarkets(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const count = ctx.consolidatedGames.length;
  if (count > 0) {
    return result('kalshi-markets', 'pass', `${count} games loaded`, startMs, { count });
  }
  return result('kalshi-markets', 'warn', 'No games loaded', startMs, { count: 0 });
}

function checkPolySlugResolve(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const withPoly = ctx.consolidatedGames.filter((g) => g.polymarket !== null);
  const total = ctx.consolidatedGames.length;
  if (withPoly.length > 0) {
    return result(
      'poly-slug-resolve',
      'pass',
      `${withPoly.length}/${total} games have Polymarket data`,
      startMs,
      { withPoly: withPoly.length, total }
    );
  }
  if (total === 0) {
    return result('poly-slug-resolve', 'skip', 'No games to check', startMs);
  }
  return result('poly-slug-resolve', 'warn', 'No Polymarket matches found', startMs, { total });
}

function checkConsolidatedStatus(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const count = ctx.consolidatedGames.length;
  if (count > 0) {
    return result('consolidated-status', 'pass', `${count} games in consolidated stream`, startMs);
  }
  return result('consolidated-status', 'warn', 'Consolidated stream empty', startMs);
}

function checkWsConnected(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  if (ctx.feedStatus === 'live') {
    return result('ws-connected', 'pass', 'Feed is live', startMs);
  }
  return result('ws-connected', 'fail', 'Feed is offline', startMs);
}

function checkDataFreshness(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  if (ctx.lastUpdateTime === '--:--:--') {
    return result('data-freshness', 'warn', 'No updates received yet', startMs);
  }
  // lastUpdateTime is a display string like "12:34:56" — we can report it but can't
  // compute exact age without a numeric timestamp. Report the display value.
  return result('data-freshness', 'pass', `Last update: ${ctx.lastUpdateTime}`, startMs, {
    lastUpdateTime: ctx.lastUpdateTime,
  });
}

function checkGameCount(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const consolidated = ctx.consolidatedGames.length;
  const sportCounts: Record<string, number> = {};
  for (const [key, games] of ctx.sportsGames) {
    sportCounts[key] = games.length;
  }
  const totalSports = Object.values(sportCounts).reduce((a, b) => a + b, 0);
  const total = consolidated + totalSports;

  if (total > 0) {
    return result('game-count', 'pass', `${total} total games`, startMs, {
      consolidated,
      sportCounts,
      total,
    });
  }
  return result('game-count', 'warn', 'No games loaded', startMs, { consolidated, sportCounts });
}

function collectAllDates(ctx: DiagnosticContext): string[] {
  const dateSet = new Set<string>();
  for (const g of ctx.consolidatedGames) {
    if (g.date) dateSet.add(g.date);
  }
  for (const [, games] of ctx.sportsGames) {
    for (const g of games) {
      if (g.date) dateSet.add(g.date);
    }
  }
  return Array.from(dateSet).sort();
}

function checkAvailableDates(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const dates = collectAllDates(ctx);
  const today = getTodayStr();
  const hasToday = dates.includes(today);

  if (dates.length === 0) {
    return result('available-dates', 'warn', 'No dates found', startMs);
  }

  const status = hasToday ? 'pass' : 'warn';
  const msg = hasToday
    ? `${dates.length} dates available, today included`
    : `${dates.length} dates available, today (${today}) NOT included`;
  return result('available-dates', status, msg, startMs, { dates, today, hasToday });
}

function checkAutosnapTest(ctx: DiagnosticContext, startMs: number): DiagnosticResult {
  const availableDates = collectAllDates(ctx);
  const today = getTodayStr();

  if (availableDates.length === 0) {
    return result('autosnap-test', 'warn', 'No dates available to snap to', startMs, { today });
  }

  const snapResult = findNearestAvailableDate(today, availableDates, today);

  if (snapResult === null) {
    return result('autosnap-test', 'pass', 'Today has games, no snap needed', startMs, {
      today,
      availableDates,
    });
  }
  return result('autosnap-test', 'warn', `Would snap from ${today} to ${snapResult}`, startMs, {
    today,
    snapTo: snapResult,
    availableDates,
  });
}
