/**
 * Hotkey System Types
 *
 * Defines types for customizable keyboard shortcuts with modifier key support.
 */

/** Modifier keys that can be combined with a key */
export interface HotkeyModifiers {
  ctrl?: boolean;
  shift?: boolean;
  alt?: boolean;
  meta?: boolean; // Cmd on Mac, Win on Windows
}

/** A hotkey configuration (key + modifiers) */
export interface HotkeyConfig extends HotkeyModifiers {
  key: string; // The main key (lowercase), e.g., 'n', 'k', '1', 'enter'
}

/** A hotkey mapping with metadata */
export interface HotkeyMapping {
  id: string;
  label: string;
  category: string;
  defaultConfig: HotkeyConfig;
  currentConfig: HotkeyConfig;
}

/** Hotkey categories for UI grouping */
export type HotkeyCategory = 'Sport' | 'Market Type' | 'Trading';

/** All hotkey IDs */
export type HotkeyId =
  | 'sport-nba'
  | 'sport-nfl'
  | 'sport-nhl'
  | 'sport-cbb'
  | 'sport-atp'
  | 'sport-wta'
  | 'market-moneyline'
  | 'market-spread'
  | 'market-total'
  | 'new-order'
  | 'kill-all';

/** Storage key for localStorage */
export const HOTKEY_STORAGE_KEY = 'galactus_hotkeys';

/** Default hotkey mappings */
export const DEFAULT_HOTKEYS: Record<HotkeyId, Omit<HotkeyMapping, 'currentConfig'>> = {
  'sport-nba': {
    id: 'sport-nba',
    label: 'NBA',
    category: 'Sport',
    defaultConfig: { key: '1' },
  },
  'sport-nfl': {
    id: 'sport-nfl',
    label: 'NFL',
    category: 'Sport',
    defaultConfig: { key: '2' },
  },
  'sport-nhl': {
    id: 'sport-nhl',
    label: 'NHL',
    category: 'Sport',
    defaultConfig: { key: '3' },
  },
  'sport-cbb': {
    id: 'sport-cbb',
    label: 'CBB',
    category: 'Sport',
    defaultConfig: { key: '4' },
  },
  'sport-atp': {
    id: 'sport-atp',
    label: 'ATP',
    category: 'Sport',
    defaultConfig: { key: '5' },
  },
  'sport-wta': {
    id: 'sport-wta',
    label: 'WTA',
    category: 'Sport',
    defaultConfig: { key: '6' },
  },
  'market-moneyline': {
    id: 'market-moneyline',
    label: 'Moneyline',
    category: 'Market Type',
    defaultConfig: { key: 'w', shift: true },
  },
  'market-spread': {
    id: 'market-spread',
    label: 'Spread',
    category: 'Market Type',
    defaultConfig: { key: 's', shift: true },
  },
  'market-total': {
    id: 'market-total',
    label: 'Total',
    category: 'Market Type',
    defaultConfig: { key: 'd', shift: true },
  },
  'new-order': {
    id: 'new-order',
    label: 'New Order',
    category: 'Trading',
    defaultConfig: { key: 'n' },
  },
  'kill-all': {
    id: 'kill-all',
    label: 'Kill All Orders',
    category: 'Trading',
    defaultConfig: { key: 'k' },
  },
};

/**
 * Format a hotkey config for display (e.g., "Ctrl+Shift+N")
 */
export function formatHotkey(config: HotkeyConfig): string {
  const parts: string[] = [];

  if (config.ctrl) parts.push('Ctrl');
  if (config.alt) parts.push('Alt');
  if (config.shift) parts.push('Shift');
  if (config.meta) parts.push('Cmd');

  // Format the key nicely
  let keyDisplay = config.key.toUpperCase();
  if (config.key === ' ') keyDisplay = 'Space';
  if (config.key === 'arrowup') keyDisplay = '↑';
  if (config.key === 'arrowdown') keyDisplay = '↓';
  if (config.key === 'arrowleft') keyDisplay = '←';
  if (config.key === 'arrowright') keyDisplay = '→';
  if (config.key === 'enter') keyDisplay = 'Enter';
  if (config.key === 'escape') keyDisplay = 'Esc';
  if (config.key === 'backspace') keyDisplay = 'Backspace';
  if (config.key === 'delete') keyDisplay = 'Delete';
  if (config.key === 'tab') keyDisplay = 'Tab';

  parts.push(keyDisplay);

  return parts.join('+');
}

/**
 * Parse a keyboard event into a HotkeyConfig
 */
export function eventToHotkeyConfig(e: KeyboardEvent): HotkeyConfig {
  return {
    key: e.key.toLowerCase(),
    ctrl: e.ctrlKey,
    shift: e.shiftKey,
    alt: e.altKey,
    meta: e.metaKey,
  };
}

/**
 * Check if a keyboard event matches a hotkey config
 */
export function matchesHotkey(e: KeyboardEvent, config: HotkeyConfig): boolean {
  const eventKey = e.key.toLowerCase();
  const configKey = config.key.toLowerCase();

  // Key must match
  if (eventKey !== configKey) return false;

  // Modifiers must match exactly
  if (!!config.ctrl !== e.ctrlKey) return false;
  if (!!config.shift !== e.shiftKey) return false;
  if (!!config.alt !== e.altKey) return false;
  if (!!config.meta !== e.metaKey) return false;

  return true;
}

/**
 * Check if two hotkey configs are equal
 */
export function hotkeyConfigsEqual(a: HotkeyConfig, b: HotkeyConfig): boolean {
  return (
    a.key.toLowerCase() === b.key.toLowerCase() &&
    !!a.ctrl === !!b.ctrl &&
    !!a.shift === !!b.shift &&
    !!a.alt === !!b.alt &&
    !!a.meta === !!b.meta
  );
}

/**
 * Check if an element is an input that should block hotkeys
 */
export function isInputElement(target: EventTarget | null): boolean {
  if (!target || !(target instanceof HTMLElement)) return false;

  const tagName = target.tagName.toUpperCase();

  // Standard form elements
  if (tagName === 'INPUT' || tagName === 'TEXTAREA' || tagName === 'SELECT') return true;

  // Content editable elements
  if (target.isContentEditable) return true;

  // ARIA roles that indicate text input
  const role = target.getAttribute('role');
  if (role === 'textbox' || role === 'searchbox' || role === 'combobox') return true;

  return false;
}
