/**
 * useHotkey Hook
 *
 * Register a hotkey callback that respects user configuration.
 * Automatically skips when typing in input fields.
 */

import { useEffect, useCallback } from 'react';
import { useHotkeyContext } from './HotkeyProvider';
import { type HotkeyId, matchesHotkey, isInputElement } from './types';

interface UseHotkeyOptions {
  /** Whether the hotkey is enabled (default: true) */
  enabled?: boolean;
  /** Allow hotkey even when focused on input (default: false) */
  allowInInput?: boolean;
}

/**
 * Register a hotkey by ID with a callback
 *
 * @param id - The hotkey ID from the configuration
 * @param callback - Function to call when hotkey is pressed
 * @param options - Additional options
 *
 * @example
 * ```tsx
 * useHotkey('new-order', () => {
 *   openOrderModal();
 * });
 * ```
 */
export function useHotkey(
  id: HotkeyId,
  callback: () => void,
  options: UseHotkeyOptions = {}
): void {
  const { getHotkey, isReady } = useHotkeyContext();
  const { enabled = true, allowInInput = false } = options;

  const handleKeyDown = useCallback(
    (e: KeyboardEvent) => {
      // Skip if not ready or disabled
      if (!isReady || !enabled) return;

      // Skip if typing in an input (unless explicitly allowed)
      if (!allowInInput && isInputElement(e.target)) return;

      // Get the current hotkey config
      const config = getHotkey(id);
      if (!config) return;

      // Check if this event matches the hotkey
      if (matchesHotkey(e, config)) {
        e.preventDefault();
        callback();
      }
    },
    [id, callback, getHotkey, isReady, enabled, allowInInput]
  );

  useEffect(() => {
    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, [handleKeyDown]);
}

/**
 * Register multiple hotkeys at once
 *
 * @example
 * ```tsx
 * useHotkeys({
 *   'new-order': () => openOrderModal(),
 *   'kill-all': () => cancelAllOrders(),
 * });
 * ```
 */
export function useHotkeys(
  handlers: Partial<Record<HotkeyId, () => void>>,
  options: UseHotkeyOptions = {}
): void {
  const { getHotkey, isReady } = useHotkeyContext();
  const { enabled = true, allowInInput = false } = options;

  const handleKeyDown = useCallback(
    (e: KeyboardEvent) => {
      // Skip if not ready or disabled
      if (!isReady || !enabled) return;

      // Skip if typing in an input (unless explicitly allowed)
      if (!allowInInput && isInputElement(e.target)) return;

      // Check each registered hotkey
      for (const [id, handler] of Object.entries(handlers)) {
        if (!handler) continue;

        const config = getHotkey(id as HotkeyId);
        if (!config) continue;

        if (matchesHotkey(e, config)) {
          e.preventDefault();
          handler();
          return; // Only trigger one handler per keypress
        }
      }
    },
    [handlers, getHotkey, isReady, enabled, allowInInput]
  );

  useEffect(() => {
    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, [handleKeyDown]);
}
