/**
 * Kalshi API client with authenticated requests
 * Uses relay server for transport (solves CORS)
 */

import type {
  Environment,
  KalshiSeries,
  KalshiMarket,
  KalshiOrderbook,
  KalshiEvent,
} from '../types';
import { buildAuthHeaders, getBaseUrl, initAuth } from './kalshiAuth';
import { relayHttp, createRelayRequest } from './relayHttp';

export interface KalshiOrder {
  order_id: string;
  ticker: string;
  side: 'yes' | 'no';
  action: 'buy' | 'sell';
  type: 'limit' | 'market';
  // API returns yes_price/no_price and remaining_count
  yes_price?: number;
  no_price?: number;
  remaining_count: number;
  status: 'pending' | 'resting' | 'filled' | 'cancelled';
  queue_position?: number;
  expires_at?: string;
  created_time: string;
}

export interface KalshiOrderResponse {
  order: KalshiOrder;
}

export interface KalshiOrdersResponse {
  orders: KalshiOrder[];
  cursor?: string;
}

export interface KalshiFill {
  fill_id: string;
  order_id: string;
  ticker: string;
  side: 'yes' | 'no';
  action: 'buy' | 'sell';
  count: number;
  price: number;
  created_at: string;
}

// Public trade from the market trades endpoint
export interface KalshiTrade {
  trade_id: string;
  ticker: string;
  count: number;
  yes_price: number;
  no_price: number;
  taker_side: 'yes' | 'no';
  created_time: string;
}

export interface KalshiPosition {
  ticker: string;
  side: 'yes' | 'no';
  position: number; // Positive = long, negative = short
  average_price: number;
  realized_pnl: number;
}

// Candlestick data from API
export interface KalshiCandlestick {
  end_period_ts: number;
  price: {
    open: number | null;
    high: number | null;
    low: number | null;
    close: number | null;
    mean?: number | null;
  };
  volume: number;
  open_interest: number;
}

export interface KalshiApiClient {
  getSeries: (params?: { category?: string; include_volume?: boolean }) => Promise<KalshiSeries[]>;
  getEvents: (params?: {
    series_ticker?: string;
    status?: 'open' | 'closed' | 'settled';
    with_nested_markets?: boolean;
    limit?: number;
  }) => Promise<KalshiEvent[]>;
  getMarkets: (params?: {
    series_ticker?: string;
    event_ticker?: string;
    status?: string;
    limit?: number;
  }) => Promise<KalshiMarket[]>;
  getMarket: (marketTicker: string) => Promise<KalshiMarket | null>;
  getOrderbook: (marketTicker: string) => Promise<KalshiOrderbook | null>;
  getCandlesticks: (
    seriesTicker: string,
    marketTicker: string,
    params: { start_ts: number; end_ts: number; period_interval: 1 | 60 | 1440 }
  ) => Promise<KalshiCandlestick[]>;
  getEvent: (eventTicker: string) => Promise<KalshiEvent | null>;
  getOpenOrders: () => Promise<KalshiOrder[]>;
  getFilledOrders: (params?: { cursor?: string; limit?: number }) => Promise<KalshiFill[]>;
  getPositions: () => Promise<KalshiPosition[]>;
  getMarketTrades: (params?: {
    ticker?: string;
    cursor?: string;
    limit?: number;
  }) => Promise<KalshiTrade[]>;
  placeOrder: (order: {
    ticker: string;
    side: 'yes' | 'no';
    action: 'buy' | 'sell';
    type: 'limit' | 'market';
    count: number;
    /** Price in cents (0..100). Back-compat: callers may pass 0..1 probability. */
    price?: number;
    post_only?: boolean;
    time_in_force?: 'fill_or_kill' | 'good_till_canceled' | 'immediate_or_cancel';
    expiration_ts?: number;
    expires_at?: string;
  }) => Promise<KalshiOrder>;
  decreaseOrder: (
    orderId: string,
    args: {
      reduce_by?: number;
      reduce_to?: number;
      client_order_id?: string;
      cancel_order_on_pause?: boolean;
    }
  ) => Promise<KalshiOrder>;
  cancelOrder: (orderId: string) => Promise<void>;
  cancelAllOrders: () => Promise<void>;
}

interface AuthState {
  accessKeyId: string;
  privateKey: CryptoKey;
  environment: Environment;
  useRelay: boolean;
}

let authState: AuthState | null = null;

/**
 * Initialize API client with credentials
 */
export async function createKalshiClient(
  accessKeyId: string,
  privateKeyPem: string,
  environment: Environment,
  useRelay: boolean = true
): Promise<KalshiApiClient> {
  try {
    const { accessKeyId: keyId, privateKey } = await initAuth(accessKeyId, privateKeyPem);
    authState = { accessKeyId: keyId, privateKey, environment, useRelay };
  } catch (error) {
    console.error('kalshiApi: Error in createKalshiClient:', error);
    throw error;
  }

  return {
    getSeries,
    getEvents,
    getMarkets,
    getMarket,
    getOrderbook,
    getCandlesticks,
    getEvent,
    getOpenOrders,
    getFilledOrders,
    getPositions,
    getMarketTrades,
    placeOrder,
    decreaseOrder,
    cancelOrder,
    cancelAllOrders,
  };
}

/**
 * Make authenticated request through relay with retry logic
 */
async function authenticatedFetch(
  method: string,
  path: string,
  query: Record<string, string> = {},
  body?: unknown
): Promise<Response> {
  if (!authState) {
    throw new Error('API client not initialized');
  }

  const baseUrl = getBaseUrl(authState.environment);
  const queryString = new URLSearchParams(query).toString();
  const fullPath = queryString ? `${path}?${queryString}` : path;
  const url = `${baseUrl}${fullPath}`;

  const bodyStr = body ? JSON.stringify(body) : '';
  const headers = await buildAuthHeaders(
    authState.accessKeyId,
    authState.privateKey,
    method,
    // Kalshi signature uses the path without query params (per docs).
    path,
    '',
    bodyStr
  );

  const requestHeaders: Record<string, string> = { ...headers };

  // Per Kalshi docs, only include Content-Type when there is a body.
  if (bodyStr) {
    requestHeaders['Content-Type'] = 'application/json';
  }

  // Retry with exponential backoff
  let lastError: Error | null = null;
  for (let attempt = 0; attempt < 3; attempt++) {
    try {
      if (authState.useRelay) {
        // Use relay server
        const requestId = `${Date.now()}-${Math.random().toString(36).slice(2, 11)}`;
        const envelope = createRelayRequest(
          requestId,
          method as 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH',
          url,
          requestHeaders,
          bodyStr || undefined
        );

        // Forward through relay
        const relayResponse = await relayHttp(envelope);

        // Convert relay response to Response-like object
        if (relayResponse.status >= 400) {
          const bodyPreview =
            typeof relayResponse.body === 'string' && relayResponse.body.length > 2000
              ? `${relayResponse.body.slice(0, 2000)}…`
              : relayResponse.body;
          throw new Error(`API error: ${relayResponse.status} ${bodyPreview}`);
        }

        // Return a Response-like object
        return new Response(relayResponse.body, {
          status: relayResponse.status,
          statusText: relayResponse.status >= 200 && relayResponse.status < 300 ? 'OK' : 'Error',
          headers: new Headers(relayResponse.headers),
        });
      } else {
        // Direct connection to Kalshi API
        const response = await fetch(url, {
          method,
          headers: requestHeaders,
          body: bodyStr || undefined,
        });

        if (!response.ok) {
          const errorText = await response.text().catch(() => 'Unknown error');
          throw new Error(`API error: ${response.status} ${errorText}`);
        }

        return response;
      }
    } catch (error) {
      lastError = error instanceof Error ? error : new Error(String(error));

      // Check if it's a CORS error when using direct connection
      if (
        !authState.useRelay &&
        error instanceof TypeError &&
        error.message.includes('Failed to fetch')
      ) {
        throw new Error(
          'CORS error: Direct connection blocked. Please use "Connect Through Server" mode or enable CORS on Kalshi API.'
        );
      }

      // Exponential backoff with jitter
      if (attempt < 2) {
        const delay = Math.pow(2, attempt) * 1000 + Math.random() * 1000;
        await new Promise((resolve) => setTimeout(resolve, delay));
      }
    }
  }

  throw lastError || new Error('Request failed after retries');
}

/**
 * Get series (categories)
 */
async function getSeries(
  params: { category?: string; include_volume?: boolean } = {}
): Promise<KalshiSeries[]> {
  const query: Record<string, string> = {};
  if (params.category) {
    query.category = params.category;
  }
  // Always include volume for richer display
  query.include_volume = params.include_volume !== false ? 'true' : 'false';

  const response = await authenticatedFetch('GET', '/trade-api/v2/series', query);
  const data = await response.json();

  // Handle different response formats and normalize ticker field
  const normalize = (series: KalshiSeries[]): KalshiSeries[] =>
    series.map((s) => ({
      ...s,
      series_ticker: s.series_ticker || s.ticker || '',
    }));

  if (Array.isArray(data)) {
    return normalize(data);
  }
  if (data.series && Array.isArray(data.series)) {
    return normalize(data.series);
  }
  if (data.results && Array.isArray(data.results)) {
    return normalize(data.results);
  }

  return [];
}

/**
 * Get events (for a series or filtered by status)
 */
async function getEvents(
  params: {
    series_ticker?: string;
    status?: 'open' | 'closed' | 'settled';
    with_nested_markets?: boolean;
    limit?: number;
  } = {}
): Promise<KalshiEvent[]> {
  const query: Record<string, string> = {};
  if (params.series_ticker) query.series_ticker = params.series_ticker;
  if (params.status) query.status = params.status;
  if (params.with_nested_markets) query.with_nested_markets = 'true';
  if (params.limit) query.limit = params.limit.toString();

  const response = await authenticatedFetch('GET', '/trade-api/v2/events', query);
  const data = await response.json();

  // Handle different response formats
  if (Array.isArray(data)) {
    return data;
  }
  if (data.events && Array.isArray(data.events)) {
    return data.events;
  }
  if (data.results && Array.isArray(data.results)) {
    return data.results;
  }

  return [];
}

/**
 * Get markets
 */
async function getMarkets(
  params: {
    series_ticker?: string;
    event_ticker?: string;
    status?: string;
    limit?: number;
  } = {}
): Promise<KalshiMarket[]> {
  // Normalize ticker field - API returns 'ticker' but we use 'market_ticker'
  // Also map `expiration_time` → `expiry_time` (API returns `expiration_time`)
  const normalize = (markets: KalshiMarket[]): KalshiMarket[] =>
    markets.map((m) => ({
      ...m,
      market_ticker: m.market_ticker || (m as unknown as { ticker?: string }).ticker || '',
      expiry_time:
        m.expiry_time || (m as unknown as { expiration_time?: string }).expiration_time || '',
    }));

  const pageLimit = params.limit ?? 200;
  const maxPages = 5; // Cap to avoid runaway pagination
  const pageDelayMs = 500; // Delay between pages to respect rate limits

  const allMarkets: KalshiMarket[] = [];
  let cursor: string | undefined;

  for (let page = 0; page < maxPages; page++) {
    const query: Record<string, string> = {};
    if (params.series_ticker) query.series_ticker = params.series_ticker;
    if (params.event_ticker) query.event_ticker = params.event_ticker;
    if (params.status) query.status = params.status;
    query.limit = pageLimit.toString();
    if (cursor) query.cursor = cursor;

    const response = await authenticatedFetch('GET', '/trade-api/v2/markets', query);
    const data = await response.json();

    // Extract markets from response
    let pageMarkets: KalshiMarket[] = [];
    if (Array.isArray(data)) {
      pageMarkets = data;
    } else if (data.markets && Array.isArray(data.markets)) {
      pageMarkets = data.markets;
    } else if (data.results && Array.isArray(data.results)) {
      pageMarkets = data.results;
    }

    allMarkets.push(...pageMarkets);

    // Check for next page cursor
    const nextCursor = data.cursor as string | undefined;
    if (!nextCursor || pageMarkets.length < pageLimit) {
      break; // No more pages
    }

    cursor = nextCursor;

    // Delay between pages to avoid rate limiting
    if (page < maxPages - 1) {
      await new Promise((resolve) => setTimeout(resolve, pageDelayMs));
    }
  }

  return normalize(allMarkets);
}

/**
 * Get a single market by ticker (to access the market title).
 */
async function getMarket(marketTicker: string): Promise<KalshiMarket | null> {
  try {
    const response = await authenticatedFetch(
      'GET',
      `/trade-api/v2/markets/${encodeURIComponent(marketTicker)}`
    );
    const data = await response.json();
    // Kalshi returns { market: { ... } } for this endpoint per docs.
    const rec = data as unknown as Record<string, unknown>;
    const market =
      (rec && typeof rec === 'object' && 'market' in rec
        ? (rec as { market?: unknown }).market
        : null) ?? data;
    return market as KalshiMarket;
  } catch (error) {
    console.warn(`Failed to fetch market ${marketTicker}:`, error);
    return null;
  }
}

/**
 * Get orderbook for a market
 */
async function getOrderbook(marketTicker: string): Promise<KalshiOrderbook | null> {
  try {
    const response = await authenticatedFetch(
      'GET',
      `/trade-api/v2/markets/${marketTicker}/orderbook`
    );
    const data = await response.json();
    return data as KalshiOrderbook;
  } catch (error) {
    console.warn(`Failed to fetch orderbook for ${marketTicker}:`, error);
    return null;
  }
}

/**
 * Get event details
 */
async function getEvent(eventTicker: string): Promise<KalshiEvent | null> {
  try {
    const response = await authenticatedFetch('GET', `/trade-api/v2/events/${eventTicker}`);
    const data = await response.json();
    // API returns { event: {...}, markets: [...] } wrapper
    if (data && typeof data === 'object' && 'event' in data) {
      const event = data.event as KalshiEvent;
      // If event doesn't have start_time but markets do have close_time, use that
      // For NBA games, close_time is when betting closes (game start)
      if (
        !event.start_time &&
        data.markets &&
        Array.isArray(data.markets) &&
        data.markets.length > 0
      ) {
        const firstMarket = data.markets[0];
        if (firstMarket.close_time) {
          event.start_time = firstMarket.close_time;
        }
      }
      return event;
    }
    return data as KalshiEvent;
  } catch (error) {
    console.warn(`Failed to fetch event ${eventTicker}:`, error);
    return null;
  }
}

/**
 * Get candlesticks for a market
 */
async function getCandlesticks(
  seriesTicker: string,
  marketTicker: string,
  params: { start_ts: number; end_ts: number; period_interval: 1 | 60 | 1440 }
): Promise<KalshiCandlestick[]> {
  try {
    const query: Record<string, string> = {
      start_ts: params.start_ts.toString(),
      end_ts: params.end_ts.toString(),
      period_interval: params.period_interval.toString(),
    };

    const response = await authenticatedFetch(
      'GET',
      `/trade-api/v2/series/${seriesTicker}/markets/${marketTicker}/candlesticks`,
      query
    );
    const data = await response.json();

    if (data.candlesticks && Array.isArray(data.candlesticks)) {
      return data.candlesticks;
    }

    return [];
  } catch (error) {
    console.warn(`Failed to fetch candlesticks for ${marketTicker}:`, error);
    return [];
  }
}

/**
 * Get open orders
 */
async function getOpenOrders(): Promise<KalshiOrder[]> {
  try {
    const response = await authenticatedFetch('GET', '/trade-api/v2/portfolio/orders', {
      status: 'resting',
    });
    const data = await response.json();

    // Handle different response formats
    let orders: KalshiOrder[] = [];
    if (Array.isArray(data)) {
      orders = data as KalshiOrder[];
    } else if (data.orders && Array.isArray(data.orders)) {
      orders = data.orders as KalshiOrder[];
    } else if (data.results && Array.isArray(data.results)) {
      orders = data.results as KalshiOrder[];
    } else {
      orders = [];
    }

    // Queue position is not reliably included on the list endpoint; fetch it via the dedicated endpoint.
    // Docs: GET /trade-api/v2/portfolio/orders/queue_positions
    // Note: Kalshi requires a filter (market_tickers or event_ticker) for this endpoint.
    // Skip entirely when there are no orders — avoids 400 "Need to specify market_tickers" error.
    if (orders.length === 0) return orders;
    try {
      const uniqueTickers = Array.from(
        new Set(orders.map((o) => String(o.ticker ?? '')).filter((t) => !!t))
      );
      const qpQuery: Record<string, string> =
        uniqueTickers.length > 0 ? { market_tickers: uniqueTickers.join(',') } : {};
      const qpResp = await authenticatedFetch(
        'GET',
        '/trade-api/v2/portfolio/orders/queue_positions',
        qpQuery
      );
      const qpData = (await qpResp.json()) as unknown;
      const qpObj = qpData as Record<string, unknown> | null;
      const qpArr = (qpObj && Array.isArray(qpObj.queue_positions) && qpObj.queue_positions) || [];

      const byId = new Map<string, number>();
      for (const rec of qpArr) {
        if (!rec || typeof rec !== 'object') continue;
        const recObj = rec as Record<string, unknown>;
        const oid = String(recObj.order_id ?? recObj.orderId ?? '');
        const qp = Number(recObj.queue_position ?? recObj.queuePosition);
        if (!oid) continue;
        if (!Number.isFinite(qp)) continue;
        byId.set(oid, qp);
      }

      if (byId.size > 0) {
        orders = orders.map((o) => ({
          ...o,
          queue_position: byId.get(o.order_id) ?? o.queue_position,
        }));
      } else if (orders.length > 0) {
        // Fallback: per-order endpoint (works even when bulk endpoint is empty).
        // Docs: GET /trade-api/v2/portfolio/orders/{order_id}/queue_position
        const perOrder = await Promise.all(
          orders.map(async (o) => {
            try {
              const r = await authenticatedFetch(
                'GET',
                `/trade-api/v2/portfolio/orders/${o.order_id}/queue_position`,
                {}
              );
              const j = (await r.json()) as Record<string, unknown> | null;
              const qp = Number(j?.queue_position ?? j?.queuePosition);
              return { orderId: o.order_id, qp: Number.isFinite(qp) ? qp : undefined };
            } catch {
              return { orderId: o.order_id, qp: undefined };
            }
          })
        );
        const fallbackMap = new Map<string, number>();
        for (const row of perOrder) {
          if (row.qp === undefined) continue;
          fallbackMap.set(row.orderId, row.qp);
        }
        if (fallbackMap.size > 0) {
          orders = orders.map((o) => ({
            ...o,
            queue_position: fallbackMap.get(o.order_id) ?? o.queue_position,
          }));
        }
      }
    } catch (e) {
      // Non-fatal: we can still show open orders without queue positions.
      console.warn('Failed to fetch queue positions:', e);
    }

    return orders;
  } catch (error) {
    const message = error instanceof Error ? error.message : String(error);
    // Don't silently swallow auth errors — callers (e.g. OrderMonitor) need to see them
    // so they can stop polling and surface a clear message to the user.
    if (
      message.includes('API error: 401') ||
      message.includes('API error: 403') ||
      message.includes('"code":"authentication_error"') ||
      message.includes('"code": "authentication_error"')
    ) {
      throw error;
    }
    console.warn('Failed to fetch open orders:', error);
    return [];
  }
}
/**
 * Place an order
 */
async function placeOrder(order: {
  ticker: string;
  side: 'yes' | 'no';
  action: 'buy' | 'sell';
  type: 'limit' | 'market';
  count: number;
  price?: number;
  /** If true, the order must not cross the book (maker / post-only). */
  post_only?: boolean;
  /** Kalshi time-in-force: fill_or_kill | good_till_canceled | immediate_or_cancel */
  time_in_force?: 'fill_or_kill' | 'good_till_canceled' | 'immediate_or_cancel';
  /** Expiration time in UTC epoch seconds (only meaningful with GTC). */
  expiration_ts?: number;
  expires_at?: string;
}): Promise<KalshiOrder> {
  const priceField = order.side === 'yes' ? 'yes_price' : 'no_price';
  const normalizePriceToCents = (raw: number): number => {
    // Back-compat: some callers still pass 0..1 probability.
    if (raw <= 1.01) return Math.round(raw * 100);
    return Math.round(raw);
  };
  const body = {
    ticker: order.ticker,
    side: order.side,
    action: order.action,
    type: order.type,
    count: order.count,
    ...(order.price !== undefined && { [priceField]: normalizePriceToCents(order.price) }),
    ...(order.post_only !== undefined && { post_only: order.post_only }),
    ...(order.time_in_force && { time_in_force: order.time_in_force }),
    ...(order.expiration_ts !== undefined && { expiration_ts: order.expiration_ts }),
    ...(order.expires_at && { expires_at: order.expires_at }),
  };

  const response = await authenticatedFetch('POST', '/trade-api/v2/portfolio/orders', {}, body);
  const data = await response.json();

  if (data.order) {
    return data.order;
  }
  return data as KalshiOrder;
}

/**
 * Decrease an order's remaining quantity without repricing.
 */
async function decreaseOrder(
  orderId: string,
  args: {
    reduce_by?: number;
    reduce_to?: number;
    client_order_id?: string;
    cancel_order_on_pause?: boolean;
  }
): Promise<KalshiOrder> {
  const body: Record<string, unknown> = {};
  if (args.reduce_by !== undefined) body.reduce_by = args.reduce_by;
  if (args.reduce_to !== undefined) body.reduce_to = args.reduce_to;
  if (args.client_order_id !== undefined) body.client_order_id = args.client_order_id;
  if (args.cancel_order_on_pause !== undefined)
    body.cancel_order_on_pause = args.cancel_order_on_pause;

  const response = await authenticatedFetch(
    'POST',
    `/trade-api/v2/portfolio/orders/${orderId}/decrease`,
    {},
    body
  );
  const data = await response.json();
  if (data.order) {
    return data.order as KalshiOrder;
  }
  return data as KalshiOrder;
}

/**
 * Cancel an order
 */
async function cancelOrder(orderId: string): Promise<void> {
  await authenticatedFetch('DELETE', `/trade-api/v2/portfolio/orders/${orderId}`);
}

/**
 * Cancel all open orders
 */
async function cancelAllOrders(): Promise<void> {
  // First get all open orders
  const orders = await getOpenOrders();

  // Cancel each one
  const cancelPromises = orders.map((order) => cancelOrder(order.order_id));
  await Promise.all(cancelPromises);
}

/**
 * Get market trades (public trade tape)
 */
async function getMarketTrades(
  params: { ticker?: string; cursor?: string; limit?: number } = {}
): Promise<KalshiTrade[]> {
  try {
    const query: Record<string, string> = {};
    if (params.ticker) query.ticker = params.ticker;
    if (params.cursor) query.cursor = params.cursor;
    if (params.limit) query.limit = params.limit.toString();

    const response = await authenticatedFetch('GET', '/trade-api/v2/markets/trades', query);
    const data = await response.json();

    // Handle different response formats
    if (Array.isArray(data)) {
      return data;
    }
    if (data.trades && Array.isArray(data.trades)) {
      return data.trades;
    }
    if (data.results && Array.isArray(data.results)) {
      return data.results;
    }

    return [];
  } catch (error) {
    console.warn('Failed to fetch market trades:', error);
    return [];
  }
}

/**
 * Get filled orders
 */
async function getFilledOrders(
  params: { cursor?: string; limit?: number } = {}
): Promise<KalshiFill[]> {
  try {
    const query: Record<string, string> = { status: 'filled' };
    if (params.cursor) query.cursor = params.cursor;
    if (params.limit) query.limit = params.limit.toString();

    const response = await authenticatedFetch('GET', '/trade-api/v2/portfolio/orders', query);
    const data = await response.json();

    // Handle different response formats
    if (Array.isArray(data)) {
      return data.filter((o: KalshiOrder) => o.status === 'filled').map(orderToFill);
    }
    if (data.orders && Array.isArray(data.orders)) {
      return data.orders.filter((o: KalshiOrder) => o.status === 'filled').map(orderToFill);
    }
    if (data.results && Array.isArray(data.results)) {
      return data.results.filter((o: KalshiOrder) => o.status === 'filled').map(orderToFill);
    }

    return [];
  } catch (error) {
    const message = error instanceof Error ? error.message : String(error);
    if (
      message.includes('API error: 401') ||
      message.includes('API error: 403') ||
      message.includes('"code":"authentication_error"') ||
      message.includes('"code": "authentication_error"')
    ) {
      throw error;
    }
    console.warn('Failed to fetch filled orders:', error);
    return [];
  }
}

/**
 * Convert order to fill (simplified - real API might have separate fill endpoint)
 */
function orderToFill(order: KalshiOrder): KalshiFill {
  const price = order.side === 'yes' ? order.yes_price : order.no_price;
  return {
    fill_id: order.order_id, // Use order_id as fill_id if API doesn't provide separate
    order_id: order.order_id,
    ticker: order.ticker,
    side: order.side,
    action: order.action,
    count: order.remaining_count,
    price: price ?? 0,
    created_at: order.created_time,
  };
}

/**
 * Normalize Kalshi "positions" payloads into our internal KalshiPosition[] shape.
 * Kalshi has had multiple response shapes over time (and can differ by endpoint/version).
 */
function normalizeKalshiPositions(raw: unknown): KalshiPosition[] {
  const asRecord = (v: unknown): Record<string, unknown> | null =>
    v && typeof v === 'object' ? (v as Record<string, unknown>) : null;
  const toNumber = (v: unknown): number | null => {
    if (typeof v === 'number' && Number.isFinite(v)) return v;
    if (typeof v === 'string') {
      const n = Number(v);
      if (Number.isFinite(n)) return n;
    }
    return null;
  };
  const toSide = (v: unknown): 'yes' | 'no' | null => {
    if (typeof v !== 'string') return null;
    const s = v.toLowerCase();
    if (s === 'yes' || s === 'no') return s;
    return null;
  };

  // Unwrap common response envelopes
  const rootRec = asRecord(raw);
  let items: unknown = raw;
  if (rootRec) {
    items =
      // Newer/alternate shapes (seen in prod): { cursor, market_positions, event_positions }
      rootRec.market_positions ??
      rootRec.marketPositions ??
      rootRec.event_positions ??
      rootRec.eventPositions ??
      // Older/common envelopes
      rootRec.positions ??
      rootRec.results ??
      rootRec.data ??
      rootRec.portfolio_positions ??
      rootRec.portfolioPositions ??
      rootRec.position ??
      raw;

    // Sometimes positions are nested like { positions: { positions: [...] } }
    const itemsRec = asRecord(items);
    if (itemsRec && (itemsRec.positions || itemsRec.results || itemsRec.data)) {
      items = itemsRec.positions ?? itemsRec.results ?? itemsRec.data;
    }
  }

  // If the response includes both market_positions + event_positions, merge them.
  // Kalshi's /portfolio/positions in prod commonly returns both arrays.
  if (rootRec) {
    const marketItems = rootRec.market_positions ?? rootRec.marketPositions;
    const eventItems = rootRec.event_positions ?? rootRec.eventPositions;
    if (Array.isArray(marketItems) || Array.isArray(eventItems)) {
      items = [
        ...(Array.isArray(marketItems) ? marketItems : []),
        ...(Array.isArray(eventItems) ? eventItems : []),
      ];
    }
  }

  if (!Array.isArray(items)) {
    return [];
  }

  const out: KalshiPosition[] = [];

  for (const item of items) {
    const rec = asRecord(item);
    if (!rec) continue;

    const ticker =
      (typeof rec.ticker === 'string' && rec.ticker) ||
      (typeof rec.market_ticker === 'string' && rec.market_ticker) ||
      (typeof rec.marketTicker === 'string' && rec.marketTicker) ||
      (typeof rec.event_ticker === 'string' && rec.event_ticker) ||
      (typeof rec.eventTicker === 'string' && rec.eventTicker) ||
      (typeof rec.symbol === 'string' && rec.symbol) ||
      null;

    // Shape A2: signed position without explicit side (common in Kalshi market_positions)
    // Convention: position > 0 => YES exposure, position < 0 => NO exposure.
    if (ticker) {
      const signed = toNumber(rec.position);
      if (signed !== null && signed !== 0 && rec.side === undefined) {
        const inferredSide: 'yes' | 'no' = signed < 0 ? 'no' : 'yes';
        const qty = Math.abs(signed);

        // Try to derive avg price from exposure / qty when avg_price isn't provided.
        const exposure =
          toNumber(
            rec.market_exposure ?? rec.marketExposure ?? rec.total_cost ?? rec.totalCost ?? rec.cost
          ) ??
          toNumber(
            rec.market_exposure_dollars ??
              rec.marketExposureDollars ??
              rec.total_cost_dollars ??
              rec.totalCostDollars ??
              rec.cost_dollars ??
              rec.costDollars
          );

        const avg =
          toNumber(rec.average_price ?? rec.avg_price ?? rec.avgPrice ?? rec.averagePrice) ??
          (exposure !== null && qty > 0 ? exposure / qty : 0);

        const realized =
          toNumber(
            rec.realized_pnl ??
              rec.realizedPnl ??
              rec.realized_pnl_dollars ??
              rec.realizedPnlDollars
          ) ?? 0;

        out.push({
          ticker,
          side: inferredSide,
          position: qty,
          average_price: avg,
          realized_pnl: realized,
        });
        continue;
      }
    }

    // Shape A: explicit side + position
    const side = toSide(rec.side);
    const position = toNumber(rec.position ?? rec.quantity ?? rec.count);

    if (ticker && side && position !== null) {
      const avg =
        toNumber(
          rec.average_price ??
            rec.avg_price ??
            rec.avgPrice ??
            rec.averagePrice ??
            rec.entry_price ??
            rec.entryPrice
        ) ?? 0;
      const realized =
        toNumber(
          rec.realized_pnl ?? rec.realizedPnl ?? rec.realized_pnl_dollars ?? rec.realizedPnlDollars
        ) ?? 0;

      out.push({
        ticker,
        side,
        position,
        average_price: avg,
        realized_pnl: realized,
      });
      continue;
    }

    // Shape B: combined yes/no positions per ticker
    if (ticker) {
      const yesPos = toNumber(
        rec.yes_position ??
          rec.yesPosition ??
          rec.yes_qty ??
          rec.yesQty ??
          rec.yes_quantity ??
          rec.yesQuantity ??
          rec.yes_contracts ??
          rec.yesContracts
      );
      const noPos = toNumber(
        rec.no_position ??
          rec.noPosition ??
          rec.no_qty ??
          rec.noQty ??
          rec.no_quantity ??
          rec.noQuantity ??
          rec.no_contracts ??
          rec.noContracts
      );

      if (yesPos !== null) {
        const yesAvg =
          toNumber(
            rec.yes_average_price ??
              rec.yes_avg_price ??
              rec.yesAvgPrice ??
              rec.average_price ??
              rec.avg_price
          ) ?? 0;
        const yesRealized =
          toNumber(
            rec.yes_realized_pnl ?? rec.yesRealizedPnl ?? rec.realized_pnl ?? rec.realizedPnl
          ) ?? 0;
        out.push({
          ticker,
          side: 'yes',
          position: yesPos,
          average_price: yesAvg,
          realized_pnl: yesRealized,
        });
      }

      if (noPos !== null) {
        const noAvg =
          toNumber(
            rec.no_average_price ??
              rec.no_avg_price ??
              rec.noAvgPrice ??
              rec.average_price ??
              rec.avg_price
          ) ?? 0;
        const noRealized =
          toNumber(
            rec.no_realized_pnl ?? rec.noRealizedPnl ?? rec.realized_pnl ?? rec.realizedPnl
          ) ?? 0;
        out.push({
          ticker,
          side: 'no',
          position: noPos,
          average_price: noAvg,
          realized_pnl: noRealized,
        });
      }
    }
  }

  // Filter out zero positions to match what users expect to see in a Positions table.
  return out.filter((p) => Number.isFinite(p.position) && p.position !== 0);
}

/**
 * Get positions
 */
async function getPositions(): Promise<KalshiPosition[]> {
  try {
    const response = await authenticatedFetch('GET', '/trade-api/v2/portfolio/positions');
    const data = await response.json();

    const positions = normalizeKalshiPositions(data);
    return positions;
  } catch (error) {
    const message = error instanceof Error ? error.message : String(error);
    if (
      message.includes('API error: 401') ||
      message.includes('API error: 403') ||
      message.includes('"code":"authentication_error"') ||
      message.includes('"code": "authentication_error"')
    ) {
      throw error;
    }
    console.warn('Failed to fetch positions:', error);
    return [];
  }
}

/**
 * Check if error is CORS-related
 * Note: With relay, CORS errors should be rare, but kept for compatibility
 */
export function isCorsError(error: unknown): boolean {
  if (error instanceof Error) {
    return (
      error.message === 'CORS_BLOCKED' ||
      error.message.includes('CORS') ||
      (error.message.includes('Failed to fetch') && !error.message.includes('Relay'))
    );
  }
  return false;
}
