import type { Environment } from '@/types';

const STORAGE_KEY = 'kalshi_profiles_v1';
const LEGACY_KEY = 'kalshi_creds_encrypted';

export interface KalshiProfile {
  id: string;
  name: string;
  accessKey: string;
  privateKey: string;
  environment: Environment;
  useRelay: boolean;
  updatedAt: number;
}

export interface KalshiProfilesStore {
  version: 1;
  activeProfileId: string | null;
  profiles: KalshiProfile[];
}

function makeId(): string {
  if (typeof crypto !== 'undefined' && typeof crypto.randomUUID === 'function') {
    return crypto.randomUUID();
  }
  return `kp-${Date.now()}-${Math.random().toString(36).slice(2, 10)}`;
}

function normalizeProfile(input: Partial<KalshiProfile>): KalshiProfile | null {
  const accessKey = String(input.accessKey || '').trim();
  const privateKey = String(input.privateKey || '').trim();
  if (!accessKey || !privateKey) return null;

  const environment: Environment = input.environment === 'demo' ? 'demo' : 'prod';
  const useRelay = input.useRelay !== false;
  const id = String(input.id || '').trim() || makeId();
  const fallbackName = accessKey;
  const name = String(input.name || '').trim() || fallbackName;
  const updatedAt = Number(input.updatedAt) || Date.now();

  return {
    id,
    name,
    accessKey,
    privateKey,
    environment,
    useRelay,
    updatedAt,
  };
}

function sanitizeStore(raw: unknown): KalshiProfilesStore {
  if (!raw || typeof raw !== 'object') {
    return { version: 1, activeProfileId: null, profiles: [] };
  }

  const rec = raw as Record<string, unknown>;
  const profilesRaw = Array.isArray(rec.profiles) ? rec.profiles : [];
  const profiles = profilesRaw
    .map((p) => normalizeProfile((p as Record<string, unknown>) || {}))
    .filter((p): p is KalshiProfile => !!p);

  const activeProfileIdRaw = String(rec.activeProfileId || '').trim();
  const activeProfileId = profiles.some((p) => p.id === activeProfileIdRaw)
    ? activeProfileIdRaw
    : profiles[0]?.id || null;

  return {
    version: 1,
    activeProfileId,
    profiles,
  };
}

function readStore(): KalshiProfilesStore | null {
  try {
    const raw = localStorage.getItem(STORAGE_KEY);
    if (!raw) return null;
    return sanitizeStore(JSON.parse(raw));
  } catch {
    return null;
  }
}

export function saveKalshiProfilesStore(store: KalshiProfilesStore): void {
  localStorage.setItem(STORAGE_KEY, JSON.stringify(sanitizeStore(store)));
}

export function ensureKalshiProfilesStore(): KalshiProfilesStore {
  const existing = readStore();
  if (existing) {
    saveKalshiProfilesStore(existing);
    return existing;
  }

  let migrated: KalshiProfilesStore = { version: 1, activeProfileId: null, profiles: [] };
  try {
    const legacyRaw = localStorage.getItem(LEGACY_KEY);
    if (legacyRaw) {
      const legacy = JSON.parse(legacyRaw) as Record<string, unknown>;
      const profile = normalizeProfile({
        name: String(legacy.accessKey || ''),
        accessKey: String(legacy.accessKey || ''),
        privateKey: String(legacy.privateKey || ''),
        environment: legacy.environment === 'demo' ? 'demo' : 'prod',
        useRelay: legacy.useRelay !== false,
      });
      if (profile) {
        migrated = {
          version: 1,
          activeProfileId: profile.id,
          profiles: [profile],
        };
      }
      localStorage.removeItem(LEGACY_KEY);
    }
  } catch {
    // Ignore malformed legacy payloads.
  }

  // Auto-seed from env vars in dev mode (VITE_KALSHI_ACCESS_KEY + VITE_KALSHI_PRIVATE_KEY)
  if (migrated.profiles.length === 0) {
    const envAccessKey = import.meta.env.VITE_KALSHI_ACCESS_KEY as string | undefined;
    const envPrivateKey = import.meta.env.VITE_KALSHI_PRIVATE_KEY as string | undefined;
    if (envAccessKey && envPrivateKey) {
      // Restore newlines in PEM key (env vars use \n literal)
      const pemKey = envPrivateKey.replace(/\\n/g, '\n');
      const profile = normalizeProfile({
        name: 'Dev (env)',
        accessKey: envAccessKey,
        privateKey: pemKey,
        environment: 'prod',
        useRelay: true,
      });
      if (profile) {
        migrated = {
          version: 1,
          activeProfileId: profile.id,
          profiles: [profile],
        };
      }
    }
  }

  saveKalshiProfilesStore(migrated);
  return migrated;
}

export function getActiveKalshiProfile(store: KalshiProfilesStore): KalshiProfile | null {
  if (!store.activeProfileId) return null;
  return store.profiles.find((p) => p.id === store.activeProfileId) || null;
}

export function upsertKalshiProfile(
  store: KalshiProfilesStore,
  input: {
    id?: string;
    name?: string;
    accessKey: string;
    privateKey: string;
    environment?: Environment;
    useRelay?: boolean;
  }
): KalshiProfilesStore {
  const normalized = normalizeProfile({
    id: input.id,
    name: input.name,
    accessKey: input.accessKey,
    privateKey: input.privateKey,
    environment: input.environment ?? 'prod',
    useRelay: input.useRelay ?? true,
  });
  if (!normalized) return store;

  const idx = store.profiles.findIndex((p) => p.id === normalized.id);
  const profiles = [...store.profiles];
  if (idx >= 0) {
    profiles[idx] = {
      ...profiles[idx],
      ...normalized,
      name: normalized.name || profiles[idx].name || normalized.accessKey,
      updatedAt: Date.now(),
    };
  } else {
    profiles.push({ ...normalized, updatedAt: Date.now() });
  }

  return {
    version: 1,
    activeProfileId: normalized.id,
    profiles,
  };
}

export function upsertKalshiProfileByAccessKey(
  store: KalshiProfilesStore,
  input: {
    name?: string;
    accessKey: string;
    privateKey: string;
    environment?: Environment;
    useRelay?: boolean;
  }
): KalshiProfilesStore {
  const existing = store.profiles.find((p) => p.accessKey === String(input.accessKey).trim());
  return upsertKalshiProfile(store, {
    id: existing?.id,
    name: input.name || existing?.name || String(input.accessKey).trim(),
    accessKey: input.accessKey,
    privateKey: input.privateKey,
    environment: input.environment,
    useRelay: input.useRelay,
  });
}

export function setActiveKalshiProfile(
  store: KalshiProfilesStore,
  profileId: string | null
): KalshiProfilesStore {
  if (!profileId) return { ...store, activeProfileId: null };
  if (!store.profiles.some((p) => p.id === profileId)) return store;
  return { ...store, activeProfileId: profileId };
}

export function deleteKalshiProfile(
  store: KalshiProfilesStore,
  profileId: string
): KalshiProfilesStore {
  const profiles = store.profiles.filter((p) => p.id !== profileId);
  const activeProfileId =
    store.activeProfileId && store.activeProfileId !== profileId
      ? store.activeProfileId
      : profiles[0]?.id || null;
  return {
    version: 1,
    activeProfileId,
    profiles,
  };
}
