/**
 * Generic Kalshi ticker parser
 *
 * Parses various Kalshi ticker formats into structured data.
 * Uses prefix-based lookup for O(1) dispatch to the right parser.
 */

import type {
  ParsedTicker,
  GameInfo,
  PlayerInfo,
  TickerCategory,
  SpreadInfo,
  TotalInfo,
  TennisMatchInfo,
} from './types';
import { parseDateFromTicker, formatDateShort } from './types';
import { teamNameFromCode } from '../nbaConsolidated/teamNames';

/**
 * Parser function signature
 */
type TickerParser = (raw: string) => ParsedTicker;

/**
 * Parsers by prefix - longer prefixes listed first to match correctly
 * (e.g., KXNBAGAME before KXNBA)
 */
const PARSERS: Array<[string, TickerParser]> = [
  // NBA
  ['KXNBASPREAD', parseNbaSpread],
  ['KXNBATOTAL', parseNbaTotal],
  ['KXNBAGAME', parseNbaGame],
  ['KXNBA', parseNbaSeason],
  // NFL
  ['KXNFLSPREAD', parseNflSpread],
  ['KXNFLTOTAL', parseNflTotal],
  ['KXNFLFIRSTTD', parseNflFirstTd],
  ['KXNFL2TD', parseNfl2Td],
  ['KXNFLGAME', parseNflGame],
  ['KXSB', parseSuperBowl],
  // NHL
  ['KXNHLSPREAD', parseNhlSpread],
  ['KXNHLTOTAL', parseNhlTotal],
  ['KXNHLGAME', parseNhlGame],
  // NCAA - longer prefixes first
  ['KXNCAAMBSPREAD', parseNcaaMbSpread],
  ['KXNCAAMBTOTAL', parseNcaaMbTotal],
  ['KXNCAAMBGAME', parseNcaaMbGame],
  ['KXNCAAWBSPREAD', parseNcaaWbSpread],
  ['KXNCAAWBTOTAL', parseNcaaWbTotal],
  ['KXNCAAWBGAME', parseNcaaWbGame],
  ['KXNCAAGAME', parseNcaaGame],
  // Tennis
  ['KXATPMATCH', parseAtpMatch],
  ['KXWTAMATCH', parseWtaMatch],
];

/**
 * Parse any Kalshi ticker into structured data
 */
export function parseKalshiTicker(ticker: string): ParsedTicker {
  const raw = ticker.trim().toUpperCase();

  for (const [prefix, parser] of PARSERS) {
    if (raw.startsWith(prefix)) {
      return parser(raw);
    }
  }

  return parseFallback(raw);
}

// =============================================================================
// Individual Parsers
// =============================================================================

/**
 * NBA game moneyline: KXNBAGAME-26JAN15BOSMIA-BOS
 */
function parseNbaGame(raw: string): ParsedTicker {
  const parts = raw.includes('--') ? raw.split('--') : raw.split('-');

  let eventPart: string;
  let teamCode: string | null = null;

  if (raw.includes('--')) {
    eventPart = parts[0] ?? '';
    teamCode = parts[1] ?? null;
  } else {
    teamCode = parts[parts.length - 1] ?? null;
    eventPart = parts.slice(0, -1).join('-');
  }

  const eventParts = eventPart.split('-');
  const mid = eventParts[1] ?? '';

  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamCodes = mid.slice(7);

  if (teamCodes.length !== 6) {
    return parseFallback(raw);
  }

  const awayCode = teamCodes.slice(0, 3);
  const homeCode = teamCodes.slice(3, 6);

  const game: GameInfo = {
    awayCode,
    homeCode,
    awayName: teamNameFromCode(awayCode),
    homeName: teamNameFromCode(homeCode),
  };

  const shortLabel = `NBA: ${awayCode} @ ${homeCode}`;
  const fullLabel = teamCode
    ? `NBA: ${game.awayName ?? awayCode} @ ${game.homeName ?? homeCode} · ${teamCode} Win`
    : `NBA: ${game.awayName ?? awayCode} @ ${game.homeName ?? homeCode}`;

  return {
    raw,
    category: 'nba-game',
    series: 'KXNBAGAME',
    date,
    game,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

/**
 * NBA season props: KXNBA-26-SAS
 */
function parseNbaSeason(raw: string): ParsedTicker {
  const parts = raw.split('-');
  const year = parts[1] ? `20${parts[1]}` : null;
  const outcome = parts[2] ?? null;

  const shortLabel = outcome ? `NBA Season: ${outcome}` : 'NBA Season Prop';
  const fullLabel = year ? `NBA ${year} Season${outcome ? ` · ${outcome}` : ''}` : shortLabel;

  return {
    raw,
    category: 'nba-season',
    series: 'KXNBA',
    date: null,
    game: null,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome,
    shortLabel,
    fullLabel,
  };
}

/**
 * NFL 2+ touchdowns prop: KXNFL2TD-26FEB08SEANE-SEAABARNER88
 */
function parseNfl2Td(raw: string): ParsedTicker {
  return parseNflPlayerProp(raw, 'KXNFL2TD', 'nfl-2td', '2+ TD');
}

/**
 * NFL first touchdown prop: KXNFLFIRSTTD-26FEB08SEANE-NEKBOUTTE9
 */
function parseNflFirstTd(raw: string): ParsedTicker {
  return parseNflPlayerProp(raw, 'KXNFLFIRSTTD', 'nfl-first-td', 'First TD');
}

/**
 * Common parser for NFL player prop tickers
 */
function parseNflPlayerProp(
  raw: string,
  series: string,
  category: TickerCategory,
  propLabel: string
): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 3) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 12) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamsStr = mid.slice(7);

  let awayCode: string;
  let homeCode: string;

  if (teamsStr.length >= 6) {
    awayCode = teamsStr.slice(0, 3);
    homeCode = teamsStr.slice(3, 6);
  } else if (teamsStr.length >= 5) {
    awayCode = teamsStr.slice(0, 3);
    homeCode = teamsStr.slice(3, 5);
  } else {
    awayCode = teamsStr.slice(0, 3);
    homeCode = teamsStr.slice(3);
  }

  const game: GameInfo = { awayCode, homeCode };
  const marketPart = parts[2] ?? '';
  const player = parsePlayerFromMarket(marketPart);

  const dateLabel = date ? formatDateShort(date) : '';
  const gameLabel = `${awayCode} @ ${homeCode}`;
  const playerLabel = player?.displayName ?? player?.playerId ?? '';

  const shortLabel = playerLabel ? `${propLabel}: ${playerLabel}` : `NFL ${propLabel}`;

  const fullLabel = [`NFL ${propLabel}`, dateLabel, gameLabel, playerLabel]
    .filter(Boolean)
    .join(' · ');

  return {
    raw,
    category,
    series,
    date,
    game,
    player,
    spread: null,
    total: null,
    tennis: null,
    outcome: marketPart,
    shortLabel,
    fullLabel,
  };
}

/**
 * Super Bowl winner: KXSB-26-SEA
 */
function parseSuperBowl(raw: string): ParsedTicker {
  const parts = raw.split('-');
  const year = parts[1] ? `20${parts[1]}` : null;
  const teamCode = parts[2] ?? null;

  const teamName = teamCode ? nflTeamFromCode(teamCode) : null;

  const shortLabel = teamCode ? `SB: ${teamName ?? teamCode}` : 'Super Bowl';

  const fullLabel = year
    ? `Super Bowl ${year}${teamCode ? ` · ${teamName ?? teamCode}` : ''}`
    : shortLabel;

  return {
    raw,
    category: 'nfl-superbowl',
    series: 'KXSB',
    date: null,
    game: null,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

/**
 * NFL game moneyline: KXNFLGAME-26FEB08SEANE-SEA
 */
function parseNflGame(raw: string): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 12) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamsStr = mid.slice(7);

  const awayCode = teamsStr.slice(0, 3);
  const homeCode = teamsStr.slice(3, 5) || teamsStr.slice(3, 6);
  const teamCode = parts[2] ?? null;

  const game: GameInfo = { awayCode, homeCode };

  const awayName = nflTeamFromCode(awayCode);
  const homeName = nflTeamFromCode(homeCode);

  const shortLabel = `NFL: ${awayCode} @ ${homeCode}`;
  const fullLabel = teamCode
    ? `NFL: ${awayName ?? awayCode} @ ${homeName ?? homeCode} · ${teamCode} Win`
    : `NFL: ${awayName ?? awayCode} @ ${homeName ?? homeCode}`;

  return {
    raw,
    category: 'nfl-game',
    series: 'KXNFLGAME',
    date,
    game,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

/**
 * NCAA Men's Basketball: KXNCAAMBGAME-26JAN28UCLAORE-UCLA
 */
function parseNcaaMbGame(raw: string): ParsedTicker {
  return parseNcaaGameGeneric(raw, 'KXNCAAMBGAME', 'NCAAMB');
}

/**
 * NCAA Women's Basketball: KXNCAAWBGAME-...
 */
function parseNcaaWbGame(raw: string): ParsedTicker {
  return parseNcaaGameGeneric(raw, 'KXNCAAWBGAME', 'NCAAWB');
}

/**
 * NCAA generic game: KXNCAAGAME-...
 */
function parseNcaaGame(raw: string): ParsedTicker {
  return parseNcaaGameGeneric(raw, 'KXNCAAGAME', 'NCAA');
}

/**
 * Generic NCAA game parser
 */
function parseNcaaGameGeneric(raw: string, series: string, label: string): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamCodes = mid.slice(7);

  if (teamCodes.length < 6) {
    return parseFallback(raw);
  }

  const awayCode = teamCodes.slice(0, 3);
  const homeCode = teamCodes.slice(3, 6);
  const teamCode = parts[2] ?? null;

  const game: GameInfo = { awayCode, homeCode };

  const shortLabel = `${label}: ${awayCode} @ ${homeCode}`;
  const fullLabel = teamCode
    ? `${label}: ${awayCode} @ ${homeCode} · ${teamCode} Win`
    : `${label}: ${awayCode} @ ${homeCode}`;

  return {
    raw,
    category: 'ncaa-game',
    series,
    date,
    game,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

// =============================================================================
// Spread Parsers
// =============================================================================

/**
 * NBA spread: KXNBASPREAD-26FEB04BOSMIA-BOS
 * The outcome team is the one being bet on to "cover" or "miss"
 */
function parseNbaSpread(raw: string): ParsedTicker {
  return parseSpreadGeneric(raw, 'KXNBASPREAD', 'nba-spread', 'NBA');
}

/**
 * NFL spread: KXNFLSPREAD-26FEB08SEANE-SEA
 */
function parseNflSpread(raw: string): ParsedTicker {
  return parseSpreadGeneric(raw, 'KXNFLSPREAD', 'nfl-spread', 'NFL');
}

/**
 * NHL spread: KXNHLSPREAD-26FEB08SEATOR-SEA
 */
function parseNhlSpread(raw: string): ParsedTicker {
  return parseSpreadGeneric(raw, 'KXNHLSPREAD', 'nhl-spread', 'NHL');
}

/**
 * NCAA Men's Basketball spread: KXNCAAMBSPREAD-26FEB04UCLAORE-UCLA
 */
function parseNcaaMbSpread(raw: string): ParsedTicker {
  return parseSpreadGeneric(raw, 'KXNCAAMBSPREAD', 'ncaa-spread', 'NCAAMB');
}

/**
 * NCAA Women's Basketball spread: KXNCAAWBSPREAD-...
 */
function parseNcaaWbSpread(raw: string): ParsedTicker {
  return parseSpreadGeneric(raw, 'KXNCAAWBSPREAD', 'ncaa-spread', 'NCAAWB');
}

/**
 * Generic spread parser for all sports
 * Format: SERIES-DATEAWAYHOME-TEAM
 */
function parseSpreadGeneric(
  raw: string,
  series: string,
  category: TickerCategory,
  label: string
): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamCodes = mid.slice(7);

  if (teamCodes.length < 6) {
    return parseFallback(raw);
  }

  const awayCode = teamCodes.slice(0, 3);
  const homeCode = teamCodes.slice(3, 6);
  const teamCode = parts[2] ?? null;

  const game: GameInfo = { awayCode, homeCode };

  // Get team names for NBA
  if (label === 'NBA') {
    game.awayName = teamNameFromCode(awayCode);
    game.homeName = teamNameFromCode(homeCode);
  }

  const shortLabel = `${label} Spread: ${awayCode} @ ${homeCode}`;
  const fullLabel = teamCode
    ? `${label} Spread: ${awayCode} @ ${homeCode} · ${teamCode}`
    : `${label} Spread: ${awayCode} @ ${homeCode}`;

  // Note: Actual spread values come from market data, not ticker
  const spread: SpreadInfo = {
    favoredTeam: teamCode ?? awayCode,
    spreadValue: 0, // Will be populated from market data
    side: 'cover',
  };

  return {
    raw,
    category,
    series,
    date,
    game,
    player: null,
    spread,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

// =============================================================================
// Total Parsers
// =============================================================================

/**
 * NBA total (over/under): KXNBATOTAL-26FEB04BOSMIA-O
 * Outcome is O (over) or U (under)
 */
function parseNbaTotal(raw: string): ParsedTicker {
  return parseTotalGeneric(raw, 'KXNBATOTAL', 'nba-total', 'NBA');
}

/**
 * NFL total: KXNFLTOTAL-26FEB08SEANE-O
 */
function parseNflTotal(raw: string): ParsedTicker {
  return parseTotalGeneric(raw, 'KXNFLTOTAL', 'nfl-total', 'NFL');
}

/**
 * NHL total: KXNHLTOTAL-26FEB08SEATOR-O
 */
function parseNhlTotal(raw: string): ParsedTicker {
  return parseTotalGeneric(raw, 'KXNHLTOTAL', 'nhl-total', 'NHL');
}

/**
 * NCAA Men's Basketball total: KXNCAAMBTOTAL-26FEB04UCLAORE-O
 */
function parseNcaaMbTotal(raw: string): ParsedTicker {
  return parseTotalGeneric(raw, 'KXNCAAMBTOTAL', 'ncaa-total', 'NCAAMB');
}

/**
 * NCAA Women's Basketball total: KXNCAAWBTOTAL-...
 */
function parseNcaaWbTotal(raw: string): ParsedTicker {
  return parseTotalGeneric(raw, 'KXNCAAWBTOTAL', 'ncaa-total', 'NCAAWB');
}

/**
 * Generic total parser for all sports
 * Format: SERIES-DATEAWAYHOME-O or SERIES-DATEAWAYHOME-U
 */
function parseTotalGeneric(
  raw: string,
  series: string,
  category: TickerCategory,
  label: string
): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamCodes = mid.slice(7);

  if (teamCodes.length < 6) {
    return parseFallback(raw);
  }

  const awayCode = teamCodes.slice(0, 3);
  const homeCode = teamCodes.slice(3, 6);
  const outcome = parts[2]?.toUpperCase() ?? null;

  const game: GameInfo = { awayCode, homeCode };

  // Get team names for NBA
  if (label === 'NBA') {
    game.awayName = teamNameFromCode(awayCode);
    game.homeName = teamNameFromCode(homeCode);
  }

  const side = outcome === 'U' ? 'under' : 'over';
  const sideLabel = side === 'over' ? 'Over' : 'Under';

  const shortLabel = `${label} Total: ${awayCode} @ ${homeCode} ${sideLabel}`;
  const fullLabel = `${label} Total: ${awayCode} @ ${homeCode} · ${sideLabel}`;

  // Note: Actual total value comes from market data, not ticker
  const total: TotalInfo = {
    totalValue: 0, // Will be populated from market data
    side,
  };

  return {
    raw,
    category,
    series,
    date,
    game,
    player: null,
    spread: null,
    total,
    tennis: null,
    outcome,
    shortLabel,
    fullLabel,
  };
}

// =============================================================================
// NHL Parsers
// =============================================================================

/**
 * NHL game moneyline: KXNHLGAME-26FEB08SEATOR-SEA
 */
function parseNhlGame(raw: string): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const teamCodes = mid.slice(7);

  if (teamCodes.length < 6) {
    return parseFallback(raw);
  }

  const awayCode = teamCodes.slice(0, 3);
  const homeCode = teamCodes.slice(3, 6);
  const teamCode = parts[2] ?? null;

  const game: GameInfo = {
    awayCode,
    homeCode,
    awayName: nhlTeamFromCode(awayCode) ?? undefined,
    homeName: nhlTeamFromCode(homeCode) ?? undefined,
  };

  const shortLabel = `NHL: ${awayCode} @ ${homeCode}`;
  const fullLabel = teamCode
    ? `NHL: ${game.awayName ?? awayCode} @ ${game.homeName ?? homeCode} · ${teamCode} Win`
    : `NHL: ${game.awayName ?? awayCode} @ ${game.homeName ?? homeCode}`;

  return {
    raw,
    category: 'nhl-game',
    series: 'KXNHLGAME',
    date,
    game,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: teamCode,
    shortLabel,
    fullLabel,
  };
}

/**
 * NHL team code to name mapping
 */
function nhlTeamFromCode(code: string): string | null {
  const NHL_TEAMS: Record<string, string> = {
    ANA: 'Anaheim Ducks',
    ARI: 'Arizona Coyotes',
    BOS: 'Boston Bruins',
    BUF: 'Buffalo Sabres',
    CGY: 'Calgary Flames',
    CAR: 'Carolina Hurricanes',
    CHI: 'Chicago Blackhawks',
    COL: 'Colorado Avalanche',
    CBJ: 'Columbus Blue Jackets',
    DAL: 'Dallas Stars',
    DET: 'Detroit Red Wings',
    EDM: 'Edmonton Oilers',
    FLA: 'Florida Panthers',
    LAK: 'Los Angeles Kings',
    MIN: 'Minnesota Wild',
    MTL: 'Montreal Canadiens',
    NSH: 'Nashville Predators',
    NJD: 'New Jersey Devils',
    NYI: 'New York Islanders',
    NYR: 'New York Rangers',
    OTT: 'Ottawa Senators',
    PHI: 'Philadelphia Flyers',
    PIT: 'Pittsburgh Penguins',
    SJS: 'San Jose Sharks',
    SEA: 'Seattle Kraken',
    STL: 'St. Louis Blues',
    TBL: 'Tampa Bay Lightning',
    TOR: 'Toronto Maple Leafs',
    UTA: 'Utah Hockey Club',
    VAN: 'Vancouver Canucks',
    VGK: 'Vegas Golden Knights',
    WSH: 'Washington Capitals',
    WPG: 'Winnipeg Jets',
  };
  return NHL_TEAMS[code] ?? null;
}

// =============================================================================
// Tennis Parsers
// =============================================================================

/**
 * ATP tennis match: KXATPMATCH-26FEB08DJOSINNER-DJO
 */
function parseAtpMatch(raw: string): ParsedTicker {
  return parseTennisMatchGeneric(raw, 'KXATPMATCH', 'ATP');
}

/**
 * WTA tennis match: KXWTAMATCH-26FEB08SWIGAUFF-SWI
 */
function parseWtaMatch(raw: string): ParsedTicker {
  return parseTennisMatchGeneric(raw, 'KXWTAMATCH', 'WTA');
}

/**
 * Generic tennis match parser
 * Format: SERIES-DATEPLAYER1PLAYER2-OUTCOME
 * Player codes are variable length (3-6 chars typically)
 */
function parseTennisMatchGeneric(raw: string, series: string, label: string): ParsedTicker {
  const parts = raw.split('-');
  if (parts.length < 2) {
    return parseFallback(raw);
  }

  const mid = parts[1] ?? '';
  if (mid.length < 13) {
    return parseFallback(raw);
  }

  const date = parseDateFromTicker(mid.slice(0, 7));
  const playersStr = mid.slice(7);

  // Tennis player codes are variable length, so we'll just use the full string
  // and extract the winner from the outcome part
  const outcome = parts[2] ?? null;

  // Create placeholder tennis info - actual player names need lookup
  const tennis: TennisMatchInfo = {
    player1: {
      playerCode: playersStr.slice(0, Math.floor(playersStr.length / 2)),
      displayName: formatTennisPlayerCode(playersStr.slice(0, Math.floor(playersStr.length / 2))),
    },
    player2: {
      playerCode: playersStr.slice(Math.floor(playersStr.length / 2)),
      displayName: formatTennisPlayerCode(playersStr.slice(Math.floor(playersStr.length / 2))),
    },
  };

  const shortLabel = outcome
    ? `${label}: ${formatTennisPlayerCode(outcome)} Win`
    : `${label} Match`;

  const fullLabel = `${label}: ${tennis.player1.displayName} vs ${tennis.player2.displayName}${outcome ? ` · ${formatTennisPlayerCode(outcome)} Win` : ''}`;

  return {
    raw,
    category: 'tennis-match',
    series,
    date,
    game: null,
    player: null,
    spread: null,
    total: null,
    tennis,
    outcome,
    shortLabel,
    fullLabel,
  };
}

/**
 * Format tennis player code to readable name
 * DJO -> Djokovic, SINNER -> Sinner, etc.
 */
function formatTennisPlayerCode(code: string): string {
  if (!code) return '';
  // Capitalize first letter, lowercase rest
  return code.charAt(0).toUpperCase() + code.slice(1).toLowerCase();
}

// =============================================================================
// Helpers
// =============================================================================

/**
 * Default fallback for unrecognized tickers
 */
function parseFallback(raw: string): ParsedTicker {
  const firstDash = raw.indexOf('-');
  const series = firstDash > 0 ? raw.slice(0, firstDash) : raw;

  let category: TickerCategory = 'unknown';
  if (series.startsWith('KXATTY') || series.startsWith('KXPRES')) {
    category = 'politics';
  } else if (series.startsWith('KXBTC') || series.startsWith('KXETH')) {
    category = 'crypto';
  }

  return {
    raw,
    category,
    series,
    date: null,
    game: null,
    player: null,
    spread: null,
    total: null,
    tennis: null,
    outcome: null,
    shortLabel: raw,
    fullLabel: raw,
  };
}

/**
 * Parse player info from market part: SEAABARNER88 -> { teamCode: SEA, playerId: ABARNER, jersey: 88 }
 */
function parsePlayerFromMarket(marketPart: string): PlayerInfo | null {
  if (marketPart.length < 4) return null;

  const teamCode = marketPart.slice(0, 3);
  const rest = marketPart.slice(3);

  const match = rest.match(/^([A-Z]+)(\d+)?$/);
  if (!match) {
    return {
      teamCode,
      playerId: rest,
      displayName: formatPlayerName(rest),
    };
  }

  const playerId = match[1] ?? rest;
  const jerseyNumber = match[2] ? Number(match[2]) : undefined;

  return {
    teamCode,
    playerId,
    jerseyNumber,
    displayName: formatPlayerName(playerId, jerseyNumber),
  };
}

/**
 * Format player ID into display name: ABARNER -> A. Barner
 */
function formatPlayerName(playerId: string, jersey?: number): string {
  if (!playerId) return '';

  const initial = playerId.charAt(0);
  const lastName = playerId.slice(1);
  const formatted = lastName.charAt(0).toUpperCase() + lastName.slice(1).toLowerCase();

  const name = `${initial}. ${formatted}`;
  return jersey !== undefined ? `${name} #${jersey}` : name;
}

/**
 * NFL team code to name mapping
 */
function nflTeamFromCode(code: string): string | null {
  const NFL_TEAMS: Record<string, string> = {
    ARI: 'Arizona Cardinals',
    ATL: 'Atlanta Falcons',
    BAL: 'Baltimore Ravens',
    BUF: 'Buffalo Bills',
    CAR: 'Carolina Panthers',
    CHI: 'Chicago Bears',
    CIN: 'Cincinnati Bengals',
    CLE: 'Cleveland Browns',
    DAL: 'Dallas Cowboys',
    DEN: 'Denver Broncos',
    DET: 'Detroit Lions',
    GB: 'Green Bay Packers',
    HOU: 'Houston Texans',
    IND: 'Indianapolis Colts',
    JAX: 'Jacksonville Jaguars',
    KC: 'Kansas City Chiefs',
    LAC: 'Los Angeles Chargers',
    LAR: 'Los Angeles Rams',
    LV: 'Las Vegas Raiders',
    MIA: 'Miami Dolphins',
    MIN: 'Minnesota Vikings',
    NE: 'New England Patriots',
    NO: 'New Orleans Saints',
    NYG: 'New York Giants',
    NYJ: 'New York Jets',
    PHI: 'Philadelphia Eagles',
    PIT: 'Pittsburgh Steelers',
    SEA: 'Seattle Seahawks',
    SF: 'San Francisco 49ers',
    TB: 'Tampa Bay Buccaneers',
    TEN: 'Tennessee Titans',
    WAS: 'Washington Commanders',
  };
  return NFL_TEAMS[code] ?? null;
}

export { nflTeamFromCode, nhlTeamFromCode };
