/**
 * Market State Manager
 *
 * Manages current prices and integrates with time-series store.
 * Provides fast lookups for current prices and access to historical data.
 */

import { TimeSeriesStore } from './timeSeriesStore';
import type { PriceUpdate, PriceTick, Candlestick, CandlestickPeriod } from '../types';

export class MarketState {
  // Current prices (fast lookup)
  private currentPrices: Map<string, number> = new Map();

  // Time-series store
  private timeSeries: TimeSeriesStore;

  // Price change tracking (for UI)
  private previousPrices: Map<string, number> = new Map();

  constructor() {
    this.timeSeries = new TimeSeriesStore();
  }

  /**
   * Update price from WebSocket update
   */
  updatePrice(update: PriceUpdate): void {
    const { ticker, price, timestamp } = update;

    // Store previous price for change calculation
    const currentPrice = this.currentPrices.get(ticker);
    if (currentPrice !== undefined) {
      this.previousPrices.set(ticker, currentPrice);
    }

    // Update current price
    this.currentPrices.set(ticker, price);

    // Add to time-series store
    const tick: PriceTick = {
      ticker,
      timestamp: timestamp || Date.now(),
      price,
      yes_bid: update.yes_bid,
      yes_ask: update.yes_ask,
      no_bid: update.no_bid,
      no_ask: update.no_ask,
      volume: update.volume,
    };

    this.timeSeries.addTick(ticker, tick);
  }

  /**
   * Get current price
   */
  getPrice(ticker: string): number | null {
    return this.currentPrices.get(ticker) || null;
  }

  /**
   * Get price change (current - previous)
   */
  getPriceChange(ticker: string): number | null {
    const current = this.currentPrices.get(ticker);
    const previous = this.previousPrices.get(ticker);

    if (current === undefined || previous === undefined) {
      return null;
    }

    return current - previous;
  }

  /**
   * Get price change percentage
   */
  getPriceChangePercent(ticker: string): number | null {
    const change = this.getPriceChange(ticker);
    const previous = this.previousPrices.get(ticker);

    if (change === null || previous === undefined || previous === 0) {
      return null;
    }

    return (change / previous) * 100;
  }

  /**
   * Get candlesticks for a market
   */
  getCandles(ticker: string, period: CandlestickPeriod, count?: number): Candlestick[] {
    return this.timeSeries.getCandles(ticker, period, count);
  }

  /**
   * Get latest ticks
   */
  getLatestTicks(ticker: string, count: number): PriceTick[] {
    return this.timeSeries.getLatestTicks(ticker, count);
  }

  /**
   * Get all ticks for a market
   */
  getAllTicks(ticker: string): PriceTick[] {
    return this.timeSeries.getAllTicks(ticker);
  }

  /**
   * Get last update timestamp
   */
  getLastUpdate(ticker: string): number | null {
    return this.timeSeries.getLastUpdate(ticker);
  }

  /**
   * Get all markets with data
   */
  getMarkets(): string[] {
    return Array.from(this.currentPrices.keys());
  }

  /**
   * Clear market data
   */
  clearMarket(ticker: string): void {
    this.currentPrices.delete(ticker);
    this.previousPrices.delete(ticker);
    this.timeSeries.clearMarket(ticker);
  }

  /**
   * Clear all data
   */
  clear(): void {
    this.currentPrices.clear();
    this.previousPrices.clear();
    this.timeSeries.clear();
  }

  /**
   * Get time-series store (for advanced operations)
   */
  getTimeSeriesStore(): TimeSeriesStore {
    return this.timeSeries;
  }
}
