/**
 * Market Data Stream
 *
 * Connects to Kalshi WebSocket (directly or via relay) and subscribes to market data.
 * Provides real-time price updates for subscribed markets.
 */

import { createRelayWs, type RelayWsClient } from './relayWs';
import type { PriceUpdate } from '../types';
import { getBaseUrl, buildAuthHeaders } from './kalshiAuth';
import {
  asRecord,
  toNumber,
  normalizeTimestampMs,
  pickFirst,
  type UnknownRecord,
} from './typeCoercers';
import type { Environment } from '@galactus/shared';

export interface OrderbookLevel {
  price: number; // cents, 0..100
  quantity: number;
}

export interface OrderbookUpdate {
  ticker: string;
  yes: OrderbookLevel[];
  no: OrderbookLevel[];
  timestamp: number;
}

export interface MarketStream {
  connect(accessKeyId: string, privateKey: CryptoKey, environment: Environment): Promise<void>;
  subscribe(tickers: string[]): void;
  unsubscribe(tickers: string[]): void;
  subscribeOrderbook(tickers: string[]): void;
  unsubscribeOrderbook(tickers: string[]): void;
  onPriceUpdate(callback: (update: PriceUpdate) => void): void;
  onOrderbookUpdate(callback: (update: OrderbookUpdate) => void): void;
  onError(callback: (error: Error) => void): void;
  disconnect(): void;
  isConnected(): boolean;
}

// Kalshi WS path per docs (Jan 2026)
const KALSHI_WS_PATH = '/trade-api/ws/v2';

/**
 * Create market data stream
 */
export function createMarketStream(useRelay: boolean = true): MarketStream {
  let relayClient: RelayWsClient | null = null;
  let directWs: WebSocket | null = null;
  let streamId: string | null = null;
  const subscribedTickers: Set<string> = new Set();
  const subscribedOrderbookTickers: Set<string> = new Set();
  const priceUpdateCallbacks: Set<(update: PriceUpdate) => void> = new Set();
  const orderbookUpdateCallbacks: Set<(update: OrderbookUpdate) => void> = new Set();
  const errorCallbacks: Set<(error: Error) => void> = new Set();
  let isConnected = false;
  let upstreamConnected = false;
  let upstreamConnectPromise: Promise<void> | null = null;
  let resolveUpstreamConnect: (() => void) | null = null;
  let rejectUpstreamConnect: ((err: Error) => void) | null = null;
  let accessKeyId: string | null = null;
  let privateKey: CryptoKey | null = null;
  const lastTradeAtMsByTicker: Map<string, number> = new Map();
  let nextMessageId = 1;

  // Orderbook state: maintain yes/no quantities by price for each ticker.
  const orderbooks: Map<
    string,
    { yes: Map<number, number>; no: Map<number, number>; lastTs: number }
  > = new Map();

  // Kalshi WebSocket URL (adjust based on actual Kalshi API)
  const getKalshiWsUrl = (environment: Environment): string => {
    const baseUrl = getBaseUrl(environment);
    // Kalshi WebSocket endpoint (per docs)
    return baseUrl.replace('https://', 'wss://').replace('http://', 'ws://') + KALSHI_WS_PATH;
  };

  const buildWsAuthHeaders = async (
    environment: Environment,
    accessKey: string,
    key: CryptoKey
  ): Promise<Record<string, string>> => {
    // Kalshi signs WS similarly to REST: sign GET + path (no body)
    // We keep the canonical request format consistent with our HTTP signing.
    return buildAuthHeaders(accessKey, key, 'GET', KALSHI_WS_PATH, '', '');
  };

  const normalizePriceCents = (raw: unknown): number | null => {
    const n = toNumber(raw);
    if (n === null) return null;
    // If already integer-ish cents
    if (n >= 0 && n <= 100) return Math.round(n);
    // If 0..1 probability
    if (n >= 0 && n <= 1) return Math.round(n * 100);
    return null;
  };

  const toOrderbookLevels = (side: Map<number, number>, sort: 'asc' | 'desc'): OrderbookLevel[] => {
    const entries = Array.from(side.entries())
      .filter(([, qty]) => Number.isFinite(qty) && qty > 0)
      .map(([price, quantity]) => ({ price, quantity }));
    entries.sort((a, b) => (sort === 'asc' ? a.price - b.price : b.price - a.price));
    return entries;
  };

  // --- Debounced emission (coalesce rapid WS updates into ~100ms batches) ---
  let priceUpdateBatch: Map<string, PriceUpdate> = new Map();
  let priceFlushScheduled = false;
  const PRICE_DEBOUNCE_MS = 100;

  const flushPriceUpdates = () => {
    priceFlushScheduled = false;
    const batch = priceUpdateBatch;
    priceUpdateBatch = new Map();
    for (const update of batch.values()) {
      priceUpdateCallbacks.forEach((cb) => cb(update));
    }
  };

  let orderbookUpdateBatch: Map<string, OrderbookUpdate> = new Map();
  let orderbookFlushScheduled = false;
  const ORDERBOOK_DEBOUNCE_MS = 100;

  const flushOrderbookUpdates = () => {
    orderbookFlushScheduled = false;
    const batch = orderbookUpdateBatch;
    orderbookUpdateBatch = new Map();
    for (const update of batch.values()) {
      orderbookUpdateCallbacks.forEach((cb) => cb(update));
    }
  };

  const emitOrderbookUpdate = (ticker: string) => {
    const ob = orderbooks.get(ticker);
    if (!ob) return;
    const update: OrderbookUpdate = {
      ticker,
      yes: toOrderbookLevels(ob.yes, 'desc'),
      no: toOrderbookLevels(ob.no, 'desc'),
      timestamp: ob.lastTs || Date.now(),
    };
    orderbookUpdateBatch.set(ticker, update);
    if (!orderbookFlushScheduled) {
      orderbookFlushScheduled = true;
      setTimeout(flushOrderbookUpdates, ORDERBOOK_DEBOUNCE_MS);
    }
  };

  const parseOrderbookDeltaAndApply = (rec: UnknownRecord): void => {
    const payload = asRecord(pickFirst(rec.msg, rec.data, rec.payload, rec));
    if (!payload) return;

    const ticker =
      (pickFirst(payload.market_ticker, payload.ticker, payload.marketTicker) as
        | string
        | undefined) ?? undefined;
    if (!ticker) return;

    const ts =
      normalizeTimestampMs(
        pickFirst(
          payload.ts,
          payload.timestamp,
          payload.time,
          payload.created_at,
          payload.updated_at
        )
      ) ?? Date.now();

    const getOrInit = () => {
      const existing = orderbooks.get(ticker);
      if (existing) return existing;
      const fresh = { yes: new Map<number, number>(), no: new Map<number, number>(), lastTs: ts };
      orderbooks.set(ticker, fresh);
      return fresh;
    };

    const emitDerivedYesMidIfUseful = (ob: {
      yes: Map<number, number>;
      no: Map<number, number>;
    }) => {
      // Only emit orderbook-derived prices if trades haven't been seen recently.
      const lastTradeAt = lastTradeAtMsByTicker.get(ticker);
      if (lastTradeAt && ts - lastTradeAt <= 15_000) return;

      // Kalshi orderbook snapshots/deltas provide aggregated price levels for YES and NO.
      // These are effectively BID books for each contract.
      //
      // Therefore:
      // - best YES bid = max(yes)
      // - best YES ask = 1 - best NO bid
      // - best NO bid  = max(no)
      // - best NO ask  = 1 - best YES bid
      //
      // Our previous logic incorrectly used min(yes) as "ask", which can severely underprice.
      let bestYesBidCents: number | null = null;
      let bestNoBidCents: number | null = null;

      for (const [price, qty] of ob.yes.entries()) {
        if (!Number.isFinite(price) || !Number.isFinite(qty) || qty <= 0) continue;
        if (bestYesBidCents === null || price > bestYesBidCents) bestYesBidCents = price;
      }
      for (const [price, qty] of ob.no.entries()) {
        if (!Number.isFinite(price) || !Number.isFinite(qty) || qty <= 0) continue;
        if (bestNoBidCents === null || price > bestNoBidCents) bestNoBidCents = price;
      }

      const bestYesAskCents = bestNoBidCents === null ? null : 100 - bestNoBidCents;
      const bestNoAskCents = bestYesBidCents === null ? null : 100 - bestYesBidCents;

      const mid01 = computeMid01FromBidAsk(bestYesBidCents, bestYesAskCents);
      if (mid01 === null) return;

      const update: PriceUpdate = {
        ticker,
        price: mid01,
        timestamp: ts,
        yes_bid: normalizePrice01(bestYesBidCents) ?? undefined,
        yes_ask: normalizePrice01(bestYesAskCents) ?? undefined,
        no_bid: normalizePrice01(bestNoBidCents) ?? undefined,
        no_ask: normalizePrice01(bestNoAskCents) ?? undefined,
      };
      priceUpdateCallbacks.forEach((cb) => cb(update));
    };

    // Snapshot shape: { yes: [[price, qty], ...], no: [[price, qty], ...] }
    const maybeYes = payload.yes ?? asRecord(payload.orderbook)?.yes ?? undefined;
    const maybeNo = payload.no ?? asRecord(payload.orderbook)?.no ?? undefined;
    if (Array.isArray(maybeYes) || Array.isArray(maybeNo)) {
      const ob = getOrInit();
      ob.yes.clear();
      ob.no.clear();

      const loadSide = (arr: unknown, side: Map<number, number>) => {
        if (!Array.isArray(arr)) return;
        for (const row of arr) {
          if (!Array.isArray(row) || row.length < 2) continue;
          const price = normalizePriceCents(row[0]);
          const qty = toNumber(row[1]);
          if (price === null || qty === null) continue;
          if (qty > 0) side.set(price, qty);
        }
      };

      loadSide(maybeYes, ob.yes);
      loadSide(maybeNo, ob.no);
      ob.lastTs = ts;
      emitOrderbookUpdate(ticker);
      emitDerivedYesMidIfUseful(ob);
      return;
    }

    // Delta shape: { side: "yes"|"no", price: 62, delta: 100 }
    const sideRaw = pickFirst(payload.side, payload.contract);
    const side = typeof sideRaw === 'string' ? sideRaw.toLowerCase() : '';
    if (side !== 'yes' && side !== 'no') return;
    const price = normalizePriceCents(payload.price);
    const delta = toNumber(payload.delta);
    if (price === null || delta === null) return;

    const ob = getOrInit();
    const book = side === 'yes' ? ob.yes : ob.no;
    const prev = book.get(price) ?? 0;
    const next = prev + delta;
    if (next <= 0) book.delete(price);
    else book.set(price, next);
    ob.lastTs = ts;
    emitOrderbookUpdate(ticker);
    emitDerivedYesMidIfUseful(ob);
  };

  const normalizePrice01 = (raw: unknown): number | null => {
    if (raw === null || raw === undefined) return null;

    if (typeof raw === 'string') {
      const s = raw.trim();
      if (!s) return null;
      const n = Number(s);
      if (!Number.isFinite(n)) return null;
      return normalizePrice01(n);
    }

    if (typeof raw !== 'number' || !Number.isFinite(raw)) return null;

    // Already 0..1 probability-like
    if (raw >= 0 && raw <= 1) return raw;

    // cents-like 0..100 (or sub-cent pricing)
    if (raw >= 0 && raw <= 100) return raw / 100;

    // dollars-like 0..1 but expressed in 0..10000 bps? If it happens, reject rather than guess.
    return null;
  };

  const computeMid01FromBidAsk = (bidRaw: unknown, askRaw: unknown): number | null => {
    const bid = normalizePrice01(bidRaw);
    const ask = normalizePrice01(askRaw);
    if (bid === null && ask === null) return null;
    if (bid !== null && ask !== null) return (bid + ask) / 2;
    return bid ?? ask;
  };

  const shouldUseTickerFallback = (ticker: string, nowMs: number): boolean => {
    const lastTradeAt = lastTradeAtMsByTicker.get(ticker);
    if (!lastTradeAt) return true;
    // If trades are flowing, don't override with ticker-derived mids.
    return nowMs - lastTradeAt > 15_000;
  };

  const emitPriceUpdate = (update: PriceUpdate) => {
    if (!update.ticker || !Number.isFinite(update.price)) return;
    // Batch by ticker — latest update wins within the debounce window
    priceUpdateBatch.set(update.ticker, update);
    if (!priceFlushScheduled) {
      priceFlushScheduled = true;
      setTimeout(flushPriceUpdates, PRICE_DEBOUNCE_MS);
    }
  };

  const parseTickerMessageToPriceUpdate = (msg: UnknownRecord): PriceUpdate | null => {
    const ticker =
      (pickFirst(msg.market_ticker, msg.ticker, msg.marketTicker) as string | undefined) ??
      undefined;
    if (!ticker) return null;

    // Preferred: best bid/ask midpoint (more stable than last)
    const mid = computeMid01FromBidAsk(
      pickFirst(msg.yes_bid, msg.best_yes_bid, msg.best_bid, msg.bid),
      pickFirst(msg.yes_ask, msg.best_yes_ask, msg.best_ask, msg.ask)
    );

    // Fallback: explicit yes price / last price / price fields
    const rawPrice = pickFirst(msg.yes_price, msg.last_price, msg.price, msg.lastPrice);
    const price = mid ?? normalizePrice01(rawPrice);
    if (price === null) return null;

    const ts =
      normalizeTimestampMs(
        pickFirst(msg.ts, msg.timestamp, msg.time, msg.created_at, msg.updated_at)
      ) ?? Date.now();

    // Hybrid policy: only emit ticker-based updates if trades haven't been seen recently.
    if (!shouldUseTickerFallback(ticker, ts)) return null;

    return {
      ticker,
      price,
      timestamp: ts,
      yes_bid:
        normalizePrice01(pickFirst(msg.yes_bid, msg.best_yes_bid, msg.best_bid, msg.bid)) ??
        undefined,
      yes_ask:
        normalizePrice01(pickFirst(msg.yes_ask, msg.best_yes_ask, msg.best_ask, msg.ask)) ??
        undefined,
      no_bid: normalizePrice01(pickFirst(msg.no_bid, msg.best_no_bid)) ?? undefined,
      no_ask: normalizePrice01(pickFirst(msg.no_ask, msg.best_no_ask)) ?? undefined,
      volume:
        typeof msg.volume === 'number' && Number.isFinite(msg.volume) ? msg.volume : undefined,
    };
  };

  const parseTradeLikeToPriceUpdate = (trade: UnknownRecord): PriceUpdate | null => {
    const ticker =
      (pickFirst(trade.market_ticker, trade.ticker, trade.marketTicker) as string | undefined) ??
      undefined;
    if (!ticker) return null;

    // Trade price can appear as cents, 0..1, or string dollars; normalize to 0..1
    const price =
      normalizePrice01(
        pickFirst(trade.price, trade.price_dollars, trade.yes_price, trade.yes_price_dollars)
      ) ?? null;
    if (price === null) return null;

    const ts =
      normalizeTimestampMs(
        pickFirst(trade.ts, trade.timestamp, trade.time, trade.created_at, trade.createdAt)
      ) ?? Date.now();

    lastTradeAtMsByTicker.set(ticker, ts);

    // Some feeds include bid/ask alongside trades; include if present.
    return {
      ticker,
      price,
      timestamp: ts,
      yes_bid: normalizePrice01(pickFirst(trade.yes_bid, trade.best_yes_bid)) ?? undefined,
      yes_ask: normalizePrice01(pickFirst(trade.yes_ask, trade.best_yes_ask)) ?? undefined,
      no_bid: normalizePrice01(pickFirst(trade.no_bid, trade.best_no_bid)) ?? undefined,
      no_ask: normalizePrice01(pickFirst(trade.no_ask, trade.best_no_ask)) ?? undefined,
      volume:
        typeof trade.size === 'number'
          ? trade.size
          : typeof trade.count === 'number'
            ? trade.count
            : undefined,
    };
  };

  const parseTradesMessageAndEmit = (msg: UnknownRecord) => {
    const payload = pickFirst(
      // Kalshi commonly nests payload under `msg`
      msg.msg,
      msg.trade,
      msg.trades,
      msg.data,
      msg.payload
    );

    if (Array.isArray(payload)) {
      for (const item of payload) {
        const rec = asRecord(item);
        if (!rec) continue;
        const update = parseTradeLikeToPriceUpdate(rec);
        if (update) emitPriceUpdate(update);
      }
      return;
    }

    const rec = asRecord(payload);
    if (rec) {
      const update = parseTradeLikeToPriceUpdate(rec);
      if (update) emitPriceUpdate(update);
    }
  };

  const parseTickerMessageAndEmit = (msg: UnknownRecord) => {
    const payload = asRecord(
      pickFirst(
        // Kalshi commonly nests payload under `msg`
        msg.msg,
        msg.ticker,
        msg.data,
        msg.payload,
        msg
      )
    );
    if (!payload) return;
    const update = parseTickerMessageToPriceUpdate(payload);
    if (update) emitPriceUpdate(update);
  };

  // Parse and handle WebSocket messages (works for both relay and direct)
  const handleWebSocketMessage = (data: unknown) => {
    try {
      // Parse Kalshi WebSocket message
      const message = typeof data === 'string' ? JSON.parse(data) : data;

      const rec = asRecord(message);
      if (!rec) return;

      // Determine channel/type with fallbacks.
      const channel =
        (typeof rec.channel === 'string' ? rec.channel : undefined) ??
        (typeof rec.type === 'string' ? rec.type : undefined) ??
        (typeof rec.event === 'string' ? rec.event : undefined) ??
        undefined;

      // Surface server errors (otherwise we can be connected but unsubscribed).
      if (channel === 'error') {
        const msgObj = asRecord(rec.msg) ?? asRecord(rec.data) ?? rec;
        const code = msgObj && typeof msgObj.code === 'number' ? msgObj.code : undefined;
        const text =
          (msgObj && typeof msgObj.msg === 'string' && msgObj.msg) ||
          (msgObj && typeof msgObj.message === 'string' && msgObj.message) ||
          JSON.stringify(msgObj);
        errorCallbacks.forEach((cb) =>
          cb(new Error(code ? `Kalshi WS error ${code}: ${text}` : `Kalshi WS error: ${text}`))
        );
        return;
      }

      if (channel === 'ticker') {
        parseTickerMessageAndEmit(rec);
        return;
      }

      if (channel === 'trades' || channel === 'public_trades' || channel === 'trade') {
        parseTradesMessageAndEmit(rec);
        return;
      }

      if (
        channel === 'orderbook' ||
        channel === 'orderbook_snapshot' ||
        channel === 'orderbook_delta' ||
        channel === 'orderbook_update'
      ) {
        parseOrderbookDeltaAndApply(rec);
        return;
      }

      // Fallback heuristics:
      // - trade-like payload present
      if (rec.trade || rec.trades) {
        parseTradesMessageAndEmit(rec);
        return;
      }
      // - ticker-like fields present
      if (
        rec.yes_bid ||
        rec.yes_ask ||
        rec.last_price ||
        rec.yes_price ||
        rec.market_ticker ||
        rec.ticker
      ) {
        parseTickerMessageAndEmit(rec);
        return;
      }
    } catch (error) {
      console.error('Failed to parse market update:', error);
    }
  };

  // Handle relay messages
  const handleRelayMessage = (frame: {
    id: string;
    type: string;
    data?: unknown;
    error?: string;
  }) => {
    if (frame.type === 'error') {
      const error = new Error(frame.error || 'WebSocket error');
      errorCallbacks.forEach((cb) => cb(error));
      if (!upstreamConnected && rejectUpstreamConnect) {
        rejectUpstreamConnect(error);
        rejectUpstreamConnect = null;
        resolveUpstreamConnect = null;
        upstreamConnectPromise = null;
      }
      return;
    }

    if (frame.type === 'message' && frame.data) {
      // Relay "connected" ack when upstream WS opens
      try {
        const msg =
          typeof frame.data === 'string'
            ? JSON.parse(frame.data)
            : (frame.data as Record<string, unknown>);
        if (msg && msg.connected === true) {
          upstreamConnected = true;
          if (resolveUpstreamConnect) {
            resolveUpstreamConnect();
            resolveUpstreamConnect = null;
            rejectUpstreamConnect = null;
            upstreamConnectPromise = null;
          }
          return;
        }
      } catch {
        // ignore; fall through to normal message handler
      }

      handleWebSocketMessage(frame.data);
    }

    if (frame.type === 'close') {
      if (!upstreamConnected && rejectUpstreamConnect) {
        rejectUpstreamConnect(new Error('Upstream WebSocket closed during connect'));
        rejectUpstreamConnect = null;
        resolveUpstreamConnect = null;
        upstreamConnectPromise = null;
      }
    }
  };

  // Handle direct WebSocket messages
  const handleDirectMessage = (event: MessageEvent) => {
    handleWebSocketMessage(event.data);
  };

  return {
    async connect(accessKeyIdParam: string, privateKeyParam: CryptoKey, environment: Environment) {
      if (isConnected) {
        return;
      }

      accessKeyId = accessKeyIdParam;
      privateKey = privateKeyParam;
      upstreamConnected = false;

      if (useRelay) {
        // Connect via relay
        relayClient = createRelayWs();

        // Set up callbacks
        relayClient.onMessage(handleRelayMessage);
        relayClient.onError((error) => {
          errorCallbacks.forEach((cb) => cb(error));
          if (!upstreamConnected && rejectUpstreamConnect) {
            rejectUpstreamConnect(error);
            rejectUpstreamConnect = null;
            resolveUpstreamConnect = null;
            upstreamConnectPromise = null;
          }
        });

        // Build signed auth headers for WebSocket handshake (required by Kalshi docs)
        const headers = await buildWsAuthHeaders(environment, accessKeyId, privateKey);

        // Connect to Kalshi WebSocket via relay
        streamId = `market-stream-${Date.now()}`;
        const kalshiWsUrl = getKalshiWsUrl(environment);

        // Wait for relay to confirm upstream is connected; otherwise subscribe() can get dropped.
        upstreamConnectPromise = new Promise<void>((resolve, reject) => {
          resolveUpstreamConnect = resolve;
          rejectUpstreamConnect = reject;
          setTimeout(() => {
            if (!upstreamConnected) {
              reject(new Error('Timed out waiting for upstream WebSocket to connect'));
              resolveUpstreamConnect = null;
              rejectUpstreamConnect = null;
              upstreamConnectPromise = null;
            }
          }, 10000);
        });

        await relayClient.connect(streamId, kalshiWsUrl, headers);
        await upstreamConnectPromise;
      } else {
        // Direct WebSocket connection
        const kalshiWsUrl = getKalshiWsUrl(environment);

        // Build auth headers for WebSocket connection
        // Note: WebSocket connections in browser can't set custom headers directly
        // Kalshi may require auth via query params or initial message
        const _timestamp = Date.now().toString();

        // Try to sign the WebSocket path if possible
        // For now, we'll connect and send auth in first message if needed
        directWs = new WebSocket(kalshiWsUrl);

        await new Promise<void>((resolve, reject) => {
          const timeout = setTimeout(() => {
            reject(new Error('WebSocket connection timeout'));
          }, 10000);

          directWs!.onopen = () => {
            clearTimeout(timeout);
            resolve();
          };

          directWs!.onerror = (_error) => {
            clearTimeout(timeout);
            const wsError = new Error('WebSocket connection failed');
            errorCallbacks.forEach((cb) => cb(wsError));
            reject(wsError);
          };

          directWs!.onmessage = handleDirectMessage;

          directWs!.onclose = () => {
            isConnected = false;
            directWs = null;
          };
        });
      }

      isConnected = true;

      // Subscribe to markets if any were subscribed before connection
      // Note: We need to call subscribe on the returned object, not 'this'
      // This will be handled by the caller after connection is established
    },

    subscribe(tickers: string[]) {
      tickers.forEach((t) => subscribedTickers.add(t));

      if (!isConnected) {
        // Store for later subscription after connection
        return;
      }

      // Send subscription message(s) to Kalshi.
      // Docs (2026): filtering at subscribe-time is `market_ticker` (singular).
      // For multiple tickers, send one subscribe per ticker.
      for (const t of tickers) {
        const subscriptionMessage = {
          id: nextMessageId++,
          cmd: 'subscribe',
          params: {
            // Channel names per docs (Jan 2026):
            // - `ticker`
            // - `trade` (public trades)
            channels: ['ticker', 'trade'],
            // Docs/examples vary between singular/plural; provide both.
            market_ticker: t,
            market_tickers: [t],
          },
          // legacy fields (harmless if ignored)
          type: 'subscribe',
          channels: ['ticker', 'trade'],
          market_ticker: t,
          market_tickers: [t],
        };

        if (useRelay && relayClient && streamId) {
          relayClient.subscribe(streamId, subscriptionMessage);
        } else if (!useRelay && directWs && directWs.readyState === WebSocket.OPEN) {
          directWs.send(JSON.stringify(subscriptionMessage));
        }
      }
    },

    unsubscribe(tickers: string[]) {
      tickers.forEach((t) => subscribedTickers.delete(t));

      if (!isConnected) {
        return;
      }

      // Send unsubscribe message
      const unsubscribeMessage = {
        cmd: 'unsubscribe',
        type: 'unsubscribe',
        channels: ['ticker', 'trade'],
        market_tickers: tickers,
      };

      if (useRelay && relayClient && streamId) {
        relayClient.send(streamId, unsubscribeMessage);
      } else if (!useRelay && directWs && directWs.readyState === WebSocket.OPEN) {
        directWs.send(JSON.stringify(unsubscribeMessage));
      }
    },

    onPriceUpdate(callback: (update: PriceUpdate) => void) {
      priceUpdateCallbacks.add(callback);
    },

    subscribeOrderbook(tickers: string[]) {
      tickers.forEach((t) => subscribedOrderbookTickers.add(t));
      if (!isConnected) return;

      for (const t of tickers) {
        const subscriptionMessage = {
          id: nextMessageId++,
          cmd: 'subscribe',
          params: {
            // Channel name per docs (Jan 2026): `orderbook_delta`
            // Stream delivers `orderbook_snapshot` first then `orderbook_delta` messages.
            channels: ['orderbook_delta'],
            market_ticker: t,
            market_tickers: [t],
          },
          // legacy fields
          type: 'subscribe',
          channels: ['orderbook_delta'],
          market_ticker: t,
          market_tickers: [t],
        };

        if (useRelay && relayClient && streamId) {
          relayClient.subscribe(streamId, subscriptionMessage);
        } else if (!useRelay && directWs && directWs.readyState === WebSocket.OPEN) {
          directWs.send(JSON.stringify(subscriptionMessage));
        }
      }
    },

    unsubscribeOrderbook(tickers: string[]) {
      tickers.forEach((t) => subscribedOrderbookTickers.delete(t));
      if (!isConnected) return;

      const unsubscribeMessage = {
        cmd: 'unsubscribe',
        type: 'unsubscribe',
        channels: ['orderbook_delta'],
        market_tickers: tickers,
      };

      if (useRelay && relayClient && streamId) {
        relayClient.send(streamId, unsubscribeMessage);
      } else if (!useRelay && directWs && directWs.readyState === WebSocket.OPEN) {
        directWs.send(JSON.stringify(unsubscribeMessage));
      }
    },

    onOrderbookUpdate(callback: (update: OrderbookUpdate) => void) {
      orderbookUpdateCallbacks.add(callback);
    },

    onError(callback: (error: Error) => void) {
      errorCallbacks.add(callback);
    },

    disconnect() {
      if (useRelay && relayClient && streamId) {
        relayClient.close(streamId);
        relayClient.disconnect();
        relayClient = null;
        streamId = null;
      } else if (!useRelay && directWs) {
        directWs.close();
        directWs = null;
      }
      isConnected = false;
      subscribedTickers.clear();
      subscribedOrderbookTickers.clear();
      orderbooks.clear();
      accessKeyId = null;
      privateKey = null;
    },

    isConnected() {
      return isConnected;
    },
  };
}
